<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Consolidated Attendance Module Migration
     * 
     * Tables (17 total):
     * 1. att_shift_types - Shift definitions with timing & geofencing
     * 2. att_leave_types - Leave categories (Annual, Sick, etc.)
     * 3. att_request_types - Request categories (Leave, Late, etc.)
     * 4. att_timesheet_types - Work types with pay multipliers
     * 5. att_off_types - Off categories (Holiday, Event break)
     * 6. att_approval_processes - Approval workflow configuration
     * 7. att_settings - Key-value settings store
     * 8. att_shifts - Shift assignments to dept/role/staff
     * 9. att_leave_applications - Staff leave requests
     * 10. att_leave_followers - Leave application followers
     * 11. att_leave_handovers - Work handover during leave
     * 12. att_staff_leave_settings - Per-staff leave overrides
     * 13. att_additional_hours - Extra work hours assigned
     * 14. att_day_offs - Company holidays/day offs
     * 15. att_annual_leave_settings - Annual leave quotas
     * 16. att_attendances - Main attendance records
     * 17. att_attendance_logs - Detailed action logs
     */
    public function up(): void
    {
        // =====================================================================
        // 1. SHIFT TYPES - Shift definitions with timing, colors & geofencing
        // =====================================================================
        Schema::create('att_shift_types', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100);
            $table->string('color', 20)->default('#3b82f6');
            $table->time('start_time');
            $table->time('end_time');
            $table->time('lunch_start_time')->nullable();
            $table->time('lunch_end_time')->nullable();
            $table->json('days_off')->nullable(); // ['saturday', 'sunday']
            $table->text('description')->nullable();
            
            // Location restriction (geofencing)
            $table->boolean('location_restricted')->default(false);
            $table->string('location_name', 255)->nullable();
            $table->decimal('location_latitude', 10, 8)->nullable();
            $table->decimal('location_longitude', 11, 8)->nullable();
            $table->unsignedInteger('allowed_radius')->default(100); // meters
            
            $table->boolean('status')->default(true);
            $table->timestamps();
        });

        // =====================================================================
        // 2. LEAVE TYPES - Leave categories (Annual, Sick, Maternity, etc.)
        // =====================================================================
        Schema::create('att_leave_types', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100);
            $table->string('code', 10)->nullable(); // AL, SL, CL, ML
            $table->string('color', 20)->default('#3b82f6');
            $table->integer('days_per_month')->default(0);
            $table->integer('days_per_year')->default(0);
            $table->boolean('is_paid')->default(true);
            $table->text('description')->nullable();
            $table->boolean('status')->default(true);
            $table->timestamps();
        });

        // =====================================================================
        // 3. REQUEST TYPES - Request categories (Leave, Late, Go home early, etc.)
        // =====================================================================
        Schema::create('att_request_types', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100);
            $table->string('color', 20)->default('#3b82f6');
            $table->boolean('requires_leave_type')->default(false);
            $table->boolean('status')->default(true);
            $table->timestamps();
        });

        // =====================================================================
        // 4. TIMESHEET TYPES - Work types with pay multipliers
        // =====================================================================
        Schema::create('att_timesheet_types', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100);
            $table->string('color', 20)->default('#3b82f6');
            $table->decimal('multiplier', 3, 2)->default(1.00); // 1.5 for overtime, 2.0 for holiday
            $table->boolean('status')->default(true);
            $table->timestamps();
        });

        // =====================================================================
        // 5. OFF TYPES - Off categories (Holiday, Event break, Unexpected break)
        // =====================================================================
        Schema::create('att_off_types', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100);
            $table->string('color', 20)->default('#3b82f6');
            $table->boolean('status')->default(true);
            $table->timestamps();
        });

        // =====================================================================
        // 6. APPROVAL PROCESSES - Approval workflow configuration
        // =====================================================================
        Schema::create('att_approval_processes', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100);
            $table->string('related_to', 50); // leave, additional_hours, etc.
            $table->json('approvers')->nullable(); // Staff IDs who can approve
            $table->boolean('status')->default(true);
            $table->timestamps();
        });

        // =====================================================================
        // 7. SETTINGS - Key-value settings store
        // =====================================================================
        Schema::create('att_settings', function (Blueprint $table) {
            $table->id();
            $table->string('key', 100)->unique();
            $table->text('value')->nullable();
            $table->string('group', 50)->default('general');
            $table->timestamps();
        });

        // =====================================================================
        // 8. SHIFTS - Shift assignments to departments/roles/staff
        // =====================================================================
        Schema::create('att_shifts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('shift_type_id')->constrained('att_shift_types')->onDelete('cascade');
            
            // Assignment targets (all nullable - at least one should be set)
            $table->string('department', 100)->nullable();
            $table->string('role_name', 100)->nullable();
            $table->foreignId('staff_id')->nullable()->constrained('staffs')->onDelete('cascade');
            
            // Date range
            $table->date('from_date');
            $table->date('to_date')->nullable();
            
            // Repeat settings
            $table->enum('repeat_type', ['weekly', 'specific'])->default('weekly');
            $table->json('repeat_days')->nullable(); // ["Mon","Tue","Wed","Thu","Fri"]
            $table->json('specific_dates')->nullable(); // ["2026-01-01","2026-01-02"]
            
            $table->boolean('status')->default(true);
            $table->timestamps();
            
            // Indexes
            $table->index(['department', 'from_date', 'to_date']);
            $table->index(['role_name', 'from_date', 'to_date']);
            $table->index(['staff_id', 'from_date', 'to_date']);
        });

        // =====================================================================
        // 9. LEAVE APPLICATIONS - Staff leave requests with approval workflow
        // =====================================================================
        Schema::create('att_leave_applications', function (Blueprint $table) {
            $table->id();
            $table->foreignId('staff_id')->constrained('staffs')->onDelete('cascade');
            $table->string('subject', 255);
            $table->foreignId('request_type_id')->constrained('att_request_types')->onDelete('cascade');
            $table->foreignId('leave_type_id')->nullable()->constrained('att_leave_types')->onDelete('set null');
            $table->decimal('number_of_days', 5, 1)->default(0);
            $table->datetime('from_date');
            $table->datetime('to_date');
            $table->text('reason')->nullable();
            $table->string('attachment')->nullable();
            $table->enum('status', ['new', 'approved', 'rejected'])->default('new');
            $table->foreignId('approved_by')->nullable()->constrained('staffs')->onDelete('set null');
            $table->datetime('approved_at')->nullable();
            $table->text('reject_reason')->nullable();
            $table->timestamps();

            $table->index(['staff_id', 'status']);
            $table->index(['from_date', 'to_date']);
        });

        // =====================================================================
        // 10. LEAVE FOLLOWERS - People following a leave application
        // =====================================================================
        Schema::create('att_leave_followers', function (Blueprint $table) {
            $table->id();
            $table->foreignId('leave_application_id')->constrained('att_leave_applications')->onDelete('cascade');
            $table->foreignId('staff_id')->constrained('staffs')->onDelete('cascade');
            $table->timestamps();
        });

        // =====================================================================
        // 11. LEAVE HANDOVERS - Work handover assignments during leave
        // =====================================================================
        Schema::create('att_leave_handovers', function (Blueprint $table) {
            $table->id();
            $table->foreignId('leave_application_id')->constrained('att_leave_applications')->onDelete('cascade');
            $table->foreignId('staff_id')->constrained('staffs')->onDelete('cascade');
            $table->timestamps();
        });

        // =====================================================================
        // 12. STAFF LEAVE SETTINGS - Per-staff leave quota overrides
        // =====================================================================
        Schema::create('att_staff_leave_settings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('staff_id')->constrained('staffs')->onDelete('cascade');
            $table->foreignId('leave_type_id')->constrained('att_leave_types')->onDelete('cascade');
            $table->integer('days_per_month')->nullable();
            $table->integer('days_per_year')->nullable();
            $table->timestamps();

            $table->unique(['staff_id', 'leave_type_id']);
        });

        // =====================================================================
        // 13. ADDITIONAL HOURS - Extra work hours assigned by admin
        // =====================================================================
        Schema::create('att_additional_hours', function (Blueprint $table) {
            $table->id();
            $table->foreignId('staff_id')->constrained('staffs')->onDelete('cascade');
            $table->foreignId('assigned_by')->nullable()->constrained('staffs')->onDelete('set null');
            $table->date('work_date');
            $table->time('time_in')->nullable();
            $table->time('time_out')->nullable();
            $table->foreignId('work_type_id')->nullable()->constrained('att_timesheet_types')->onDelete('set null');
            $table->decimal('hours', 5, 2)->default(1);
            $table->text('reason')->nullable();
            $table->enum('status', ['assigned', 'acknowledged', 'completed', 'cancelled'])->default('assigned');
            $table->datetime('acknowledged_at')->nullable();
            $table->timestamps();

            $table->index(['staff_id', 'work_date']);
            $table->index(['status']);
        });

        // =====================================================================
        // 14. DAY OFFS - Company holidays/day offs
        // =====================================================================
        Schema::create('att_day_offs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('off_type_id')->constrained('att_off_types')->onDelete('cascade');
            $table->string('reason', 255);
            $table->date('off_date');
            $table->unsignedBigInteger('department_id')->nullable();
            $table->string('role_name', 100)->nullable();
            $table->boolean('repeat_yearly')->default(false);
            $table->foreignId('added_by')->nullable()->constrained('staffs')->onDelete('set null');
            $table->boolean('status')->default(true);
            $table->timestamps();

            $table->index(['off_date']);
        });

        // =====================================================================
        // 15. ANNUAL LEAVE SETTINGS - Annual leave quotas per staff
        // =====================================================================
        Schema::create('att_annual_leave_settings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('staff_id')->constrained('staffs')->onDelete('cascade');
            $table->foreignId('leave_type_id')->constrained('att_leave_types')->onDelete('cascade');
            $table->decimal('annual_days', 5, 1)->default(0);
            $table->timestamps();

            $table->unique(['staff_id', 'leave_type_id']);
        });

        // =====================================================================
        // 16. ATTENDANCES - Main attendance records (check-in/out, hours, status)
        // =====================================================================
        Schema::create('att_attendances', function (Blueprint $table) {
            $table->id();
            
            // Support BOTH staff_id and admin_id
            // - Staff with admin login: staff_id filled, admin_id from staff->admin_id
            // - Admin without staff (super-admin): staff_id NULL, admin_id filled
            $table->unsignedBigInteger('staff_id')->nullable();
            $table->unsignedBigInteger('admin_id')->nullable();
            
            $table->date('attendance_date');
            $table->unsignedBigInteger('shift_id')->nullable();
            
            $table->time('check_in')->nullable();
            $table->time('check_out')->nullable();
            $table->string('check_in_location', 255)->nullable();
            $table->string('check_out_location', 255)->nullable();
            $table->string('check_in_ip', 45)->nullable();
            $table->string('check_out_ip', 45)->nullable();
            
            $table->decimal('worked_hours', 5, 2)->default(0);
            $table->decimal('overtime_hours', 5, 2)->default(0);
            
            $table->enum('status', [
                'present', 'absent', 'half_day', 'late', 
                'early_leave', 'on_leave', 'holiday', 'weekend'
            ])->default('present');
            
            $table->integer('late_minutes')->default(0);
            $table->integer('early_leave_minutes')->default(0);
            $table->text('remarks')->nullable();
            
            $table->boolean('is_manual')->default(false);
            $table->unsignedBigInteger('marked_by')->nullable();
            
            $table->timestamps();
            
            // Foreign keys
            $table->foreign('staff_id')->references('id')->on('staffs')->cascadeOnDelete();
            $table->foreign('admin_id')->references('id')->on('admins')->cascadeOnDelete();
            $table->foreign('shift_id')->references('id')->on('att_shifts')->nullOnDelete();
            $table->foreign('marked_by')->references('id')->on('admins')->nullOnDelete();
            
            // Indexes
            $table->index('attendance_date');
            $table->index('status');
            $table->index(['staff_id', 'attendance_date']);
            $table->index(['admin_id', 'attendance_date']);
        });

        // =====================================================================
        // 17. ATTENDANCE LOGS - Detailed action logs with GPS coordinates
        // =====================================================================
        Schema::create('att_attendance_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('attendance_id')->constrained('att_attendances')->cascadeOnDelete();
            $table->enum('action', ['check_in', 'check_out', 'break_start', 'break_end', 'manual_edit']);
            $table->time('action_time');
            $table->string('ip_address', 45)->nullable();
            $table->string('location', 255)->nullable();
            $table->decimal('latitude', 10, 8)->nullable();
            $table->decimal('longitude', 11, 8)->nullable();
            $table->string('device_info', 255)->nullable();
            $table->unsignedBigInteger('performed_by')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
            
            $table->foreign('performed_by')->references('id')->on('admins')->nullOnDelete();
        });
    }

    /**
     * Reverse the migrations (drop in reverse order to avoid FK constraint errors)
     */
    public function down(): void
    {
        Schema::dropIfExists('att_attendance_logs');
        Schema::dropIfExists('att_attendances');
        Schema::dropIfExists('att_annual_leave_settings');
        Schema::dropIfExists('att_day_offs');
        Schema::dropIfExists('att_additional_hours');
        Schema::dropIfExists('att_staff_leave_settings');
        Schema::dropIfExists('att_leave_handovers');
        Schema::dropIfExists('att_leave_followers');
        Schema::dropIfExists('att_leave_applications');
        Schema::dropIfExists('att_shifts');
        Schema::dropIfExists('att_settings');
        Schema::dropIfExists('att_approval_processes');
        Schema::dropIfExists('att_off_types');
        Schema::dropIfExists('att_timesheet_types');
        Schema::dropIfExists('att_request_types');
        Schema::dropIfExists('att_leave_types');
        Schema::dropIfExists('att_shift_types');
    }
};