<?php

namespace Modules\Attendance\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use App\Traits\DataTable;
use Illuminate\Http\Request;
use Modules\Attendance\Models\AdditionalHour;
use Modules\Attendance\Models\TimesheetType;
use App\Models\Admin\Staff;

class AdditionalHourController extends AdminController
{
    use DataTable;

    protected $model = AdditionalHour::class;
    protected $viewPrefix = 'attendance::additional-hours';
    protected $routePrefix = 'admin.attendance.additional_hours';
    protected $with = ['staff', 'timesheetType', 'assignedBy'];

    protected $searchable = ['reason'];
    protected $sortable = ['id', 'work_date', 'created_at', 'status'];
    protected $filterable = ['status', 'timesheet_type_id', 'staff_id'];

    protected function mapRow($item)
    {
        return [
            'id' => $item->id,
            'staff_name' => $item->staff ? trim($item->staff->first_name . ' ' . $item->staff->last_name) : '-',
            'department' => $item->staff->department ?? '-',
            'work_date' => $item->work_date->format('d-m-Y'),
            'timesheet_type' => $item->timesheetType->name ?? '-',
            'timesheet_color' => $item->timesheetType->color ?? '#3b82f6',
            'hours_value' => $item->hours_value,
            'time_in' => $item->time_in ? \Carbon\Carbon::parse($item->time_in)->format('h:i A') : '-',
            'time_out' => $item->time_out ? \Carbon\Carbon::parse($item->time_out)->format('h:i A') : '-',
            'assigned_by' => $item->assignedBy ? trim($item->assignedBy->first_name . ' ' . $item->assignedBy->last_name) : '-',
            'status' => $item->status,
            'acknowledged_at' => $item->acknowledged_at ? $item->acknowledged_at->format('d-m-Y h:i A') : null,
            'reason' => $item->reason,
            'created_at' => $item->created_at->format('d-m-Y'),
            '_edit_url' => route('admin.attendance.additional_hours.edit', $item->id),
            '_delete_url' => route('admin.attendance.additional_hours.destroy', $item->id),
        ];
    }

    public function index(Request $request)
    {
        $pageTitle = 'Extra Work Hours';
        $currentStatus = $request->get('status', 'all');
        
        // Stats
        $stats = [
            'assigned' => AdditionalHour::where('status', 'assigned')->count(),
            'acknowledged' => AdditionalHour::where('status', 'acknowledged')->count(),
            'completed' => AdditionalHour::where('status', 'completed')->count(),
            'total' => AdditionalHour::count(),
        ];

        // Query with filters
        $query = AdditionalHour::with(['staff', 'workType', 'assigner'])
            ->orderBy('created_at', 'desc');
        
        if ($currentStatus !== 'all') {
            $query->where('status', $currentStatus);
        }

        $additionalHours = $query->paginate(20);

        return view('attendance::additional-hours.index', compact('pageTitle', 'stats', 'currentStatus', 'additionalHours'));
    }

    public function create()
    {
        $pageTitle = 'Assign Extra Work';
        $workTypes = TimesheetType::where('status', true)->orderBy('name')->get();
        
        // Only get staff WITH active shifts
        $staffList = \Modules\Attendance\Services\AttendanceHelper::getStaffWithShifts();

        return view('attendance::additional-hours.create', compact('pageTitle', 'workTypes', 'staffList'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'staff_ids' => 'required|array|min:1',
            'staff_ids.*' => 'exists:staffs,id',
            'work_date' => 'required|date',
            'time_in' => 'nullable',
            'time_out' => 'nullable',
            'work_type_id' => 'nullable',
            'hours' => 'required|numeric|min:0.5|max:24',
            'reason' => 'nullable|string|max:500',
        ]);

        // Get current admin's staff ID
        $assignedBy = auth()->guard('admin')->user()->staff->id ?? null;
        
        $created = 0;
        foreach ($validated['staff_ids'] as $staffId) {
            AdditionalHour::create([
                'staff_id' => $staffId,
                'assigned_by' => $assignedBy,
                'work_date' => $validated['work_date'],
                'time_in' => $validated['time_in'] ?? null,
                'time_out' => $validated['time_out'] ?? null,
                'work_type_id' => $validated['work_type_id'] ?? null,
                'hours' => $validated['hours'],
                'reason' => $validated['reason'] ?? null,
                'status' => 'assigned',
            ]);
            $created++;
        }

        return redirect()->route('admin.attendance.additional_hours.index')
            ->with('success', "Extra work assigned to {$created} staff member(s) successfully");
    }

    public function edit($id)
    {
        $additionalHour = AdditionalHour::with(['staff', 'workType', 'assigner'])->findOrFail($id);
        $pageTitle = 'View Extra Work Assignment';

        return view('attendance::additional-hours.edit', compact('additionalHour', 'pageTitle'));
    }

    public function update(Request $request, $id)
    {
        $additionalHour = AdditionalHour::findOrFail($id);

        $validated = $request->validate([
            'status' => 'nullable|in:assigned,acknowledged,completed,cancelled',
        ]);

        $additionalHour->update($validated);

        return redirect()->route('admin.attendance.additional_hours.index')
            ->with('success', 'Extra work updated successfully');
    }

    public function destroy($id)
    {
        AdditionalHour::findOrFail($id)->delete();
        return response()->json(['success' => true, 'message' => 'Record deleted successfully']);
    }

    /**
     * Mark as completed
     */
    public function complete($id)
    {
        $record = AdditionalHour::findOrFail($id);
        $record->update([
            'status' => 'completed',
        ]);

        return redirect()->back()->with('success', 'Marked as completed');
    }

    /**
     * Cancel assignment
     */
    public function cancel(Request $request, $id)
    {
        $record = AdditionalHour::findOrFail($id);
        $record->update([
            'status' => 'cancelled',
        ]);

        return redirect()->back()->with('success', 'Assignment cancelled');
    }

    /**
     * Staff view - My Extra Hours page
     */
    public function myExtra()
    {
        $pageTitle = 'My Extra Hours';
        return view('attendance::additional-hours.my-extra', compact('pageTitle'));
    }
}
