<?php

namespace Modules\Attendance\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use Modules\Attendance\Models\Attendance;
use Modules\Attendance\Models\AttendanceLog;
use Modules\Attendance\Models\Shift;
use Modules\Attendance\Models\DayOff;
use Modules\Attendance\Models\LeaveApplication;
use Modules\Attendance\Models\AttendanceSetting;
use App\Models\Admin\Staff;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class AttendanceDashboard extends Component
{
    public $month;
    public $departmentId = '';
    public $roleFilter = '';
    public $staffFilter = '';
    
    public $departments = [];
    public $roles = [];
    public $allStaff = [];

    protected $listeners = ['attendanceUpdated' => '$refresh'];

    protected $queryString = ['month', 'departmentId', 'roleFilter', 'staffFilter'];

    public function mount()
    {
        $this->month = request('month', now()->format('Y-m'));
        
        // Load filter options
        if (DB::getSchemaBuilder()->hasTable('departments')) {
            $this->departments = DB::table('departments')
                ->where('is_active', true)
                ->orderBy('name')
                ->get()
                ->toArray();
        }
        
        $this->roles = \Spatie\Permission\Models\Role::where('guard_name', 'admin')
            ->orderBy('name')
            ->get()
            ->toArray();
        
        $this->allStaff = Staff::where('status', true)
            ->orderBy('first_name')
            ->get()
            ->toArray();
    }

    public function getMonthDateProperty()
    {
        return Carbon::parse($this->month . '-01');
    }

    public function getStartDateProperty()
    {
        return $this->monthDate->copy()->startOfMonth();
    }

    public function getEndDateProperty()
    {
        return $this->monthDate->copy()->endOfMonth();
    }

    public function getDaysInMonthProperty()
    {
        return $this->monthDate->daysInMonth;
    }

    public function getStaffListProperty()
    {
        $query = Staff::where('status', true);
        
        // Filter by join date
        $query->where(function($q) {
            $q->whereNull('join_date')
              ->orWhere('join_date', '<=', $this->endDate->format('Y-m-d'));
        });
        
        // Filter by exit date
        $query->where(function($q) {
            $q->whereNull('exit_date')
              ->orWhere('exit_date', '>=', $this->startDate->format('Y-m-d'));
        });
        
        // Exclude exempt roles
        $exemptRoles = json_decode(AttendanceSetting::get('exempt_roles', '[]'), true) ?: [];
        if (!empty($exemptRoles)) {
            $query->where(function($q) use ($exemptRoles) {
                $q->whereDoesntHave('admin')
                  ->orWhereHas('admin', function($aq) use ($exemptRoles) {
                      $aq->whereDoesntHave('roles', function($rq) use ($exemptRoles) {
                          $rq->whereIn('name', $exemptRoles);
                      });
                  });
            });
        }
        
        // Apply filters
        if ($this->departmentId) {
            $query->where('department_id', $this->departmentId);
        }
        
        if ($this->roleFilter) {
            $query->whereHas('admin.roles', function($q) {
                $q->where('name', $this->roleFilter);
            });
        }
        
        if ($this->staffFilter) {
            $query->where('id', $this->staffFilter);
        }
        
        return $query->orderBy('first_name')->get();
    }

    public function getAttendanceDataProperty()
    {
        $staffIds = $this->staffList->pluck('id');
        
        return Attendance::whereBetween('attendance_date', [$this->startDate, $this->endDate])
            ->whereIn('staff_id', $staffIds)
            ->get()
            ->groupBy('staff_id');
    }

    public function getShiftsDataProperty()
    {
        $staffIds = $this->staffList->pluck('id');
        
        return Shift::with('shiftType')
            ->whereIn('staff_id', $staffIds)
            ->where(function($q) {
                $q->where('from_date', '<=', $this->endDate)
                  ->where(function($sq) {
                      $sq->whereNull('to_date')
                         ->orWhere('to_date', '>=', $this->startDate);
                  });
            })
            ->where('status', true)
            ->get()
            ->groupBy('staff_id');
    }

    public function getHolidaysProperty()
    {
        return DayOff::whereBetween('off_date', [$this->startDate, $this->endDate])
            ->pluck('off_date')
            ->map(fn($d) => Carbon::parse($d)->format('Y-m-d'))
            ->toArray();
    }

    public function getLeavesDataProperty()
    {
        $staffIds = $this->staffList->pluck('id');
        
        return LeaveApplication::whereIn('staff_id', $staffIds)
            ->where('status', 'approved')
            ->where(function($q) {
                $q->whereBetween('from_date', [$this->startDate, $this->endDate])
                  ->orWhereBetween('to_date', [$this->startDate, $this->endDate])
                  ->orWhere(function($sq) {
                      $sq->where('from_date', '<=', $this->startDate)
                         ->where('to_date', '>=', $this->endDate);
                  });
            })
            ->get()
            ->groupBy('staff_id');
    }

    public function getStatusForDay($staffId, $date)
    {
        $dateStr = $date->format('Y-m-d');
        $dayName = $date->format('l');
        
        // Future date
        if ($date->isFuture()) {
            return ['class' => '', 'value' => '', 'title' => ''];
        }
        
        // Holiday
        if (in_array($dateStr, $this->holidays)) {
            return ['class' => 'H', 'value' => 'H', 'title' => 'Holiday'];
        }
        
        // Leave
        $leaves = $this->leavesData[$staffId] ?? collect();
        $isOnLeave = $leaves->first(function($leave) use ($date) {
            return $date->between(Carbon::parse($leave->from_date), Carbon::parse($leave->to_date));
        });
        if ($isOnLeave) {
            return ['class' => 'L', 'value' => 'L', 'title' => 'On Leave'];
        }
        
        // Get shift
        $shifts = $this->shiftsData[$staffId] ?? collect();
        $shift = $shifts->first(function($s) use ($date) {
            $from = Carbon::parse($s->from_date);
            $to = $s->to_date ? Carbon::parse($s->to_date) : null;
            return $date->gte($from) && (!$to || $date->lte($to));
        });
        
        // Get attendance
        $attendances = $this->attendanceData[$staffId] ?? collect();
        $attendance = $attendances->first(function($a) use ($dateStr) {
            return $a->attendance_date->format('Y-m-d') === $dateStr;
        });
        
        // Has attendance record
        if ($attendance && $attendance->check_in) {
            // Check if currently working (for today)
            if ($date->isToday()) {
                $lastLog = AttendanceLog::where('attendance_id', $attendance->id)
                    ->whereIn('action', ['check_in', 'check_out'])
                    ->orderBy('action_time', 'desc')
                    ->first();
                
                if ($lastLog && $lastLog->action === 'check_in') {
                    return [
                        'class' => 'P',
                        'value' => 'P',
                        'title' => 'Currently working since ' . Carbon::parse($lastLog->action_time)->format('h:i A')
                    ];
                }
            }
            
            if ($attendance->check_out) {
                $hours = $attendance->worked_hours ?: 0;
                return [
                    'class' => 'W',
                    'value' => 'W:' . number_format($hours, 1),
                    'title' => 'Worked ' . number_format($hours, 1) . ' hours'
                ];
            } else {
                return ['class' => 'P', 'value' => 'P', 'title' => 'Present (checked in at ' . Carbon::parse($attendance->check_in)->format('h:i A') . ')'];
            }
        }
        
        // Day off (from shift)
        if ($shift && $shift->shiftType) {
            $daysOff = $shift->shiftType->days_off;
            // Handle both string (JSON) and array
            if (is_string($daysOff)) {
                $daysOff = json_decode($daysOff, true) ?: [];
            }
            // Convert to lowercase for comparison
            if (is_array($daysOff)) {
                $daysOffLower = array_map('strtolower', $daysOff);
                if (in_array(strtolower($dayName), $daysOffLower)) {
                    return ['class' => 'OFF', 'value' => 'OFF', 'title' => 'Day Off'];
                }
            }
        }
        
        // No shift assigned
        if (!$shift) {
            return ['class' => 'NS', 'value' => 'NS', 'title' => 'No Shift Assigned'];
        }
        
        // Past or today with no attendance = Absent
        if ($date->isPast() || $date->isToday()) {
            return ['class' => 'A', 'value' => 'A', 'title' => 'Absent'];
        }
        
        return ['class' => '', 'value' => '', 'title' => ''];
    }

    public function previousMonth()
    {
        $this->month = Carbon::parse($this->month . '-01')->subMonth()->format('Y-m');
    }

    public function nextMonth()
    {
        $this->month = Carbon::parse($this->month . '-01')->addMonth()->format('Y-m');
    }

    public function render()
    {
        return view('attendance::livewire.attendance-dashboard');
    }
}
