<?php

namespace Modules\Attendance\Livewire;

use Livewire\Component;
use Modules\Attendance\Models\Attendance;
use Modules\Attendance\Models\AttendanceLog;
use Modules\Attendance\Models\Shift;
use Modules\Attendance\Models\ShiftType;
use Modules\Attendance\Models\AttendanceSetting;
use App\Models\Admin\Staff;
use Carbon\Carbon;

class CheckInOut extends Component
{
    public $selectedStaffId = 'self';
    public $canMarkOthers = false;
    public $staffList = [];
    
    // Location
    public $requireLocation = false;
    public $latitude = null;
    public $longitude = null;
    public $locationText = '';
    
    // Messages
    public $message = '';
    public $messageType = 'success';

    protected $listeners = ['locationCaptured', 'refreshData' => '$refresh'];

    public function mount()
    {
        $admin = auth('admin')->user();
        $this->canMarkOthers = $admin && $admin->can('attendance.mark');
        
        if ($this->canMarkOthers) {
            $this->staffList = Staff::where('status', true)->orderBy('first_name')->get()->toArray();
        }
        
        // Location is ALWAYS required
        $this->requireLocation = true;
    }
    
    public function locationCaptured($lat, $lng, $text)
    {
        $this->latitude = $lat;
        $this->longitude = $lng;
        $this->locationText = $text;
    }

    public function getStaffProperty()
    {
        $admin = auth('admin')->user();
        return $admin ? Staff::where('admin_id', $admin->id)->first() : null;
    }

    public function getEffectiveStaffIdProperty()
    {
        if ($this->selectedStaffId === 'self') {
            return $this->staff?->id;
        }
        return $this->selectedStaffId;
    }

    public function getHasShiftProperty()
    {
        if (!$this->effectiveStaffId) return false;
        
        return Shift::where('staff_id', $this->effectiveStaffId)
            ->where('status', true)
            ->where('from_date', '<=', today())
            ->where(function($q) {
                $q->whereNull('to_date')
                  ->orWhere('to_date', '>=', today());
            })
            ->exists();
    }

    public function getTodayAttendanceProperty()
    {
        if (!$this->effectiveStaffId) return null;
        
        return Attendance::where('staff_id', $this->effectiveStaffId)
            ->whereDate('attendance_date', today())
            ->first();
    }

    public function getTodayLogsProperty()
    {
        if (!$this->todayAttendance) return collect();
        
        return AttendanceLog::where('attendance_id', $this->todayAttendance->id)
            ->whereIn('action', ['check_in', 'check_out'])
            ->orderBy('action_time', 'asc')
            ->get();
    }

    public function getIsCurrentlyCheckedInProperty()
    {
        $logs = $this->todayLogs;
        if ($logs->isEmpty()) return false;
        
        // If last action is check_in, they're currently in
        return $logs->last()->action === 'check_in';
    }

    public function getCurrentSessionStartProperty()
    {
        if (!$this->isCurrentlyCheckedIn) return null;
        
        $lastCheckIn = $this->todayLogs->where('action', 'check_in')->last();
        return $lastCheckIn ? Carbon::parse($lastCheckIn->action_time) : null;
    }

    public function getTotalWorkedSecondsProperty()
    {
        $logs = $this->todayLogs;
        $totalSeconds = 0;
        $lastCheckIn = null;
        
        foreach ($logs as $log) {
            if ($log->action === 'check_in') {
                $lastCheckIn = Carbon::parse($log->action_time);
            } elseif ($log->action === 'check_out' && $lastCheckIn) {
                $checkOut = Carbon::parse($log->action_time);
                $totalSeconds += $lastCheckIn->diffInSeconds($checkOut);
                $lastCheckIn = null;
            }
        }
        
        // Add current session if still checked in
        if ($lastCheckIn) {
            $totalSeconds += $lastCheckIn->diffInSeconds(now());
        }
        
        return $totalSeconds;
    }

    public function getTotalBreakSecondsProperty()
    {
        $logs = $this->todayLogs;
        $totalBreak = 0;
        $lastCheckOut = null;
        
        foreach ($logs as $log) {
            if ($log->action === 'check_out') {
                $lastCheckOut = Carbon::parse($log->action_time);
            } elseif ($log->action === 'check_in' && $lastCheckOut) {
                $checkIn = Carbon::parse($log->action_time);
                $totalBreak += $lastCheckOut->diffInSeconds($checkIn);
                $lastCheckOut = null;
            }
        }
        
        return $totalBreak;
    }

    public function getFormattedWorkedTimeProperty()
    {
        $seconds = $this->totalWorkedSeconds;
        $hours = floor($seconds / 3600);
        $mins = floor(($seconds % 3600) / 60);
        
        if ($hours > 0) {
            return sprintf('%dh %02dm', $hours, $mins);
        }
        return sprintf('%dm', $mins);
    }

    public function getFormattedBreakTimeProperty()
    {
        $seconds = $this->totalBreakSeconds;
        $hours = floor($seconds / 3600);
        $mins = floor(($seconds % 3600) / 60);
        return $hours > 0 ? sprintf('%dh %02dm', $hours, $mins) : sprintf('%dm', $mins);
    }

    public function checkIn()
    {
        $this->message = '';
        
        $staffId = $this->effectiveStaffId;
        if (!$staffId) {
            $this->message = 'Staff not found';
            $this->messageType = 'error';
            return;
        }
        
        // Check shift
        if (!$this->hasShift) {
            $this->message = 'No shift assigned. Contact admin.';
            $this->messageType = 'error';
            return;
        }
        
        // Check if already checked in
        if ($this->isCurrentlyCheckedIn) {
            $this->message = 'Already checked in. Please check out first.';
            $this->messageType = 'error';
            return;
        }
        
        // Location is ALWAYS required
        if (!$this->latitude || !$this->longitude) {
            $this->message = 'Please allow location access and refresh page';
            $this->messageType = 'error';
            return;
        }
        
        // Get current shift for geofencing check
        $shift = Shift::with('shiftType')
            ->where('staff_id', $staffId)
            ->where('status', true)
            ->where('from_date', '<=', today())
            ->where(function($q) {
                $q->whereNull('to_date')->orWhere('to_date', '>=', today());
            })
            ->first();
        
        // Build location text
        $locationText = $this->locationText ?: ($this->latitude . ', ' . $this->longitude);
        
        // Check geofencing if shift type has location restriction
        if ($shift && $shift->shiftType && $shift->shiftType->location_restricted) {
            $shiftType = $shift->shiftType;
            
            if (!$shiftType->isWithinAllowedRadius($this->latitude, $this->longitude)) {
                $distance = $shiftType->getDistanceFrom($this->latitude, $this->longitude);
                $locationName = $shiftType->location_name ?: 'office';
                $this->message = 'You are ' . round($distance) . 'm away from ' . $locationName . '. Max allowed: ' . $shiftType->allowed_radius . 'm';
                $this->messageType = 'error';
                return;
            }
            
            // Use location name from shift type
            $locationText = $shiftType->location_name ?: $locationText;
        }
        
        $admin = auth('admin')->user();
        $staff = Staff::find($staffId);
        
        // Get or create attendance record for today
        $attendance = Attendance::firstOrCreate(
            ['staff_id' => $staffId, 'attendance_date' => today()],
            [
                'staff_id' => $staffId,
                'admin_id' => $staff->admin_id,
                'status' => 'present',
            ]
        );
        
        // If this is first check-in of the day, set check_in time
        if (!$attendance->check_in) {
            $attendance->update([
                'check_in' => now()->format('H:i:s'),
                'check_in_ip' => request()->ip(),
                'check_in_location' => $locationText,
            ]);
        }
        
        // Log the check-in
        AttendanceLog::create([
            'attendance_id' => $attendance->id,
            'action' => 'check_in',
            'action_time' => now()->format('H:i:s'),
            'ip_address' => request()->ip(),
            'location' => $locationText ?: null,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
            'device_info' => request()->userAgent(),
            'performed_by' => $admin->id,
        ]);
        
        $this->message = 'Checked in at ' . now()->format('h:i A');
        $this->messageType = 'success';
        
        // Dispatch event for dashboard refresh
        $this->dispatch('attendanceUpdated');
    }

    public function checkOut()
    {
        $this->message = '';
        
        $staffId = $this->effectiveStaffId;
        if (!$staffId) {
            $this->message = 'Staff not found';
            $this->messageType = 'error';
            return;
        }
        
        // Check if checked in
        if (!$this->isCurrentlyCheckedIn) {
            $this->message = 'Not checked in. Please check in first.';
            $this->messageType = 'error';
            return;
        }
        
        $admin = auth('admin')->user();
        $attendance = $this->todayAttendance;
        
        // Log the check-out
        AttendanceLog::create([
            'attendance_id' => $attendance->id,
            'action' => 'check_out',
            'action_time' => now()->format('H:i:s'),
            'ip_address' => request()->ip(),
            'location' => $this->locationText ?: null,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
            'device_info' => request()->userAgent(),
            'performed_by' => $admin->id,
        ]);
        
        // Update attendance record with latest check-out and calculated hours
        $this->updateAttendanceSummary($attendance);
        
        $this->message = 'Checked out at ' . now()->format('h:i A');
        $this->messageType = 'success';
        
        // Dispatch event for dashboard refresh
        $this->dispatch('attendanceUpdated');
    }

    protected function updateAttendanceSummary(Attendance $attendance)
    {
        // Recalculate total worked hours from all logs
        $logs = AttendanceLog::where('attendance_id', $attendance->id)
            ->whereIn('action', ['check_in', 'check_out'])
            ->orderBy('action_time', 'asc')
            ->get();
        
        $totalMinutes = 0;
        $lastCheckIn = null;
        $lastCheckOutTime = null;
        
        foreach ($logs as $log) {
            if ($log->action === 'check_in') {
                $lastCheckIn = Carbon::parse($log->action_time);
            } elseif ($log->action === 'check_out' && $lastCheckIn) {
                $checkOut = Carbon::parse($log->action_time);
                $totalMinutes += $lastCheckIn->diffInMinutes($checkOut);
                $lastCheckOutTime = $log->action_time;
                $lastCheckIn = null;
            }
        }
        
        $attendance->update([
            'check_out' => $lastCheckOutTime,
            'check_out_ip' => request()->ip(),
            'check_out_location' => $this->locationText ?: null,
            'worked_hours' => round($totalMinutes / 60, 2),
            'overtime_hours' => max(0, round(($totalMinutes - 480) / 60, 2)), // Over 8 hours
        ]);
    }

    public function render()
    {
        return view('attendance::livewire.check-in-out');
    }
}
