<?php

namespace Modules\Attendance\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\Admin;
use App\Models\Admin\Staff;
use Carbon\Carbon;

class Attendance extends Model
{
    protected $table = 'att_attendances';

    protected $fillable = [
        'staff_id',
        'admin_id',
        'attendance_date',
        'shift_id',
        'check_in',
        'check_out',
        'check_in_location',
        'check_out_location',
        'check_in_ip',
        'check_out_ip',
        'worked_hours',
        'overtime_hours',
        'status',
        'late_minutes',
        'early_leave_minutes',
        'remarks',
        'is_manual',
        'marked_by',
    ];

    protected $casts = [
        'attendance_date' => 'date',
        'check_in' => 'datetime:H:i:s',
        'check_out' => 'datetime:H:i:s',
        'worked_hours' => 'decimal:2',
        'overtime_hours' => 'decimal:2',
        'is_manual' => 'boolean',
    ];

    // =====================
    // RELATIONSHIPS
    // =====================
    
    public function staff(): BelongsTo
    {
        return $this->belongsTo(Staff::class);
    }

    public function admin(): BelongsTo
    {
        return $this->belongsTo(Admin::class);
    }

    public function shift(): BelongsTo
    {
        return $this->belongsTo(Shift::class);
    }

    public function markedByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'marked_by');
    }

    public function logs(): HasMany
    {
        return $this->hasMany(AttendanceLog::class);
    }

    // =====================
    // ACCESSORS
    // =====================

    /**
     * Get the person's name (from staff or admin)
     */
    public function getPersonNameAttribute(): string
    {
        if ($this->staff) {
            return $this->staff->first_name . ' ' . $this->staff->last_name;
        }
        if ($this->admin) {
            return $this->admin->name;
        }
        return 'Unknown';
    }

    /**
     * Get initials for avatar
     */
    public function getPersonInitialsAttribute(): string
    {
        if ($this->staff) {
            return strtoupper(substr($this->staff->first_name, 0, 1) . substr($this->staff->last_name, 0, 1));
        }
        if ($this->admin) {
            $parts = explode(' ', $this->admin->name);
            return strtoupper(substr($parts[0] ?? '', 0, 1) . substr($parts[1] ?? '', 0, 1));
        }
        return '??';
    }

    /**
     * Get designation/role
     */
    public function getPersonRoleAttribute(): string
    {
        if ($this->staff) {
            return $this->staff->designation ?? 'Staff';
        }
        if ($this->admin) {
            $roles = $this->admin->roles->pluck('name')->toArray();
            return !empty($roles) ? ucfirst(str_replace('-', ' ', $roles[0])) : 'Admin';
        }
        return '-';
    }

    /**
     * Get department name
     */
    public function getPersonDepartmentAttribute(): ?string
    {
        if ($this->staff && $this->staff->departmentRelation) {
            return $this->staff->departmentRelation->name;
        }
        if ($this->staff && $this->staff->department) {
            return $this->staff->department;
        }
        return null;
    }

    public function getIsCheckedInAttribute(): bool
    {
        return !is_null($this->check_in);
    }

    public function getIsCheckedOutAttribute(): bool
    {
        return !is_null($this->check_out);
    }

    public function getIsCompletedAttribute(): bool
    {
        return $this->is_checked_in && $this->is_checked_out;
    }

    public function getCheckInTimeAttribute(): ?string
    {
        return $this->check_in ? Carbon::parse($this->check_in)->format('h:i A') : null;
    }

    public function getCheckOutTimeAttribute(): ?string
    {
        return $this->check_out ? Carbon::parse($this->check_out)->format('h:i A') : null;
    }

    public function getWorkedDurationAttribute(): string
    {
        if (!$this->check_in) return '0h 0m';
        
        $checkOut = $this->check_out ? Carbon::parse($this->check_out) : now();
        $checkIn = Carbon::parse($this->check_in);
        
        $diff = $checkIn->diff($checkOut);
        return $diff->h . 'h ' . $diff->i . 'm';
    }

    public function getStatusBadgeAttribute(): string
    {
        $badges = [
            'present' => 'success',
            'absent' => 'danger',
            'half_day' => 'warning',
            'late' => 'warning',
            'early_leave' => 'warning',
            'on_leave' => 'info',
            'holiday' => 'primary',
            'weekend' => 'secondary',
        ];

        return $badges[$this->status] ?? 'secondary';
    }

    // =====================
    // SCOPES
    // =====================
    
    public function scopeToday($query)
    {
        return $query->whereDate('attendance_date', today());
    }

    public function scopeForStaff($query, $staffId)
    {
        return $query->where('staff_id', $staffId);
    }

    public function scopeForAdmin($query, $adminId)
    {
        return $query->where('admin_id', $adminId);
    }

    /**
     * For current logged in user (checks both staff and admin)
     */
    public function scopeForCurrentUser($query)
    {
        $admin = auth('admin')->user();
        if (!$admin) return $query->whereRaw('1=0');
        
        $staff = Staff::where('admin_id', $admin->id)->first();
        
        if ($staff) {
            return $query->where('staff_id', $staff->id);
        } else {
            return $query->where('admin_id', $admin->id);
        }
    }

    public function scopePresent($query)
    {
        return $query->whereIn('status', ['present', 'late', 'early_leave']);
    }

    public function scopeAbsent($query)
    {
        return $query->where('status', 'absent');
    }

    public function scopeOnLeave($query)
    {
        return $query->where('status', 'on_leave');
    }

    public function scopeDateRange($query, $from, $to)
    {
        return $query->whereBetween('attendance_date', [$from, $to]);
    }

    public function scopeMonth($query, $month = null, $year = null)
    {
        $month = $month ?? now()->month;
        $year = $year ?? now()->year;
        
        return $query->whereMonth('attendance_date', $month)
                     ->whereYear('attendance_date', $year);
    }

    // =====================
    // METHODS
    // =====================
    
    public function calculateWorkedHours(): float
    {
        if (!$this->check_in || !$this->check_out) {
            return 0;
        }

        $checkIn = Carbon::parse($this->check_in);
        $checkOut = Carbon::parse($this->check_out);
        
        return round($checkIn->diffInMinutes($checkOut) / 60, 2);
    }

    public function calculateOvertimeHours(float $standardHours = 8): float
    {
        $worked = $this->calculateWorkedHours();
        return max(0, $worked - $standardHours);
    }

    public function calculateLateMinutes(): int
    {
        if (!$this->check_in || !$this->shift) {
            return 0;
        }

        $shiftStart = Carbon::parse($this->shift->start_time);
        $checkIn = Carbon::parse($this->check_in);

        if ($checkIn->gt($shiftStart)) {
            return $shiftStart->diffInMinutes($checkIn);
        }

        return 0;
    }

    public function calculateEarlyLeaveMinutes(): int
    {
        if (!$this->check_out || !$this->shift) {
            return 0;
        }

        $shiftEnd = Carbon::parse($this->shift->end_time);
        $checkOut = Carbon::parse($this->check_out);

        if ($checkOut->lt($shiftEnd)) {
            return $checkOut->diffInMinutes($shiftEnd);
        }

        return 0;
    }

    /**
     * Get today's attendance for current logged-in user
     */
    public static function getTodayForCurrentUser(): ?self
    {
        $admin = auth('admin')->user();
        if (!$admin) return null;
        
        $staff = Staff::where('admin_id', $admin->id)->first();
        
        if ($staff) {
            return self::where('staff_id', $staff->id)
                ->whereDate('attendance_date', today())
                ->first();
        } else {
            return self::where('admin_id', $admin->id)
                ->whereNull('staff_id')
                ->whereDate('attendance_date', today())
                ->first();
        }
    }

    /**
     * Get today's attendance for a staff
     */
    public static function getTodayForStaff($staffId): ?self
    {
        return self::where('staff_id', $staffId)
            ->whereDate('attendance_date', today())
            ->first();
    }

    /**
     * Check-in for current user
     */
    public static function checkInCurrentUser(array $data = []): self
    {
        $admin = auth('admin')->user();
        if (!$admin) {
            throw new \Exception('No authenticated user');
        }
        
        $staff = Staff::where('admin_id', $admin->id)->first();
        
        $attendanceData = [
            'check_in' => now()->format('H:i:s'),
            'check_in_ip' => request()->ip(),
            'check_in_location' => $data['location'] ?? null,
            'status' => 'present',
        ];
        
        if ($staff) {
            // Staff member - use staff_id
            $attendance = self::firstOrCreate(
                ['staff_id' => $staff->id, 'attendance_date' => today()],
                array_merge($attendanceData, [
                    'staff_id' => $staff->id, 
                    'admin_id' => $admin->id
                ])
            );
        } else {
            // Admin without staff record (like super-admin)
            $attendance = self::firstOrCreate(
                ['admin_id' => $admin->id, 'attendance_date' => today(), 'staff_id' => null],
                array_merge($attendanceData, ['admin_id' => $admin->id])
            );
        }

        // If already exists but no check_in, update it
        if (!$attendance->check_in) {
            $attendance->update([
                'check_in' => now()->format('H:i:s'),
                'check_in_ip' => request()->ip(),
                'check_in_location' => $data['location'] ?? null,
                'status' => 'present',
            ]);
        }

        // Log the check-in
        $attendance->logs()->create([
            'action' => 'check_in',
            'action_time' => now()->format('H:i:s'),
            'ip_address' => request()->ip(),
            'location' => $data['location'] ?? null,
            'latitude' => $data['latitude'] ?? null,
            'longitude' => $data['longitude'] ?? null,
            'device_info' => request()->userAgent(),
            'performed_by' => $admin->id,
        ]);

        return $attendance;
    }

    /**
     * Check-out for current user
     */
    public static function checkOutCurrentUser(array $data = []): ?self
    {
        $attendance = self::getTodayForCurrentUser();
        
        if (!$attendance || !$attendance->check_in) {
            return null;
        }

        $admin = auth('admin')->user();

        $attendance->update([
            'check_out' => now()->format('H:i:s'),
            'check_out_ip' => request()->ip(),
            'check_out_location' => $data['location'] ?? null,
        ]);

        // Recalculate after setting check_out
        $attendance->refresh();
        $attendance->update([
            'worked_hours' => $attendance->calculateWorkedHours(),
            'overtime_hours' => $attendance->calculateOvertimeHours(),
            'late_minutes' => $attendance->calculateLateMinutes(),
            'early_leave_minutes' => $attendance->calculateEarlyLeaveMinutes(),
        ]);

        // Log the check-out
        $attendance->logs()->create([
            'action' => 'check_out',
            'action_time' => now()->format('H:i:s'),
            'ip_address' => request()->ip(),
            'location' => $data['location'] ?? null,
            'latitude' => $data['latitude'] ?? null,
            'longitude' => $data['longitude'] ?? null,
            'device_info' => request()->userAgent(),
            'performed_by' => $admin->id,
        ]);

        return $attendance;
    }

    /**
     * Check-in for staff (admin marking for someone else)
     */
    public static function checkIn($staffId, array $data = []): self
    {
        $staff = Staff::findOrFail($staffId);
        $admin = auth('admin')->user();

        $attendance = self::firstOrCreate(
            [
                'staff_id' => $staffId,
                'attendance_date' => today(),
            ],
            [
                'staff_id' => $staffId,
                'admin_id' => $staff->admin_id,
                'check_in' => now()->format('H:i:s'),
                'check_in_ip' => request()->ip(),
                'check_in_location' => $data['location'] ?? null,
                'status' => 'present',
                'is_manual' => true,
                'marked_by' => $admin?->id,
            ]
        );

        // If already exists but no check_in, update it
        if (!$attendance->check_in) {
            $attendance->update([
                'check_in' => now()->format('H:i:s'),
                'check_in_ip' => request()->ip(),
                'check_in_location' => $data['location'] ?? null,
                'status' => 'present',
                'is_manual' => true,
                'marked_by' => $admin?->id,
            ]);
        }

        // Log the check-in
        $attendance->logs()->create([
            'action' => 'check_in',
            'action_time' => now()->format('H:i:s'),
            'ip_address' => request()->ip(),
            'location' => $data['location'] ?? null,
            'latitude' => $data['latitude'] ?? null,
            'longitude' => $data['longitude'] ?? null,
            'device_info' => request()->userAgent(),
            'performed_by' => $admin?->id,
            'notes' => 'Marked by admin',
        ]);

        return $attendance;
    }

    /**
     * Check-out for staff (admin marking for someone else)
     */
    public static function checkOut($staffId, array $data = []): ?self
    {
        $attendance = self::getTodayForStaff($staffId);
        
        if (!$attendance || !$attendance->check_in) {
            return null;
        }

        $admin = auth('admin')->user();

        $attendance->update([
            'check_out' => now()->format('H:i:s'),
            'check_out_ip' => request()->ip(),
            'check_out_location' => $data['location'] ?? null,
            'is_manual' => true,
            'marked_by' => $admin?->id,
        ]);

        // Recalculate
        $attendance->refresh();
        $attendance->update([
            'worked_hours' => $attendance->calculateWorkedHours(),
            'overtime_hours' => $attendance->calculateOvertimeHours(),
            'late_minutes' => $attendance->calculateLateMinutes(),
            'early_leave_minutes' => $attendance->calculateEarlyLeaveMinutes(),
        ]);

        // Log the check-out
        $attendance->logs()->create([
            'action' => 'check_out',
            'action_time' => now()->format('H:i:s'),
            'ip_address' => request()->ip(),
            'location' => $data['location'] ?? null,
            'latitude' => $data['latitude'] ?? null,
            'longitude' => $data['longitude'] ?? null,
            'device_info' => request()->userAgent(),
            'performed_by' => $admin?->id,
            'notes' => 'Marked by admin',
        ]);

        return $attendance;
    }
}
