<?php

namespace Modules\Attendance\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\Admin\Staff;
use Spatie\Permission\Models\Role;

class Shift extends Model
{
    protected $table = 'att_shifts';

    protected $fillable = [
        'shift_type_id',
        'department',
        'role_name',
        'staff_id',
        'from_date',
        'to_date',
        'repeat_type',
        'repeat_days',
        'specific_dates',
        'status',
    ];

    protected $casts = [
        'from_date' => 'date',
        'to_date' => 'date',
        'repeat_days' => 'array',
        'specific_dates' => 'array',
        'status' => 'boolean',
    ];

    // =========================================
    // RELATIONSHIPS
    // =========================================

    public function shiftType()
    {
        return $this->belongsTo(ShiftType::class, 'shift_type_id');
    }

    public function staff()
    {
        return $this->belongsTo(Staff::class, 'staff_id');
    }

    // =========================================
    // ACCESSORS
    // =========================================

    // Get assignment type description
    public function getAssignmentTypeAttribute(): string
    {
        $parts = [];
        if ($this->department) $parts[] = "Dept: {$this->department}";
        if ($this->role_name) $parts[] = "Role: {$this->role_name}";
        if ($this->staff_id && $this->staff) {
            $parts[] = "Staff: " . trim($this->staff->first_name . ' ' . $this->staff->last_name);
        }
        return implode(' + ', $parts) ?: 'Not Set';
    }

    // Get date range string
    public function getDateRangeAttribute(): string
    {
        $from = $this->from_date->format('d M Y');
        $to = $this->to_date ? $this->to_date->format('d M Y') : 'Ongoing';
        return "{$from} - {$to}";
    }

    // Get repeat days as string
    public function getRepeatDaysStringAttribute(): string
    {
        if ($this->repeat_type === 'specific') {
            $count = count($this->specific_dates ?? []);
            return "{$count} specific dates";
        }
        if (!$this->repeat_days || empty($this->repeat_days)) {
            return '-';
        }
        return implode(', ', $this->repeat_days);
    }

    // =========================================
    // METHODS
    // =========================================

    /**
     * Get all staff affected by this shift assignment
     */
    public function getAffectedStaff()
    {
        $query = Staff::where('status', true);

        // If specific staff assigned
        if ($this->staff_id) {
            return $query->where('id', $this->staff_id)->get();
        }

        // Filter by department
        if ($this->department) {
            $query->where('department', $this->department);
        }

        // Filter by role
        if ($this->role_name) {
            $query->whereHas('admin.roles', function ($q) {
                $q->where('name', $this->role_name);
            });
        }

        return $query->orderBy('first_name')->get();
    }

    /**
     * Check if shift is active on a specific date for a staff
     */
    public function isActiveOn($date, $staffId = null): bool
    {
        if (!$this->status) return false;

        $checkDate = $date instanceof \Carbon\Carbon ? $date : \Carbon\Carbon::parse($date);

        // Check date range
        if ($checkDate->lt($this->from_date)) return false;
        if ($this->to_date && $checkDate->gt($this->to_date)) return false;

        // Check if staff is affected
        if ($staffId) {
            $affected = $this->getAffectedStaff()->pluck('id')->toArray();
            if (!in_array($staffId, $affected)) return false;
        }

        // Check repeat type
        if ($this->repeat_type === 'weekly' && $this->repeat_days) {
            $dayName = $checkDate->format('D'); // Mon, Tue, etc.
            return in_array($dayName, $this->repeat_days);
        }

        if ($this->repeat_type === 'specific' && $this->specific_dates) {
            return in_array($checkDate->format('Y-m-d'), $this->specific_dates);
        }

        return true;
    }
}
