<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        if (Schema::hasTable('expenses')) {
            return;
        }

        Schema::create('expenses', function (Blueprint $table) {
            $table->id();
            
            // Basic info
            $table->string('expense_number', 50)->nullable()->comment('Auto-generated expense number');
            $table->string('title', 255)->comment('Expense title/description');
            $table->text('notes')->nullable();
            $table->date('expense_date');
            
            // Category and Unit
            $table->unsignedBigInteger('category_id');
            $table->unsignedBigInteger('unit_id')->nullable();
            
            // Amount calculation
            $table->decimal('unit_price', 15, 4)->default(0)->comment('Price per unit');
            $table->decimal('quantity', 15, 4)->default(1)->comment('Quantity of units');
            $table->decimal('subtotal', 15, 2)->default(0)->comment('Amount before taxes');
            $table->decimal('tax_rate', 5, 2)->default(0)->comment('Tax percentage');
            $table->decimal('tax_amount', 15, 2)->default(0)->comment('Calculated tax amount');
            $table->decimal('tax2_rate', 5, 2)->default(0)->comment('Secondary tax percentage');
            $table->decimal('tax2_amount', 15, 2)->default(0)->comment('Secondary tax amount');
            $table->decimal('amount', 15, 2)->default(0)->comment('Total amount');
            $table->string('currency', 10)->default('INR');
            
            // Vendor info
            $table->string('vendor', 255)->nullable()->comment('Vendor/supplier name');
            $table->string('reference_no', 100)->nullable()->comment('External reference number');
            $table->string('account_code', 50)->nullable()->comment('Accounting code');
            
            // Payment info
            $table->enum('paid_by', ['employee', 'company'])->default('employee')->comment('Who paid initially');
            $table->unsignedBigInteger('payment_method_id')->nullable();
            
            // Employee and Manager
            $table->unsignedBigInteger('employee_id')->comment('Employee who incurred the expense');
            $table->unsignedBigInteger('manager_id')->nullable()->comment('Assigned manager for approval');
            
            // Workflow status
            $table->enum('status', ['draft', 'submitted', 'approved', 'rejected', 'paid'])->default('draft');
            
            // Submission tracking
            $table->unsignedBigInteger('submitted_by')->nullable();
            $table->timestamp('submitted_at')->nullable();
            
            // Approval tracking
            $table->unsignedBigInteger('approved_by')->nullable();
            $table->timestamp('approved_at')->nullable();
            $table->text('rejection_reason')->nullable();
            
            // Payment tracking
            $table->enum('payment_status', ['unpaid', 'partially_paid', 'paid'])->default('unpaid');
            $table->decimal('total_paid', 15, 2)->default(0);
            $table->timestamp('paid_at')->nullable();
            $table->unsignedBigInteger('paid_by_admin')->nullable()->comment('Admin who processed payment');
            
            // Billable to client
            $table->boolean('is_billable')->default(false);
            $table->unsignedBigInteger('client_id')->nullable();
            
            // Created by
            $table->unsignedBigInteger('created_by');
            $table->timestamps();
            $table->softDeletes();
            
            // Indexes
            $table->index('expense_number');
            $table->index('expense_date');
            $table->index('category_id');
            $table->index('employee_id');
            $table->index('manager_id');
            $table->index('status');
            $table->index('payment_status');
            $table->index('created_by');
            $table->index(['status', 'payment_status']);
            
            // Foreign keys
            $table->foreign('category_id')->references('id')->on('expense_categories')->onDelete('restrict');
            $table->foreign('unit_id')->references('id')->on('expense_units')->onDelete('set null');
            $table->foreign('employee_id')->references('id')->on('admins')->onDelete('cascade');
            $table->foreign('manager_id')->references('id')->on('admins')->onDelete('set null');
            $table->foreign('approved_by')->references('id')->on('admins')->onDelete('set null');
            $table->foreign('submitted_by')->references('id')->on('admins')->onDelete('set null');
            $table->foreign('paid_by_admin')->references('id')->on('admins')->onDelete('set null');
            $table->foreign('created_by')->references('id')->on('admins')->onDelete('cascade');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('expenses');
    }
};
