<?php

namespace Modules\Expense\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Expense\Models\Expense;
use Modules\Expense\Models\ExpensePayment;
use Modules\Expense\Models\PaymentMethod;
use Modules\Expense\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class ExpenseApprovalController extends AdminController
{
    // ==================== APPROVAL DASHBOARD ====================

    /**
     * Show approval dashboard - Only for super-admin
     */
    public function dashboard()
    {
        $user = Auth::guard('admin')->user();
        
        // Only super-admin can access approval dashboard
        if (!$user->hasRole('super-admin')) {
            abort(403, 'Only super-admin can access the approval dashboard.');
        }
        
        // Get pending expenses for approval
        $pendingExpenses = Expense::with(['category', 'employee', 'attachments'])
            ->pendingApproval()
            ->orderBy('created_at', 'asc')
            ->get();

        // Get approved expenses ready for payment
        $approvedExpenses = Expense::with(['category', 'employee', 'manager', 'payments'])
            ->approvedUnpaid()
            ->orderBy('approved_at', 'asc')
            ->get();

        // Summary stats
        $summary = [
            'pending_count' => $pendingExpenses->count(),
            'pending_amount' => $pendingExpenses->sum('amount'),
            'approved_count' => $approvedExpenses->count(),
            'approved_amount' => $approvedExpenses->sum('amount'),
            'approved_unpaid' => $approvedExpenses->sum('remaining_balance'),
            'urgent_count' => $pendingExpenses->filter(fn($e) => $e->is_urgent)->count(),
        ];

        // Recent payments
        $recentPayments = ExpensePayment::with(['expense', 'paymentMethod', 'createdByAdmin'])
            ->where('status', 'completed')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        $paymentMethods = PaymentMethod::forExpenses()->ordered()->get();
        $currency = Currency::getDefault();

        return view('expense::approval_dashboard', compact(
            'pendingExpenses', 
            'approvedExpenses', 
            'summary', 
            'recentPayments',
            'paymentMethods',
            'currency'
        ));
    }

    // ==================== SUBMISSION ====================

    /**
     * Submit expense for approval
     */
    public function submit($id)
    {
        $expense = Expense::findOrFail($id);
        
        $this->authorizeSubmit($expense);

        if (!$expense->canBeSubmitted()) {
            return back()->with('warning', 'This expense cannot be submitted in its current status.');
        }

        // Validate before submission
        $errors = $this->validateForSubmission($expense);
        if (!empty($errors)) {
            return back()->with('error', 'Cannot submit: ' . implode(', ', $errors));
        }

        DB::beginTransaction();
        try {
            $expense->update([
                'status' => 'submitted',
                'submitted_by' => Auth::guard('admin')->id(),
                'submitted_at' => now(),
                'rejection_reason' => null,
            ]);

            $expense->logActivity('submitted', 'Expense submitted for approval to ' . ($expense->manager?->name ?? 'Admin'));

            // TODO: Send notification to manager
            // $this->sendSubmissionNotification($expense);

            DB::commit();

            return back()->with('success', 'Expense submitted for approval successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to submit expense: ' . $e->getMessage());
        }
    }

    /**
     * Validate expense for submission
     */
    protected function validateForSubmission(Expense $expense): array
    {
        $errors = [];

        if (empty($expense->title)) {
            $errors[] = 'Title is required';
        }

        if ($expense->amount <= 0) {
            $errors[] = 'Amount must be greater than zero';
        }

        if (config('expense.require_manager_selection', true) && empty($expense->manager_id)) {
            $errors[] = 'Manager selection is required';
        }

        if (config('expense.require_receipt', false) && $expense->attachments->isEmpty()) {
            $errors[] = 'At least one receipt/attachment is required';
        }

        return $errors;
    }

    // ==================== APPROVAL PROCESSING ====================

    /**
     * Approve expense
     */
    public function approve($id)
    {
        $expense = Expense::findOrFail($id);
        
        $this->checkApprovalPermission();
        $this->checkSelfApproval($expense);

        if (!$expense->canBeApproved()) {
            return back()->with('warning', 'This expense cannot be approved in its current status.');
        }

        DB::beginTransaction();
        try {
            $expense->update([
                'status' => 'approved',
                'approved_by' => Auth::guard('admin')->id(),
                'approved_at' => now(),
                'rejection_reason' => null,
            ]);

            $expense->logActivity('approved', 'Expense approved');

            // TODO: Send notification to employee
            // $this->sendApprovalNotification($expense);

            DB::commit();

            if (request()->ajax()) {
                return response()->json(['success' => true, 'message' => 'Expense approved successfully!']);
            }

            return back()->with('success', 'Expense approved successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            if (request()->ajax()) {
                return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
            }
            return back()->with('error', 'Failed to approve expense: ' . $e->getMessage());
        }
    }

    /**
     * Reject expense
     */
    public function reject(Request $request, $id)
    {
        $expense = Expense::findOrFail($id);
        
        $this->checkApprovalPermission();

        if (!$expense->canBeApproved()) {
            return back()->with('warning', 'This expense cannot be rejected in its current status.');
        }

        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:1000',
        ]);

        DB::beginTransaction();
        try {
            $expense->update([
                'status' => 'rejected',
                'approved_by' => Auth::guard('admin')->id(),
                'approved_at' => now(),
                'rejection_reason' => $validated['rejection_reason'],
            ]);

            $expense->logActivity('rejected', 'Expense rejected: ' . $validated['rejection_reason']);

            // TODO: Send notification to employee
            // $this->sendRejectionNotification($expense);

            DB::commit();

            if (request()->ajax()) {
                return response()->json(['success' => true, 'message' => 'Expense rejected.']);
            }

            return back()->with('success', 'Expense rejected.');
        } catch (\Exception $e) {
            DB::rollBack();
            if (request()->ajax()) {
                return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
            }
            return back()->with('error', 'Failed to reject expense: ' . $e->getMessage());
        }
    }

    /**
     * Bulk approval
     */
    public function bulkApprove(Request $request)
    {
        $this->checkApprovalPermission();

        $ids = $request->input('expense_ids', []);
        $action = $request->input('action', 'approve');
        
        if (empty($ids)) {
            return back()->with('warning', 'No expenses selected.');
        }

        $user = Auth::guard('admin')->user();
        $allowSelfApproval = config('expense.allow_self_approval', false);
        
        $processed = 0;
        $skipped = 0;
        $selfSkipped = 0;

        DB::beginTransaction();
        try {
            foreach ($ids as $id) {
                $expense = Expense::find($id);
                if (!$expense || !$expense->canBeApproved()) {
                    $skipped++;
                    continue;
                }

                // Check self-approval
                if (!$allowSelfApproval && !$user->hasRole('super-admin')) {
                    if ($expense->employee_id == $user->id || $expense->created_by == $user->id) {
                        $selfSkipped++;
                        continue;
                    }
                }

                if ($action === 'approve') {
                    $expense->update([
                        'status' => 'approved',
                        'approved_by' => $user->id,
                        'approved_at' => now(),
                    ]);
                    $expense->logActivity('approved', 'Bulk approved');
                }

                $processed++;
            }

            DB::commit();

            $message = "{$processed} expense(s) {$action}d.";
            if ($skipped > 0) $message .= " {$skipped} skipped (invalid status).";
            if ($selfSkipped > 0) $message .= " {$selfSkipped} skipped (self-approval disabled).";

            return back()->with('success', $message);
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Bulk operation failed: ' . $e->getMessage());
        }
    }

    // ==================== PAYMENT PROCESSING ====================

    /**
     * Show payment form
     */
    public function paymentForm($id)
    {
        $this->checkPaymentPermission();

        $expense = Expense::with(['category', 'employee', 'payments.paymentMethod'])
            ->findOrFail($id);

        if (!$expense->canBePaid()) {
            return redirect()->route('admin.expense.approval.dashboard')
                ->with('warning', 'This expense cannot be paid in its current status.');
        }

        $paymentMethods = PaymentMethod::forExpenses()->ordered()->get();

        return view('expense::payment_form', compact('expense', 'paymentMethods'));
    }

    /**
     * Process payment
     */
    public function processPayment(Request $request, $id)
    {
        $this->checkPaymentPermission();

        $expense = Expense::findOrFail($id);

        if (!$expense->canBePaid()) {
            return back()->with('warning', 'This expense cannot be paid.');
        }

        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01|max:' . $expense->remaining_balance,
            'payment_date' => 'required|date',
            'payment_method_id' => 'required|exists:payment_methods,id',
            'transaction_id' => 'nullable|string|max:255',
            'note' => 'nullable|string|max:1000',
            'bank_details' => 'nullable|string|max:1000',
        ]);

        DB::beginTransaction();
        try {
            $payment = ExpensePayment::create([
                'expense_id' => $expense->id,
                'payment_reference' => ExpensePayment::generatePaymentReference(),
                'amount' => $validated['amount'],
                'payment_date' => $validated['payment_date'],
                'payment_method_id' => $validated['payment_method_id'],
                'transaction_id' => $validated['transaction_id'] ?? null,
                'note' => $validated['note'] ?? null,
                'bank_details' => $validated['bank_details'] ?? null,
                'status' => 'completed',
                'created_by' => Auth::guard('admin')->id(),
            ]);

            // Update expense payment status (handled in model boot)
            $expense->update(['paid_by_admin' => Auth::guard('admin')->id()]);

            $expense->logActivity('payment_added', 'Payment of ₹' . number_format($validated['amount'], 2) . ' recorded');

            // TODO: Send payment notification
            // $this->sendPaymentNotification($expense, $payment);

            DB::commit();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true, 
                    'message' => 'Payment processed successfully!',
                    'payment_reference' => $payment->payment_reference,
                ]);
            }

            return redirect()->route('admin.expense.approval.dashboard')
                ->with('success', 'Payment processed successfully! Reference: ' . $payment->payment_reference);
        } catch (\Exception $e) {
            DB::rollBack();
            if (request()->ajax()) {
                return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
            }
            return back()->with('error', 'Payment failed: ' . $e->getMessage());
        }
    }

    /**
     * Quick pay (AJAX from dashboard)
     */
    public function quickPay(Request $request): JsonResponse
    {
        $this->checkPaymentPermission();

        $validated = $request->validate([
            'expense_id' => 'required|exists:expenses,id',
            'amount' => 'required|numeric|min:0.01',
            'payment_method_id' => 'required|exists:payment_methods,id',
            'transaction_id' => 'nullable|string|max:255',
            'note' => 'nullable|string|max:500',
        ]);

        $expense = Expense::findOrFail($validated['expense_id']);

        if (!$expense->canBePaid()) {
            return response()->json(['success' => false, 'message' => 'Expense cannot be paid'], 400);
        }

        if ($validated['amount'] > $expense->remaining_balance) {
            return response()->json(['success' => false, 'message' => 'Amount exceeds remaining balance'], 400);
        }

        DB::beginTransaction();
        try {
            $payment = ExpensePayment::create([
                'expense_id' => $expense->id,
                'payment_reference' => ExpensePayment::generatePaymentReference(),
                'amount' => $validated['amount'],
                'payment_date' => now(),
                'payment_method_id' => $validated['payment_method_id'],
                'transaction_id' => $validated['transaction_id'] ?? null,
                'note' => $validated['note'] ?? null,
                'status' => 'completed',
                'created_by' => Auth::guard('admin')->id(),
            ]);

            $expense->update(['paid_by_admin' => Auth::guard('admin')->id()]);
            $expense->logActivity('payment_added', 'Quick payment of ₹' . number_format($validated['amount'], 2));

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment successful!',
                'payment_reference' => $payment->payment_reference,
                'new_balance' => $expense->fresh()->remaining_balance,
                'payment_status' => $expense->fresh()->payment_status,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Payment history for an expense
     */
    public function paymentHistory($id)
    {
        $expense = Expense::with(['payments.paymentMethod', 'payments.createdByAdmin'])
            ->findOrFail($id);

        return view('expense::payment_history', compact('expense'));
    }

    // ==================== MY EXPENSES ====================

    /**
     * Show current user's expense history
     */
    public function myExpenses()
    {
        $user = Auth::guard('admin')->user();
        
        $stats = [
            'total' => Expense::forEmployee($user->id)->count(),
            'draft' => Expense::forEmployee($user->id)->status('draft')->count(),
            'submitted' => Expense::forEmployee($user->id)->status('submitted')->count(),
            'approved' => Expense::forEmployee($user->id)->status('approved')->count(),
            'rejected' => Expense::forEmployee($user->id)->status('rejected')->count(),
            'paid' => Expense::forEmployee($user->id)->status('paid')->count(),
            'total_amount' => Expense::forEmployee($user->id)->sum('amount'),
            'total_paid' => Expense::forEmployee($user->id)->sum('total_paid'),
        ];
        
        $currency = Currency::getDefault();

        return view('expense::my_expenses', compact('stats', 'currency'));
    }

    /**
     * DataTable for my expenses
     */
    public function myExpensesData(Request $request): JsonResponse
    {
        $user = Auth::guard('admin')->user();
        
        $query = Expense::with(['category', 'manager'])
            ->forEmployee($user->id);

        if ($search = $request->input('search')) {
            $query->search($search);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }

        $sortCol = $request->input('sort', 'id');
        $sortDir = $request->input('dir', 'desc');
        $query->orderBy($sortCol, $sortDir);

        $perPage = $request->input('per_page', 15);
        $data = $query->paginate($perPage);

        $items = collect($data->items())->map(function ($item) {
            return [
                'id' => $item->id,
                'expense_number' => $item->expense_number,
                'title' => $item->title,
                'category_name' => $item->category?->name,
                'manager_name' => $item->manager?->name,
                'amount' => $item->amount,
                'expense_date' => $item->expense_date?->format('Y-m-d'),
                'status' => $item->status,
                'status_label' => $item->status_label,
                'status_badge' => $item->status_badge,
                'payment_status' => $item->payment_status,
                'payment_status_label' => $item->payment_status_label,
                'total_paid' => $item->total_paid,
                'rejection_reason' => $item->rejection_reason,
                '_edit_url' => route('admin.expense.edit', $item->id),
                '_show_url' => route('admin.expense.show', $item->id),
            ];
        });

        return response()->json([
            'data' => $items,
            'total' => $data->total(),
            'current_page' => $data->currentPage(),
            'last_page' => $data->lastPage(),
        ]);
    }

    // ==================== PERMISSION HELPERS ====================

    protected function checkApprovalPermission(): void
    {
        $user = Auth::guard('admin')->user();
        if (!$user->hasRole('super-admin')) {
            abort(403, 'Only super-admin can approve/reject expenses.');
        }
    }

    protected function checkPaymentPermission(): void
    {
        $user = Auth::guard('admin')->user();
        if (!$user->hasRole('super-admin')) {
            abort(403, 'Only super-admin can process expense payments.');
        }
    }

    protected function checkSelfApproval(Expense $expense): void
    {
        $user = Auth::guard('admin')->user();
        $allowSelfApproval = config('expense.allow_self_approval', false);

        if (!$allowSelfApproval && !$user->hasRole('super-admin')) {
            if ($expense->employee_id == $user->id || $expense->created_by == $user->id) {
                abort(403, 'You cannot approve your own expenses.');
            }
        }
    }

    protected function authorizeSubmit(Expense $expense): void
    {
        $user = Auth::guard('admin')->user();
        if ($expense->created_by !== $user->id && $expense->employee_id !== $user->id) {
            abort(403, 'You cannot submit this expense.');
        }
    }

    protected function canPayExpenses(): bool
    {
        $user = Auth::guard('admin')->user();
        return $user->hasRole('super-admin') || $user->can('expense.pay');
    }
}
