<?php

namespace Modules\Expense\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Expense\Models\ExpenseCategory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;

class ExpenseCategoryController extends AdminController
{
    /**
     * Check if user is super-admin for all category operations
     */
    protected function checkSuperAdmin(): void
    {
        if (!Auth::guard('admin')->user()->hasRole('super-admin')) {
            abort(403, 'Only super-admin can manage expense categories.');
        }
    }

    /**
     * Display categories listing
     */
    public function index()
    {
        $this->checkSuperAdmin();
        
        $stats = [
            'total' => ExpenseCategory::count(),
            'active' => ExpenseCategory::active()->count(),
            'inactive' => ExpenseCategory::where('is_active', false)->count(),
            'travel' => ExpenseCategory::travel()->count(),
        ];
        
        return view('expense::categories.index', compact('stats'));
    }

    /**
     * DataTable endpoint
     */
    public function dataTable(Request $request): JsonResponse
    {
        $this->checkSuperAdmin();
        
        $query = ExpenseCategory::withCount('expenses');

        if ($search = $request->input('search')) {
            $query->search($search);
        }

        if ($request->filled('is_active')) {
            $query->where('is_active', $request->input('is_active'));
        }

        if ($request->filled('is_travel')) {
            $query->where('is_travel_category', $request->input('is_travel'));
        }

        $sortCol = $request->input('sort', 'id');
        $sortDir = $request->input('dir', 'asc');
        $sortable = ['id', 'name', 'unit_price', 'is_active', 'created_at'];
        
        if (in_array($sortCol, $sortable)) {
            $query->orderBy($sortCol, $sortDir);
        } else {
            $query->orderBy('name', 'asc');
        }

        $perPage = $request->input('per_page', 15);
        $data = $query->paginate($perPage);

        $items = collect($data->items())->map(function ($item) {
            return [
                'id' => $item->id,
                'name' => $item->name,
                'description' => $item->description,
                'unit_price' => $item->unit_price,
                'is_travel_category' => $item->is_travel_category,
                'is_active' => $item->is_active,
                'expenses_count' => $item->expenses_count,
                '_edit_url' => route('admin.expense.categories.edit', $item->id),
            ];
        });

        return response()->json([
            'data' => $items,
            'total' => $data->total(),
            'current_page' => $data->currentPage(),
            'last_page' => $data->lastPage(),
        ]);
    }

    /**
     * Show create form
     */
    public function create()
    {
        $this->checkSuperAdmin();
        return view('expense::categories.create');
    }

    /**
     * Store category
     */
    public function store(Request $request)
    {
        $this->checkSuperAdmin();
        
        $validated = $request->validate([
            'name' => 'required|string|max:191|unique:expense_categories,name',
            'description' => 'nullable|string',
            'unit_price' => 'nullable|numeric|min:0',
            'is_travel_category' => 'boolean',
            'is_project_category' => 'boolean',
            'is_active' => 'boolean',
        ]);

        $validated['is_travel_category'] = $request->boolean('is_travel_category');
        $validated['is_project_category'] = $request->boolean('is_project_category');
        $validated['is_active'] = $request->boolean('is_active', true);

        ExpenseCategory::create($validated);

        return redirect()->route('admin.expense.categories.index')
            ->with('success', 'Category created successfully!');
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $this->checkSuperAdmin();
        $category = ExpenseCategory::findOrFail($id);
        return view('expense::categories.edit', compact('category'));
    }

    /**
     * Update category
     */
    public function update(Request $request, $id)
    {
        $this->checkSuperAdmin();
        $category = ExpenseCategory::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:191|unique:expense_categories,name,' . $id,
            'description' => 'nullable|string',
            'unit_price' => 'nullable|numeric|min:0',
            'is_travel_category' => 'boolean',
            'is_project_category' => 'boolean',
            'is_active' => 'boolean',
        ]);

        $validated['is_travel_category'] = $request->boolean('is_travel_category');
        $validated['is_project_category'] = $request->boolean('is_project_category');
        $validated['is_active'] = $request->boolean('is_active', true);

        $category->update($validated);

        return redirect()->route('admin.expense.categories.index')
            ->with('success', 'Category updated successfully!');
    }

    /**
     * Delete category
     */
    public function destroy($id)
    {
        $this->checkSuperAdmin();
        $category = ExpenseCategory::withCount('expenses')->findOrFail($id);

        if ($category->expenses_count > 0) {
            if (request()->ajax()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Cannot delete category with existing expenses.'
                ], 400);
            }
            return back()->with('error', 'Cannot delete category with existing expenses.');
        }

        $category->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'Category deleted!']);
        }
        
        return redirect()->route('admin.expense.categories.index')
            ->with('success', 'Category deleted!');
    }

    /**
     * Toggle status
     */
    public function toggleStatus($id): JsonResponse
    {
        $this->checkSuperAdmin();
        $category = ExpenseCategory::findOrFail($id);
        $category->is_active = !$category->is_active;
        $category->save();
        
        return response()->json([
            'success' => true, 
            'is_active' => $category->is_active,
            'message' => 'Status updated!'
        ]);
    }

    /**
     * Get categories for dropdown (AJAX)
     */
    public function search(Request $request): JsonResponse
    {
        $categories = ExpenseCategory::active()
            ->search($request->input('q', ''))
            ->limit(20)
            ->get(['id', 'name', 'unit_price', 'is_travel_category']);

        return response()->json([
            'results' => $categories->map(fn($c) => [
                'id' => $c->id,
                'text' => $c->name,
                'name' => $c->name,
                'unit_price' => $c->unit_price,
                'is_travel' => $c->is_travel_category,
            ])
        ]);
    }
}
