<?php

namespace Modules\Expense\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Expense\Models\Expense;
use Modules\Expense\Models\ExpenseCategory;
use Modules\Expense\Models\ExpenseUnit;
use Modules\Expense\Models\ExpenseAttachment;
use Modules\Expense\Models\ExpensePayment;
use Modules\Expense\Models\ExpenseManagerAssignment;
use Modules\Expense\Models\PaymentMethod;
use Modules\Expense\Models\Currency;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class ExpenseController extends AdminController
{
    // ==================== MAIN VIEWS ====================

    /**
     * Display expense listing with stats
     * Only super-admin can see all expenses, others redirected to My Expenses
     */
    public function index()
    {
        $user = Auth::guard('admin')->user();
        
        // Non super-admin redirected to My Expenses
        if (!$user->hasRole('super-admin')) {
            return redirect()->route('admin.expense.my.index');
        }
        
        $stats = $this->getExpenseStats();
        $currency = Currency::getDefault();
        
        return view('expense::index', compact('stats', 'currency'));
    }

    /**
     * Get expense statistics
     */
    protected function getExpenseStats(): array
    {
        $user = Auth::guard('admin')->user();
        $baseQuery = Expense::query();
        
        // If not admin, filter by employee or manager
        if (!$user->hasRole('super-admin')) {
            $baseQuery->where(function ($q) use ($user) {
                $q->where('employee_id', $user->id)
                  ->orWhere('manager_id', $user->id)
                  ->orWhere('created_by', $user->id);
            });
        }

        return [
            'total' => (clone $baseQuery)->count(),
            'draft' => (clone $baseQuery)->status('draft')->count(),
            'submitted' => (clone $baseQuery)->status('submitted')->count(),
            'approved' => (clone $baseQuery)->status('approved')->count(),
            'rejected' => (clone $baseQuery)->status('rejected')->count(),
            'paid' => (clone $baseQuery)->status('paid')->count(),
            'pending_payment' => (clone $baseQuery)->approvedUnpaid()->count(),
            'total_amount' => (clone $baseQuery)->sum('amount'),
            'total_paid' => (clone $baseQuery)->sum('total_paid'),
            'pending_approval' => (clone $baseQuery)->pendingApproval()->count(),
        ];
    }

    /**
     * DataTable endpoint
     */
    public function dataTable(Request $request)
    {
        if ($request->isMethod('post') && $request->hasFile('file')) {
            return $this->importExpenses($request);
        }

        if ($request->has('template')) {
            return $this->downloadTemplate();
        }

        if ($request->has('export')) {
            return $this->exportExpenses($request);
        }

        return $this->listExpenses($request);
    }

    /**
     * List expenses with filters
     */
    protected function listExpenses(Request $request): JsonResponse
    {
        $user = Auth::guard('admin')->user();
        $query = Expense::with(['category', 'employee', 'manager', 'createdByAdmin']);

        // Permission-based filtering
        if (!$user->hasRole('super-admin')) {
            if ($this->canApproveExpenses()) {
                // Managers see expenses assigned to them + their own
                $query->where(function ($q) use ($user) {
                    $q->where('manager_id', $user->id)
                      ->orWhere('employee_id', $user->id)
                      ->orWhere('created_by', $user->id);
                });
            } else {
                // Regular users see only their own
                $query->where(function ($q) use ($user) {
                    $q->where('employee_id', $user->id)
                      ->orWhere('created_by', $user->id);
                });
            }
        }

        // Search
        if ($search = $request->input('search')) {
            $query->search($search);
        }

        // Filters
        $this->applyFilters($query, $request);

        // Sort
        $sortCol = $request->input('sort', 'id');
        $sortDir = $request->input('dir', 'desc');
        $sortable = ['id', 'expense_number', 'title', 'amount', 'expense_date', 'status', 'payment_status', 'created_at'];
        if (in_array($sortCol, $sortable)) {
            $query->orderBy($sortCol, $sortDir);
        } else {
            $query->orderBy('id', 'desc');
        }

        // Paginate
        $perPage = $request->input('per_page', config('expense.per_page', 15));
        $data = $query->paginate($perPage);

        $items = collect($data->items())->map(function ($item) {
            return [
                'id' => $item->id,
                'expense_number' => $item->expense_number,
                'title' => $item->title,
                'category_name' => $item->category?->name,
                'employee_name' => $item->employee?->name,
                'manager_name' => $item->manager?->name,
                'amount' => $item->amount,
                'expense_date' => $item->expense_date?->format('Y-m-d'),
                'status' => $item->status,
                'status_label' => $item->status_label,
                'status_badge' => $item->status_badge,
                'payment_status' => $item->payment_status,
                'payment_status_label' => $item->payment_status_label,
                'payment_status_badge' => $item->payment_status_badge,
                'total_paid' => $item->total_paid,
                'remaining_balance' => $item->remaining_balance,
                'days_pending' => $item->days_pending,
                'is_urgent' => $item->is_urgent,
                '_edit_url' => route('admin.expense.edit', $item->id),
                '_show_url' => route('admin.expense.show', $item->id),
            ];
        });

        return response()->json([
            'data' => $items,
            'total' => $data->total(),
            'current_page' => $data->currentPage(),
            'last_page' => $data->lastPage(),
        ]);
    }

    /**
     * Apply filters
     */
    protected function applyFilters($query, Request $request)
    {
        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }

        if ($request->filled('payment_status')) {
            $query->where('payment_status', $request->input('payment_status'));
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->input('category_id'));
        }

        if ($request->filled('employee_id')) {
            $query->where('employee_id', $request->input('employee_id'));
        }

        if ($request->filled('manager_id')) {
            $query->where('manager_id', $request->input('manager_id'));
        }

        if ($request->filled('from_date')) {
            $query->whereDate('expense_date', '>=', $request->input('from_date'));
        }

        if ($request->filled('to_date')) {
            $query->whereDate('expense_date', '<=', $request->input('to_date'));
        }

        if ($request->filled('min_amount')) {
            $query->where('amount', '>=', $request->input('min_amount'));
        }

        if ($request->filled('max_amount')) {
            $query->where('amount', '<=', $request->input('max_amount'));
        }
    }

    // ==================== CRUD OPERATIONS ====================

    /**
     * Show create form - All users can create their own expenses
     */
    public function create()
    {
        $data = $this->getFormData();
        return view('expense::create', $data);
    }

    /**
     * Store new expense - All users can create their own expenses
     */
    public function store(Request $request)
    {
        $validated = $this->validateExpense($request);
        
        DB::beginTransaction();
        try {
            // Calculate amounts
            $subtotal = (float) $validated['unit_price'] * (float) $validated['quantity'];
            $taxAmount = $subtotal * ((float) ($validated['tax_rate'] ?? 0) / 100);
            $tax2Amount = $subtotal * ((float) ($validated['tax2_rate'] ?? 0) / 100);
            $amount = $subtotal + $taxAmount + $tax2Amount;

            $expense = Expense::create([
                'expense_number' => Expense::generateExpenseNumber(),
                'title' => $validated['title'],
                'notes' => $validated['notes'] ?? null,
                'expense_date' => $validated['expense_date'],
                'category_id' => $validated['category_id'],
                'unit_id' => $validated['unit_id'] ?? null,
                'unit_price' => $validated['unit_price'],
                'quantity' => $validated['quantity'],
                'subtotal' => $subtotal,
                'tax_rate' => $validated['tax_rate'] ?? 0,
                'tax_amount' => $taxAmount,
                'tax2_rate' => $validated['tax2_rate'] ?? 0,
                'tax2_amount' => $tax2Amount,
                'amount' => $amount,
                'currency' => $validated['currency'] ?? config('expense.default_currency', 'INR'),
                'vendor' => $validated['vendor'] ?? null,
                'reference_no' => $validated['reference_no'] ?? null,
                'account_code' => $validated['account_code'] ?? null,
                'paid_by' => $validated['paid_by'] ?? 'employee',
                'payment_method_id' => $validated['payment_method_id'] ?? null,
                'employee_id' => $validated['employee_id'],
                'manager_id' => $validated['manager_id'] ?? null,
                'status' => 'draft',
                'is_billable' => $request->boolean('is_billable'),
                'client_id' => $validated['client_id'] ?? null,
                'created_by' => Auth::guard('admin')->id(),
            ]);

            // Handle attachments
            if ($request->hasFile('attachments')) {
                $this->handleAttachments($expense, $request->file('attachments'), $request->input('attachment_types', []));
            }

            // Log activity
            $expense->logActivity('created', 'Expense created');

            DB::commit();
            
            return redirect()->route('admin.expense.index')
                ->with('success', 'Expense created successfully! Expense #' . $expense->expense_number);
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Failed to create expense: ' . $e->getMessage());
        }
    }

    /**
     * Show expense details
     */
    public function show($id)
    {
        $expense = Expense::with([
            'category', 'unit', 'employee', 'manager', 
            'approvedByAdmin', 'paidByAdmin', 'createdByAdmin',
            'attachments', 'payments.paymentMethod', 'payments.createdByAdmin',
            'activityLogs.createdByAdmin'
        ])->findOrFail($id);

        $this->authorizeView($expense);

        return view('expense::show', compact('expense'));
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $expense = Expense::with(['attachments'])->findOrFail($id);
        
        $this->authorizeEdit($expense);

        if (!$expense->canBeEdited()) {
            return redirect()->route('admin.expense.show', $id)
                ->with('warning', 'This expense cannot be edited in its current status.');
        }

        $data = $this->getFormData();
        $data['expense'] = $expense;
        
        return view('expense::edit', $data);
    }

    /**
     * Update expense
     */
    public function update(Request $request, $id)
    {
        $expense = Expense::findOrFail($id);
        
        $this->authorizeEdit($expense);

        if (!$expense->canBeEdited()) {
            return redirect()->route('admin.expense.show', $id)
                ->with('warning', 'This expense cannot be edited in its current status.');
        }

        $validated = $this->validateExpense($request, $id);

        DB::beginTransaction();
        try {
            $oldValues = $expense->toArray();

            // Calculate amounts
            $subtotal = (float) $validated['unit_price'] * (float) $validated['quantity'];
            $taxAmount = $subtotal * ((float) ($validated['tax_rate'] ?? 0) / 100);
            $tax2Amount = $subtotal * ((float) ($validated['tax2_rate'] ?? 0) / 100);
            $amount = $subtotal + $taxAmount + $tax2Amount;

            $expense->update([
                'title' => $validated['title'],
                'notes' => $validated['notes'] ?? null,
                'expense_date' => $validated['expense_date'],
                'category_id' => $validated['category_id'],
                'unit_id' => $validated['unit_id'] ?? null,
                'unit_price' => $validated['unit_price'],
                'quantity' => $validated['quantity'],
                'subtotal' => $subtotal,
                'tax_rate' => $validated['tax_rate'] ?? 0,
                'tax_amount' => $taxAmount,
                'tax2_rate' => $validated['tax2_rate'] ?? 0,
                'tax2_amount' => $tax2Amount,
                'amount' => $amount,
                'vendor' => $validated['vendor'] ?? null,
                'reference_no' => $validated['reference_no'] ?? null,
                'account_code' => $validated['account_code'] ?? null,
                'paid_by' => $validated['paid_by'] ?? 'employee',
                'payment_method_id' => $validated['payment_method_id'] ?? null,
                'employee_id' => $validated['employee_id'],
                'manager_id' => $validated['manager_id'] ?? null,
                'is_billable' => $request->boolean('is_billable'),
                'client_id' => $validated['client_id'] ?? null,
                // Reset rejection if resubmitting
                'status' => $expense->status === 'rejected' ? 'draft' : $expense->status,
                'rejection_reason' => $expense->status === 'rejected' ? null : $expense->rejection_reason,
            ]);

            // Handle new attachments
            if ($request->hasFile('attachments')) {
                $this->handleAttachments($expense, $request->file('attachments'), $request->input('attachment_types', []));
            }

            // Log activity
            $expense->logActivity('updated', 'Expense updated', $oldValues, $expense->fresh()->toArray());

            DB::commit();

            return redirect()->route('admin.expense.index')
                ->with('success', 'Expense updated successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Failed to update expense: ' . $e->getMessage());
        }
    }

    /**
     * Delete expense
     */
    public function destroy($id)
    {
        $expense = Expense::findOrFail($id);
        $user = Auth::guard('admin')->user();
        $userId = (int) $user->id;
        
        // Check ownership (except super-admin)
        if (!$user->hasRole('super-admin')) {
            $isOwner = ((int) $expense->created_by === $userId) || ((int) $expense->employee_id === $userId);
            
            if (!$isOwner) {
                if (request()->ajax()) {
                    return response()->json(['success' => false, 'message' => 'You can only delete your own expenses.'], 403);
                }
                return back()->with('error', 'You can only delete your own expenses.');
            }
            
            // Only draft expenses can be deleted by regular users
            if ($expense->status !== 'draft') {
                $statusMessages = [
                    'submitted' => 'This expense is pending approval and cannot be deleted.',
                    'approved' => 'You don\'t have access to delete approved expenses.',
                    'rejected' => 'You don\'t have access to delete rejected expenses.',
                    'paid' => 'You don\'t have access to delete paid expenses.',
                ];
                $message = $statusMessages[$expense->status] ?? 'You cannot delete this expense.';
                
                if (request()->ajax()) {
                    return response()->json(['success' => false, 'message' => $message], 403);
                }
                return back()->with('error', $message);
            }
        } else {
            // Super-admin can delete draft/rejected only
            if (!in_array($expense->status, ['draft', 'rejected'])) {
                if (request()->ajax()) {
                    return response()->json(['success' => false, 'message' => 'Cannot delete submitted/approved/paid expenses.'], 400);
                }
                return back()->with('error', 'Cannot delete submitted/approved/paid expenses.');
            }
        }

        // Delete attachments
        foreach ($expense->attachments as $attachment) {
            $attachment->deleteFile();
        }

        $expense->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'Expense deleted successfully!']);
        }
        
        return redirect()->route('admin.expense.my.index')->with('success', 'Expense deleted successfully!');
    }

    /**
     * Bulk delete - Only super-admin can bulk delete
     */
    public function bulkDelete(Request $request): JsonResponse
    {
        $user = Auth::guard('admin')->user();
        
        if (!$user->hasRole('super-admin')) {
            return response()->json(['success' => false, 'message' => 'Only administrators can bulk delete expenses.'], 403);
        }
        
        $ids = $request->input('ids', []);
        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No items selected'], 400);
        }

        $deletable = Expense::whereIn('id', $ids)
            ->whereIn('status', ['draft', 'rejected'])
            ->get();

        foreach ($deletable as $expense) {
            foreach ($expense->attachments as $attachment) {
                $attachment->deleteFile();
            }
            $expense->delete();
        }

        $count = $deletable->count();
        $skipped = count($ids) - $count;
        
        $message = "{$count} expense(s) deleted!";
        if ($skipped > 0) {
            $message .= " {$skipped} skipped (cannot delete submitted/approved expenses).";
        }

        return response()->json(['success' => true, 'message' => $message]);
    }

    // ==================== HELPER METHODS ====================

    /**
     * Get form data for create/edit
     */
    protected function getFormData(): array
    {
        $user = Auth::guard('admin')->user();
        
        // Get projects from the projects table
        $projects = [];
        if (\Schema::hasTable('projects')) {
            $projects = \DB::table('projects')
                ->select('id', 'title')
                ->orderBy('title')
                ->get();
        }
        
        // Get default currency from database
        $defaultCurrency = Currency::getDefault();
        $currencySymbol = $defaultCurrency ? $defaultCurrency->symbol : config('expense.currency_symbol', '₹');
        
        return [
            'categories' => ExpenseCategory::active()->orderBy('name')->get(),
            'units' => ExpenseUnit::active()->orderBy('name')->get(),
            'paymentMethods' => PaymentMethod::forExpenses()->ordered()->get(),
            'employees' => Admin::where('is_active', true)->orderBy('name')->get(),
            'managers' => $this->getAvailableManagers(),
            'defaultManager' => ExpenseManagerAssignment::getPrimaryManagerFor($user->id),
            'projects' => $projects,
            'currency' => $defaultCurrency,
            'currencySymbol' => $currencySymbol,
            'travelRate' => config('expense.default_travel_rate_per_km', 10),
            'lockTravelRate' => config('expense.lock_travel_rate_for_users', true) && !$user->hasRole('super-admin'),
        ];
    }

    /**
     * Get available managers
     */
    protected function getAvailableManagers()
    {
        // Get admins with manager/admin roles
        return Admin::where('is_active', true)
            ->whereHas('roles', function ($q) {
                $q->whereIn('name', ['super-admin', 'admin', 'manager', 'supervisor']);
            })
            ->orderBy('name')
            ->get();
    }

    /**
     * Validate expense request
     */
    protected function validateExpense(Request $request, $id = null): array
    {
        return $request->validate([
            'title' => 'required|string|max:255',
            'notes' => 'nullable|string',
            'expense_date' => 'required|date',
            'category_id' => 'required|exists:expense_categories,id',
            'project_id' => 'nullable|integer',
            'unit_id' => 'nullable|exists:expense_units,id',
            'unit_price' => 'required|numeric|min:0',
            'quantity' => 'required|numeric|min:0.0001',
            'tax_rate' => 'nullable|numeric|min:0|max:100',
            'tax2_rate' => 'nullable|numeric|min:0|max:100',
            'vendor' => 'nullable|string|max:255',
            'reference_no' => 'nullable|string|max:100',
            'account_code' => 'nullable|string|max:50',
            'paid_by' => 'nullable|in:employee,company',
            'payment_method_id' => 'nullable|exists:payment_methods,id',
            'employee_id' => 'required|exists:admins,id',
            'manager_id' => 'nullable|exists:admins,id',
            'currency' => 'nullable|string|max:10',
            'is_billable' => 'nullable|boolean',
            'client_id' => 'nullable|integer',
            'attachments.*' => 'nullable|file|max:' . config('expense.max_file_size', 5120),
        ]);
    }

    /**
     * Handle file attachments
     */
    protected function handleAttachments(Expense $expense, array $files, array $types = []): void
    {
        foreach ($files as $index => $file) {
            if (!$file->isValid()) continue;

            $path = $file->store('expenses/' . $expense->id, 'public');
            
            ExpenseAttachment::create([
                'expense_id' => $expense->id,
                'original_name' => $file->getClientOriginalName(),
                'file_path' => $path,
                'file_type' => $file->getMimeType(),
                'file_size' => $file->getSize(),
                'attachment_type' => $types[$index] ?? 'receipt',
                'created_by' => Auth::guard('admin')->id(),
            ]);
        }
    }

    /**
     * Check permission helper
     */
    protected function checkPermission(string $permission): void
    {
        $user = Auth::guard('admin')->user();
        if (!$user->hasRole('super-admin') && !$user->can($permission)) {
            abort(403, 'Unauthorized access.');
        }
    }

    /**
     * Check if user can approve expenses
     */
    protected function canApproveExpenses(): bool
    {
        $user = Auth::guard('admin')->user();
        return $user->hasRole('super-admin') || $user->can('expense.approve');
    }

    /**
     * Check if user can pay expenses
     */
    protected function canPayExpenses(): bool
    {
        $user = Auth::guard('admin')->user();
        return $user->hasRole('super-admin') || $user->can('expense.pay');
    }

    /**
     * Authorize view
     */
    protected function authorizeView(Expense $expense): void
    {
        $user = Auth::guard('admin')->user();
        if ($user->hasRole('super-admin')) return;
        
        if ($expense->employee_id !== $user->id && 
            $expense->manager_id !== $user->id && 
            $expense->created_by !== $user->id &&
            !$this->canApproveExpenses()) {
            abort(403, 'You do not have permission to view this expense.');
        }
    }

    /**
     * Authorize edit
     */
    protected function authorizeEdit(Expense $expense): void
    {
        $user = Auth::guard('admin')->user();
        if ($user->hasRole('super-admin')) return;
        
        if ($expense->created_by !== $user->id && $expense->employee_id !== $user->id) {
            abort(403, 'You do not have permission to edit this expense.');
        }
    }

    /**
     * Authorize delete
     */
    protected function authorizeDelete(Expense $expense): void
    {
        $user = Auth::guard('admin')->user();
        if ($user->hasRole('super-admin')) return;
        
        if ($expense->created_by !== $user->id) {
            abort(403, 'You do not have permission to delete this expense.');
        }
    }

    /**
     * Download attachment
     */
    public function downloadAttachment($id)
    {
        $attachment = ExpenseAttachment::with('expense')->findOrFail($id);
        $expense = $attachment->expense;
        
        // Check if user can view this expense
        $user = Auth::guard('admin')->user();
        if (!$user->hasRole('super-admin')) {
            $userId = (int) $user->id;
            $canView = ((int) $expense->employee_id === $userId) || 
                       ((int) $expense->manager_id === $userId) || 
                       ((int) $expense->created_by === $userId);
            
            if (!$canView) {
                abort(403, 'You do not have permission to view this attachment.');
            }
        }
        
        // Check if file exists
        if (!Storage::disk('public')->exists($attachment->file_path)) {
            abort(404, 'File not found.');
        }
        
        return Storage::disk('public')->download($attachment->file_path, $attachment->original_name);
    }

    /**
     * Generate PDF for expense
     */
    public function generatePdf($id)
    {
        $expense = Expense::with([
            'category', 
            'unit', 
            'employee', 
            'manager', 
            'approvedByAdmin', 
            'submittedByAdmin',
            'paidByAdmin',
            'payments.paymentMethod',
            'payments.processedByAdmin',
            'attachments'
        ])->findOrFail($id);
        
        // Check if user can view this expense
        $user = Auth::guard('admin')->user();
        if (!$user->hasRole('super-admin')) {
            $userId = (int) $user->id;
            $canView = ((int) $expense->employee_id === $userId) || 
                       ((int) $expense->manager_id === $userId) || 
                       ((int) $expense->created_by === $userId);
            
            if (!$canView) {
                abort(403, 'You do not have permission to view this expense.');
            }
        }
        
        // Generate PDF using DomPDF - try multiple view paths
        $html = view('expense::pdf.expense', compact('expense'))->render();
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
        
        // Set paper size and orientation
        $pdf->setPaper('A4', 'portrait');
        
        // Return PDF as download or stream
        $filename = 'Expense-' . $expense->expense_number . '.pdf';
        
        return $pdf->download($filename);
    }

    /**
     * Stream PDF for expense (view in browser)
     */
    public function viewPdf($id)
    {
        $expense = Expense::with([
            'category', 
            'unit', 
            'employee', 
            'manager', 
            'approvedByAdmin', 
            'submittedByAdmin',
            'paidByAdmin',
            'payments.paymentMethod',
            'payments.processedByAdmin',
            'attachments'
        ])->findOrFail($id);
        
        // Check if user can view this expense
        $user = Auth::guard('admin')->user();
        if (!$user->hasRole('super-admin')) {
            $userId = (int) $user->id;
            $canView = ((int) $expense->employee_id === $userId) || 
                       ((int) $expense->manager_id === $userId) || 
                       ((int) $expense->created_by === $userId);
            
            if (!$canView) {
                abort(403, 'You do not have permission to view this expense.');
            }
        }
        
        // Generate PDF using DomPDF
        $html = view('expense::pdf.expense', compact('expense'))->render();
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadHTML($html);
        
        // Set paper size and orientation
        $pdf->setPaper('A4', 'portrait');
        
        // Return PDF as stream (view in browser)
        return $pdf->stream('Expense-' . $expense->expense_number . '.pdf');
    }
}