<?php

namespace Modules\Expense\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Models\Admin;
use Modules\Expense\Models\Currency;

class Expense extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'expense_number',
        'title',
        'notes',
        'expense_date',
        'category_id',
        'project_id',
        'unit_id',
        'unit_price',
        'quantity',
        'subtotal',
        'tax_rate',
        'tax_amount',
        'tax2_rate',
        'tax2_amount',
        'amount',
        'currency',
        'vendor',
        'reference_no',
        'account_code',
        'paid_by',
        'payment_method_id',
        'employee_id',
        'manager_id',
        'status',
        'submitted_by',
        'submitted_at',
        'approved_by',
        'approved_at',
        'rejection_reason',
        'payment_status',
        'total_paid',
        'paid_at',
        'paid_by_admin',
        'is_billable',
        'client_id',
        'created_by',
    ];

   protected $casts = [
    'expense_date' => 'date',
    'unit_price' => 'float',
    'quantity' => 'float',
    'subtotal' => 'float',
    'tax_rate' => 'float',
    'tax_amount' => 'float',
    'tax2_rate' => 'float',
    'tax2_amount' => 'float',
    'amount' => 'float',
    'total_paid' => 'float',
    'is_billable' => 'boolean',
    'submitted_at' => 'datetime',
    'approved_at' => 'datetime',
    'paid_at' => 'datetime',
];

    protected $appends = [
        'status_label',
        'status_badge',
        'payment_status_label',
        'payment_status_badge',
        'remaining_balance',
        'formatted_amount',
    ];

    // ==================== RELATIONSHIPS ====================

    public function category(): BelongsTo
    {
        return $this->belongsTo(ExpenseCategory::class, 'category_id');
    }

    public function unit(): BelongsTo
    {
        return $this->belongsTo(ExpenseUnit::class, 'unit_id');
    }

    public function project(): BelongsTo
    {
        // Try to get project using dynamic model discovery
        if (class_exists('\App\Models\Admin\Project')) {
            return $this->belongsTo(\App\Models\Admin\Project::class, 'project_id');
        } elseif (class_exists('\App\Models\Project')) {
            return $this->belongsTo(\App\Models\Project::class, 'project_id');
        } elseif (class_exists('\Modules\Project\Models\Project')) {
            return $this->belongsTo(\Modules\Project\Models\Project::class, 'project_id');
        }
        // Fallback - return empty relationship
        return $this->belongsTo(self::class, 'project_id')->whereRaw('1 = 0');
    }

    /**
     * Get project title directly from database
     */
    public function getProjectTitleAttribute(): ?string
    {
        if (!$this->project_id) return null;
        return \DB::table('projects')->where('id', $this->project_id)->value('title');
    }

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'employee_id');
    }

    public function manager(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'manager_id');
    }

    public function approvedByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'approved_by');
    }

    public function submittedByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'submitted_by');
    }

    public function paidByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'paid_by_admin');
    }

    public function createdByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    public function paymentMethod(): BelongsTo
    {
        return $this->belongsTo(PaymentMethod::class, 'payment_method_id');
    }

    public function currencyModel(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'currency', 'code');
    }

    public function attachments(): HasMany
    {
        return $this->hasMany(ExpenseAttachment::class, 'expense_id');
    }

    public function payments(): HasMany
    {
        return $this->hasMany(ExpensePayment::class, 'expense_id');
    }

    public function activityLogs(): HasMany
    {
        return $this->hasMany(ExpenseActivityLog::class, 'expense_id');
    }

    // ==================== SCOPES ====================

    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) return $query;
        return $query->where(function ($q) use ($search) {
            $q->where('title', 'LIKE', "%{$search}%")
              ->orWhere('expense_number', 'LIKE', "%{$search}%")
              ->orWhere('vendor', 'LIKE', "%{$search}%")
              ->orWhere('reference_no', 'LIKE', "%{$search}%");
        });
    }

    public function scopeStatus(Builder $query, string $status): Builder
    {
        return $query->where('status', $status);
    }

    public function scopeDraft(Builder $query): Builder
    {
        return $query->where('status', 'draft');
    }

    public function scopeSubmitted(Builder $query): Builder
    {
        return $query->where('status', 'submitted');
    }

    public function scopeApproved(Builder $query): Builder
    {
        return $query->where('status', 'approved');
    }

    public function scopeRejected(Builder $query): Builder
    {
        return $query->where('status', 'rejected');
    }

    public function scopePaid(Builder $query): Builder
    {
        return $query->where('status', 'paid');
    }

    public function scopePendingApproval(Builder $query): Builder
    {
        return $query->whereIn('status', ['draft', 'submitted']);
    }

    public function scopeApprovedUnpaid(Builder $query): Builder
    {
        return $query->where('status', 'approved')
                     ->where('payment_status', '!=', 'paid');
    }

    public function scopeForEmployee(Builder $query, int $employeeId): Builder
    {
        return $query->where('employee_id', $employeeId);
    }

    public function scopeForManager(Builder $query, int $managerId): Builder
    {
        return $query->where('manager_id', $managerId);
    }

    public function scopeDateRange(Builder $query, ?string $from, ?string $to): Builder
    {
        if ($from) {
            $query->whereDate('expense_date', '>=', $from);
        }
        if ($to) {
            $query->whereDate('expense_date', '<=', $to);
        }
        return $query;
    }

    // ==================== ACCESSORS ====================

    public function getStatusLabelAttribute(): string
    {
        return config('expense.statuses.' . $this->status, ucfirst($this->status));
    }

    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'draft' => 'secondary',
            'submitted' => 'info',
            'approved' => 'success',
            'rejected' => 'danger',
            'paid' => 'primary',
            default => 'secondary',
        };
    }

    public function getPaymentStatusLabelAttribute(): string
    {
        return config('expense.payment_statuses.' . $this->payment_status, ucfirst($this->payment_status));
    }

    public function getPaymentStatusBadgeAttribute(): string
    {
        return match($this->payment_status) {
            'unpaid' => 'danger',
            'partially_paid' => 'warning',
            'paid' => 'success',
            default => 'secondary',
        };
    }

    public function getRemainingBalanceAttribute(): float
    {
        return (float) $this->amount - (float) $this->total_paid;
    }

    public function getFormattedAmountAttribute(): string
    {
        // Try to get currency from database
        $currency = $this->currencyModel ?? Currency::getDefault();
        
        if ($currency) {
            return $currency->formatWithoutSymbol($this->amount);
        }
        
        // Fallback to config
        $symbol = config('expense.currency_symbol', '₹');
        return $symbol . number_format($this->amount, 2);
    }

    public function getFormattedRemainingBalanceAttribute(): string
    {
        // Try to get currency from database
        $currency = $this->currencyModel ?? Currency::getDefault();
        
        if ($currency) {
            return $currency->formatWithoutSymbol($this->remaining_balance);
        }
        
        // Fallback to config
        $symbol = config('expense.currency_symbol', '₹');
        return $symbol . number_format($this->remaining_balance, 2);
    }

    public function getDaysPendingAttribute(): int
    {
        if (!in_array($this->status, ['draft', 'submitted'])) {
            return 0;
        }
        $date = $this->submitted_at ?? $this->created_at;
        return $date ? $date->diffInDays(now()) : 0;
    }

    public function getIsUrgentAttribute(): bool
    {
        return $this->days_pending > 7;
    }

    // ==================== METHODS ====================

    public static function generateExpenseNumber(): string
    {
        $format = config('expense.number_format', 'EXP{YYYY}{MM}{0000}');
        $year = date('Y');
        $month = date('m');
        
        // Get the last expense number for this month
        $lastExpense = self::whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->orderBy('id', 'desc')
            ->first();
        
        $sequence = 1;
        if ($lastExpense && $lastExpense->expense_number) {
            // Extract the sequence number from the last expense number
            preg_match('/(\d{4})$/', $lastExpense->expense_number, $matches);
            if (!empty($matches[1])) {
                $sequence = (int) $matches[1] + 1;
            }
        }
        
        $number = str_replace(
            ['{YYYY}', '{MM}', '{0000}'],
            [$year, $month, str_pad($sequence, 4, '0', STR_PAD_LEFT)],
            $format
        );
        
        return $number;
    }

    public function calculateAmounts(): void
    {
        $this->subtotal = (float) $this->unit_price * (float) $this->quantity;
        $this->tax_amount = $this->subtotal * ((float) $this->tax_rate / 100);
        $this->tax2_amount = $this->subtotal * ((float) $this->tax2_rate / 100);
        $this->amount = $this->subtotal + $this->tax_amount + $this->tax2_amount;
    }

    public function updatePaymentStatus(): void
    {
        $totalPaid = $this->payments()->where('status', 'completed')->sum('amount');
        $this->total_paid = $totalPaid;
        
        if ($totalPaid >= $this->amount) {
            $this->payment_status = 'paid';
            if ($this->status === 'approved') {
                $this->status = 'paid';
                $this->paid_at = now();
            }
        } elseif ($totalPaid > 0) {
            $this->payment_status = 'partially_paid';
        } else {
            $this->payment_status = 'unpaid';
        }
        
        $this->save();
    }

    public function canBeEdited(): bool
    {
        return in_array($this->status, ['draft', 'rejected']);
    }

    public function canBeSubmitted(): bool
    {
        return in_array($this->status, ['draft', 'rejected']);
    }

    public function canBeApproved(): bool
    {
        return in_array($this->status, ['draft', 'submitted']);
    }

    public function canBePaid(): bool
    {
        return $this->status === 'approved' && $this->payment_status !== 'paid';
    }

    public function logActivity(string $action, ?string $description = null, ?array $oldValues = null, ?array $newValues = null): void
    {
        ExpenseActivityLog::create([
            'expense_id' => $this->id,
            'action' => $action,
            'description' => $description,
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'created_by' => auth()->guard('admin')->id(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
        ]);
    }
}
