<?php

namespace Modules\Expense\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\Admin;

class ExpenseActivityLog extends Model
{
    public $timestamps = false;

    protected $table = 'expense_activity_log';

    protected $fillable = [
        'expense_id',
        'action',
        'description',
        'old_values',
        'new_values',
        'manager_id',
        'created_by',
        'ip_address',
        'user_agent',
        'created_at',
    ];

    protected $casts = [
        'old_values' => 'array',
        'new_values' => 'array',
        'created_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function expense(): BelongsTo
    {
        return $this->belongsTo(Expense::class, 'expense_id');
    }

    public function manager(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'manager_id');
    }

    public function createdByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    // ==================== ACCESSORS ====================

    public function getActionLabelAttribute(): string
    {
        return match($this->action) {
            'created' => 'Created',
            'updated' => 'Updated',
            'submitted' => 'Submitted for Approval',
            'approved' => 'Approved',
            'rejected' => 'Rejected',
            'payment_added' => 'Payment Added',
            'attachment_added' => 'Attachment Added',
            'attachment_removed' => 'Attachment Removed',
            default => ucfirst(str_replace('_', ' ', $this->action)),
        };
    }

    public function getActionIconAttribute(): string
    {
        return match($this->action) {
            'created' => 'plus-circle',
            'updated' => 'edit',
            'submitted' => 'send',
            'approved' => 'check-circle',
            'rejected' => 'x-circle',
            'payment_added' => 'credit-card',
            'attachment_added' => 'paperclip',
            'attachment_removed' => 'trash-2',
            default => 'activity',
        };
    }

    public function getActionColorAttribute(): string
    {
        return match($this->action) {
            'created' => 'primary',
            'updated' => 'info',
            'submitted' => 'warning',
            'approved' => 'success',
            'rejected' => 'danger',
            'payment_added' => 'success',
            default => 'secondary',
        };
    }

    // ==================== BOOT ====================

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            $model->created_at = $model->created_at ?? now();
        });
    }
}
