<?php

namespace Modules\Expense\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Storage;
use App\Models\Admin;

class ExpenseAttachment extends Model
{
    protected $fillable = [
        'expense_id',
        'original_name',
        'file_path',
        'file_type',
        'file_size',
        'attachment_type',
        'attachment_label',
        'is_required',
        'validation_status',
        'validated_by',
        'validated_at',
        'created_by',
    ];

    protected $casts = [
        'file_size' => 'integer',
        'is_required' => 'boolean',
        'validated_at' => 'datetime',
    ];

    protected $appends = [
        'file_url',
        'formatted_file_size',
        'type_label',
    ];

    // ==================== RELATIONSHIPS ====================

    public function expense(): BelongsTo
    {
        return $this->belongsTo(Expense::class, 'expense_id');
    }

    public function validatedByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'validated_by');
    }

    public function createdByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    // ==================== ACCESSORS ====================

    public function getFileUrlAttribute(): ?string
    {
        if (!$this->file_path) return null;
        
        // Use download route instead of direct storage URL
        return route('admin.expense.attachment.download', $this->id);
    }

    public function getDirectUrlAttribute(): ?string
    {
        if (!$this->file_path) return null;
        return Storage::disk('public')->url($this->file_path);
    }

    public function getFormattedFileSizeAttribute(): string
    {
        $bytes = $this->file_size;
        if ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        }
        return $bytes . ' B';
    }

    public function getTypeLabelAttribute(): string
    {
        return config('expense.attachment_types.' . $this->attachment_type, ucfirst(str_replace('_', ' ', $this->attachment_type)));
    }

    public function getValidationStatusLabelAttribute(): string
    {
        return match($this->validation_status) {
            'pending' => 'Pending',
            'approved' => 'Approved',
            'rejected' => 'Rejected',
            default => 'Unknown',
        };
    }

    public function getValidationStatusBadgeAttribute(): string
    {
        return match($this->validation_status) {
            'pending' => 'warning',
            'approved' => 'success',
            'rejected' => 'danger',
            default => 'secondary',
        };
    }

    public function getIsImageAttribute(): bool
    {
        return in_array(strtolower($this->file_type), ['image/jpeg', 'image/png', 'image/gif', 'image/webp']);
    }

    public function getIsPdfAttribute(): bool
    {
        return strtolower($this->file_type) === 'application/pdf';
    }

    // ==================== METHODS ====================

    public function deleteFile(): bool
    {
        if ($this->file_path && Storage::exists($this->file_path)) {
            return Storage::delete($this->file_path);
        }
        return false;
    }

    protected static function boot()
    {
        parent::boot();
        
        static::deleting(function ($attachment) {
            $attachment->deleteFile();
        });
    }
}