<?php

namespace Modules\Expense\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ExpenseCategory extends Model
{
    protected $fillable = [
        'name',
        'description',
        'unit_price',
        'is_travel_category',
        'is_project_category',
        'is_active',
    ];

    protected $casts = [
        'unit_price' => 'decimal:2',
        'is_travel_category' => 'boolean',
        'is_project_category' => 'boolean',
        'is_active' => 'boolean',
    ];

    // ==================== RELATIONSHIPS ====================

    public function expenses(): HasMany
    {
        return $this->hasMany(Expense::class, 'category_id');
    }

    // ==================== SCOPES ====================

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_active', true);
    }

    public function scopeTravel(Builder $query): Builder
    {
        return $query->where('is_travel_category', true);
    }

    public function scopeProject(Builder $query): Builder
    {
        return $query->where('is_project_category', true);
    }

    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) return $query;
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
              ->orWhere('description', 'LIKE', "%{$search}%");
        });
    }

    // ==================== ACCESSORS ====================

    public function getStatusLabelAttribute(): string
    {
        return $this->is_active ? 'Active' : 'Inactive';
    }

    public function getStatusBadgeAttribute(): string
    {
        return $this->is_active ? 'success' : 'danger';
    }

    public function getFormattedUnitPriceAttribute(): string
    {
        if (!$this->unit_price) return '-';
        return '₹' . number_format($this->unit_price, 2);
    }
}
