<?php

namespace Modules\Expense\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Models\Admin;

class ExpenseManagerAssignment extends Model
{
    protected $fillable = [
        'employee_id',
        'manager_id',
        'department',
        'is_primary',
        'is_active',
        'notes',
        'assigned_by',
    ];

    protected $casts = [
        'is_primary' => 'boolean',
        'is_active' => 'boolean',
    ];

    // ==================== RELATIONSHIPS ====================

    public function employee(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'employee_id');
    }

    public function manager(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'manager_id');
    }

    public function assignedByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'assigned_by');
    }

    // ==================== SCOPES ====================

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_active', true);
    }

    public function scopePrimary(Builder $query): Builder
    {
        return $query->where('is_primary', true);
    }

    public function scopeForEmployee(Builder $query, int $employeeId): Builder
    {
        return $query->where('employee_id', $employeeId);
    }

    public function scopeForManager(Builder $query, int $managerId): Builder
    {
        return $query->where('manager_id', $managerId);
    }

    // ==================== STATIC METHODS ====================

    public static function getPrimaryManagerFor(int $employeeId): ?Admin
    {
        $assignment = self::active()
            ->primary()
            ->forEmployee($employeeId)
            ->first();
        
        return $assignment?->manager;
    }

    public static function getManagersFor(int $employeeId): \Illuminate\Database\Eloquent\Collection
    {
        return self::active()
            ->forEmployee($employeeId)
            ->with('manager')
            ->get()
            ->pluck('manager');
    }

    public static function getEmployeesFor(int $managerId): \Illuminate\Database\Eloquent\Collection
    {
        return self::active()
            ->forManager($managerId)
            ->with('employee')
            ->get()
            ->pluck('employee');
    }
}
