<?php

namespace Modules\Expense\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Storage;
use App\Models\Admin;

class ExpensePayment extends Model
{
    protected $fillable = [
        'expense_id',
        'payment_reference',
        'amount',
        'payment_date',
        'payment_method_id',
        'transaction_id',
        'note',
        'bank_details',
        'receipt_file',
        'status',
        'created_by',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'payment_date' => 'date',
    ];

    protected $appends = [
        'formatted_amount',
        'status_label',
        'status_badge',
    ];

    // ==================== RELATIONSHIPS ====================

    public function expense(): BelongsTo
    {
        return $this->belongsTo(Expense::class, 'expense_id');
    }

    public function paymentMethod(): BelongsTo
    {
        return $this->belongsTo(PaymentMethod::class, 'payment_method_id');
    }

    public function createdByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    /**
     * Alias for createdByAdmin - the admin who processed the payment
     */
    public function processedByAdmin(): BelongsTo
    {
        return $this->belongsTo(Admin::class, 'created_by');
    }

    // ==================== ACCESSORS ====================

    public function getFormattedAmountAttribute(): string
    {
        return '₹' . number_format($this->amount, 2);
    }

    public function getStatusLabelAttribute(): string
    {
        return match($this->status) {
            'pending' => 'Pending',
            'completed' => 'Completed',
            'failed' => 'Failed',
            'cancelled' => 'Cancelled',
            default => 'Unknown',
        };
    }

    public function getStatusBadgeAttribute(): string
    {
        return match($this->status) {
            'pending' => 'warning',
            'completed' => 'success',
            'failed' => 'danger',
            'cancelled' => 'secondary',
            default => 'secondary',
        };
    }

    public function getReceiptUrlAttribute(): ?string
    {
        if (!$this->receipt_file) return null;
        return Storage::url($this->receipt_file);
    }

    // ==================== METHODS ====================

    public static function generatePaymentReference(): string
    {
        $format = config('expense.payment_reference_format', 'PAY{YYYY}{MM}{0000}');
        $year = date('Y');
        $month = date('m');
        
        $lastPayment = self::whereYear('created_at', $year)
            ->whereMonth('created_at', $month)
            ->orderBy('id', 'desc')
            ->first();
        
        $sequence = 1;
        if ($lastPayment && $lastPayment->payment_reference) {
            preg_match('/(\d{4})$/', $lastPayment->payment_reference, $matches);
            if (!empty($matches[1])) {
                $sequence = (int) $matches[1] + 1;
            }
        }
        
        return str_replace(
            ['{YYYY}', '{MM}', '{0000}'],
            [$year, $month, str_pad($sequence, 4, '0', STR_PAD_LEFT)],
            $format
        );
    }

    protected static function boot()
    {
        parent::boot();
        
        static::created(function ($payment) {
            if ($payment->status === 'completed') {
                $payment->expense->updatePaymentStatus();
            }
        });

        static::updated(function ($payment) {
            $payment->expense->updatePaymentStatus();
        });

        static::deleted(function ($payment) {
            $payment->expense->updatePaymentStatus();
        });
    }
}
