<?php

namespace Modules\Expense\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ExpenseUnit extends Model
{
    protected $fillable = [
        'name',
        'symbol',
        'type',
        'description',
        'conversion_factor',
        'is_active',
        'created_by',
    ];

    protected $casts = [
        'conversion_factor' => 'decimal:4',
        'is_active' => 'boolean',
    ];

    // ==================== RELATIONSHIPS ====================

    public function expenses(): HasMany
    {
        return $this->hasMany(Expense::class, 'unit_id');
    }

    // ==================== SCOPES ====================

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_active', true);
    }

    public function scopeOfType(Builder $query, string $type): Builder
    {
        return $query->where('type', $type);
    }

    public function scopeDistance(Builder $query): Builder
    {
        return $query->where('type', 'distance');
    }

    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) return $query;
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
              ->orWhere('symbol', 'LIKE', "%{$search}%");
        });
    }

    // ==================== ACCESSORS ====================

    public function getDisplayNameAttribute(): string
    {
        return "{$this->name} ({$this->symbol})";
    }

    public function getStatusLabelAttribute(): string
    {
        return $this->is_active ? 'Active' : 'Inactive';
    }
}
