<?php

namespace Modules\Expense\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

/**
 * PaymentMethod Model
 * Compatible with existing payment_methods table structure:
 * - id, name, slug, description, icon, is_active, is_default, show_on_invoice, sort_order, created_at, updated_at
 */
class PaymentMethod extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'description',
        'icon',
        'is_active',
        'is_default',
        'show_on_invoice',
        'sort_order',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_default' => 'boolean',
        'show_on_invoice' => 'boolean',
    ];

    // ==================== SCOPES ====================

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_active', 1);
    }

    /**
     * For expenses - uses only is_active column
     * Compatible with existing payment_methods table (no show_on_expenses column)
     */
    public function scopeForExpenses(Builder $query): Builder
    {
        return $query->where('is_active', 1);
    }

    public function scopeOrdered(Builder $query): Builder
    {
        // sort_order may not exist in some older tables
        if (\Illuminate\Support\Facades\Schema::hasColumn('payment_methods', 'sort_order')) {
            return $query->orderBy('sort_order', 'asc')->orderBy('name', 'asc');
        }
        return $query->orderBy('name', 'asc');
    }

    // ==================== ACCESSORS ====================

    public function getStatusLabelAttribute(): string
    {
        return $this->is_active ? 'Active' : 'Inactive';
    }

    public function getStatusBadgeAttribute(): string
    {
        return $this->is_active ? 'success' : 'danger';
    }
}
