<?php

namespace Modules\Product\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Product\Models\Product;
use Illuminate\Http\Request;
use App\Traits\DataTable;

class ProductController extends AdminController
{
    use DataTable;

    // ==================== DATATABLE CONFIGURATION ====================
    protected $model = Product::class;
    
    protected $searchable = ['name', 'sku', 'description'];
    
    protected $sortable = ['id', 'name', 'sku', 'purchase_price', 'sale_price', 'mrp', 'is_active', 'created_at'];
    
    protected $filterable = ['is_active'];
    
    protected $uniqueField = 'sku';
    
    protected $exportTitle = 'Products Export';

    // Import validation rules
    protected $importable = [
        'name'           => 'required|string|max:191',
        'sku'            => 'required|string|max:100',
        'description'    => 'nullable|string',
        'purchase_price' => 'required|numeric|min:0',
        'sale_price'     => 'required|numeric|min:0',
        'mrp'            => 'nullable|numeric|min:0',
        'is_active'      => 'nullable|boolean',
    ];

    // ==================== CUSTOM ROW MAPPING FOR LIST ====================
    protected function mapRow($item)
    {
        return [
            'id' => $item->id,
            'name' => $item->name,
            'sku' => $item->sku,
            'description' => $item->description,
            'purchase_price' => number_format($item->purchase_price, 2),
            'sale_price' => number_format($item->sale_price, 2),
            'mrp' => number_format($item->mrp ?? 0, 2),
            'is_active' => $item->is_active,
            'status_label' => $item->status_label,
            '_edit_url' => route('admin.product.edit', $item->id),
            '_show_url' => route('admin.product.show', $item->id),
        ];
    }

    // ==================== CUSTOM EXPORT ROW MAPPING ====================
    protected function mapExportRow($item)
    {
        return [
            'ID' => $item->id,
            'Name' => $item->name,
            'SKU' => $item->sku,
            'Description' => $item->description,
            'Purchase Price' => $item->purchase_price,
            'Sale Price' => $item->sale_price,
            'MRP' => $item->mrp ?? 0,
            'Status' => $item->is_active ? 'Active' : 'Inactive',
            'Created At' => $item->created_at?->format('Y-m-d H:i'),
        ];
    }

    // ==================== CUSTOM IMPORT ROW HANDLER ====================
    protected function importRow($data, $row)
    {
        $data['is_active'] = $data['is_active'] ?? true;
        
        $existing = Product::where('sku', $data['sku'])->first();
        
        if ($existing) {
            $existing->update($data);
            return $existing;
        }
        
        return Product::create($data);
    }

    // ==================== DATA ENDPOINT ====================
    public function dataTable(Request $request)
    {
        return $this->handleData($request);
    }

    /**
     * Display listing
     */
    public function index()
    {
        $this->authorize('product.products-list.read');
        $stats = [
            'total' => Product::count(),
            'active' => Product::active()->count(),
            'inactive' => Product::where('is_active', false)->count(),
        ];
        return view('product::index', compact('stats'));
    }

    /**
     * Show create form
     */
    public function create()
    {
        $this->authorize('product.products-list.create');
        return view('product::create');
    }

    /**
     * Store new product
     */
    public function store(Request $request)
    {
        $this->authorize('product.products-list.create');
        $validated = $request->validate([
            'name' => 'required|string|max:191',
            'sku' => 'required|string|max:100|unique:products,sku',
            'description' => 'nullable|string',
            'purchase_price' => 'required|numeric|min:0',
            'sale_price' => 'required|numeric|min:0',
            'mrp' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);
        Product::create($validated);

        return redirect()->route('admin.product.index')->with('success', 'Product created!');
    }

    /**
     * Show single product
     */
    public function show($id)
    {
        $this->authorize('product.products-list.show');
        $product = Product::findOrFail($id);
        return view('product::show', compact('product'));
    }

    /**
     * Show edit form
     */
    public function edit($id)
    {
        $this->authorize('product.products-list.edit');
        $product = Product::findOrFail($id);
        return view('product::edit', compact('product'));
    }

    /**
     * Update product
     */
    public function update(Request $request, $id)
    {
        $this->authorize('product.products-list.edit');
        $product = Product::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:191',
            'sku' => 'required|string|max:100|unique:products,sku,' . $id,
            'description' => 'nullable|string',
            'purchase_price' => 'required|numeric|min:0',
            'sale_price' => 'required|numeric|min:0',
            'mrp' => 'nullable|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        $validated['is_active'] = $request->boolean('is_active', true);
        $product->update($validated);

        return redirect()->route('admin.product.index')->with('success', 'Product updated!');
    }

    /**
     * Delete product
     */
    public function destroy($id)
    {
        $this->authorize('product.products-list.delete');
        Product::findOrFail($id)->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'Product deleted!']);
        }
        return redirect()->route('admin.product.index')->with('success', 'Product deleted!');
    }

    /**
     * Bulk delete
     */
    public function bulkDelete(Request $request)
    {
        $ids = $request->input('ids', []);
        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No items selected'], 400);
        }
        $deleted = Product::whereIn('id', $ids)->delete();
        return response()->json(['success' => true, 'message' => "{$deleted} product(s) deleted!"]);
    }

    /**
     * Toggle product status
     */
    public function toggleStatus($id)
    {
        $product = Product::findOrFail($id);
        $product->is_active = !$product->is_active;
        $product->save();
        return response()->json(['success' => true, 'is_active' => $product->is_active]);
    }

    /**
     * Search products (for dropdowns)
     */
    public function search(Request $request)
    {
        $products = Product::active()
            ->search($request->input('q', ''))
            ->limit(10)
            ->get(['id', 'name', 'sku', 'sale_price', 'purchase_price']);

        return response()->json([
            'results' => $products->map(fn($p) => [
                'id' => $p->id,
                'text' => "{$p->name} ({$p->sku})",
                'name' => $p->name,
                'sku' => $p->sku,
                'sale_price' => $p->sale_price,
                'purchase_price' => $p->purchase_price,
            ])
        ]);
    }
}