<?php

namespace Modules\Product\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Schema;

class Product extends Model
{
    protected $table = 'products';

    protected $fillable = [
        'name',
        'sku',
        'description',
        'purchase_price',
        'sale_price',
        'mrp',
        'is_active',
        // Extra fields from Inventory module (if exists)
        'quantity',
        'unit_id',
        'category_id',
        'barcode',
        'hsn_code',
        'tax_rate',
    ];

    protected $casts = [
        'purchase_price' => 'decimal:2',
        'sale_price' => 'decimal:2',
        'mrp' => 'decimal:2',
        'is_active' => 'boolean',
    ];

    public function scopeActive(Builder $query): Builder
    {
        // Only apply if column exists
        if (Schema::hasColumn('products', 'is_active')) {
            return $query->where('is_active', true);
        }
        return $query;
    }

    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) return $query;
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
              ->orWhere('sku', 'LIKE', "%{$search}%");
        });
    }

    public function getFormattedSalePriceAttribute(): string
    {
        return number_format($this->sale_price ?? 0, 2);
    }

    public function getFormattedPurchasePriceAttribute(): string
    {
        return number_format($this->purchase_price ?? 0, 2);
    }

    public function getFormattedMrpAttribute(): string
    {
        return $this->mrp ? number_format($this->mrp, 2) : '-';
    }

    public function getStatusLabelAttribute(): string
    {
        return $this->is_active ? 'Active' : 'Inactive';
    }

    public function getStatusBadgeAttribute(): string
    {
        return $this->is_active ? 'success' : 'danger';
    }
}
