<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // ==================== PROJECT STATUSES ====================
        if (!Schema::hasTable('project_statuses')) {
            Schema::create('project_statuses', function (Blueprint $table) {
                $table->id();
                $table->string('label', 100);
                $table->string('color', 20)->default('#3b82f6');
                $table->boolean('show_by_default')->default(false);
                $table->integer('sort_order')->default(0);
                $table->boolean('is_active')->default(true);
                $table->timestamps();
            });
        }

        // Always ensure default statuses exist
        $this->seedDefaultStatuses();

        // ==================== PROJECTS ====================
        if (!Schema::hasTable('projects')) {
            Schema::create('projects', function (Blueprint $table) {
                $table->id();
                $table->string('title', 255);
                $table->text('description')->nullable();
                
                // Status
                $table->unsignedBigInteger('status_id')->nullable();
                $table->foreign('status_id')->references('id')->on('project_statuses')->onDelete('set null');
                
                // Client relationship
                $table->unsignedBigInteger('client_id')->nullable();
                
                // Dates
                $table->date('start_date')->nullable();
                $table->date('due_date')->nullable();
                $table->date('completed_date')->nullable();
                
                // Financial
                $table->decimal('budget', 15, 2)->default(0);
                $table->decimal('cost', 15, 2)->default(0);
                $table->enum('billing_type', ['fixed', 'hourly', 'milestone'])->default('fixed');
                $table->decimal('hourly_rate', 10, 2)->default(0);
                $table->decimal('estimated_hours', 10, 2)->default(0);
                
                // Progress
                $table->decimal('progress', 5, 2)->default(0);
                $table->enum('priority', ['low', 'medium', 'high', 'urgent'])->default('medium');
                
                // Settings
                $table->boolean('is_pinned')->default(false);
                $table->boolean('is_archived')->default(false);
                $table->boolean('is_public')->default(false);
                
                // Ownership
                $table->unsignedBigInteger('created_by')->nullable();
                $table->unsignedBigInteger('updated_by')->nullable();
                
                // Metadata
                $table->json('settings')->nullable();
                $table->json('tags')->nullable();
                
                $table->timestamps();
                $table->softDeletes();
                
                // Indexes
                $table->index('status_id');
                $table->index('client_id');
                $table->index('start_date');
                $table->index('due_date');
                $table->index('is_pinned');
                $table->index('is_archived');
                $table->index('created_by');
            });
        }

        // ==================== PROJECT TEAM ====================
        if (!Schema::hasTable('project_team')) {
            Schema::create('project_team', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('project_id');
                $table->unsignedBigInteger('staff_id');
                $table->string('role', 100)->nullable();
                $table->decimal('hourly_rate', 10, 2)->nullable();
                $table->boolean('is_lead')->default(false);
                $table->boolean('can_edit')->default(true);
                $table->boolean('receive_notifications')->default(true);
                $table->timestamps();
                
                $table->foreign('project_id')->references('id')->on('projects')->onDelete('cascade');
                $table->unique(['project_id', 'staff_id']);
                $table->index('staff_id');
            });
        }

        // ==================== PROJECT MILESTONES ====================
        if (!Schema::hasTable('project_milestones')) {
            Schema::create('project_milestones', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('project_id');
                $table->string('name', 255);
                $table->text('description')->nullable();
                $table->date('target_date')->nullable();
                $table->date('completed_date')->nullable();
                $table->boolean('is_completed')->default(false);
                $table->decimal('amount', 15, 2)->default(0);
                $table->boolean('is_billable')->default(false);
                $table->boolean('is_invoiced')->default(false);
                $table->integer('sort_order')->default(0);
                $table->unsignedBigInteger('created_by')->nullable();
                $table->timestamps();
                
                $table->foreign('project_id')->references('id')->on('projects')->onDelete('cascade');
                $table->index('target_date');
                $table->index('is_completed');
            });
        }

        // ==================== PROJECT DISCUSSIONS ====================
        if (!Schema::hasTable('project_discussions')) {
            Schema::create('project_discussions', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('project_id');
                $table->string('title', 255);
                $table->longText('message')->nullable();
                $table->unsignedBigInteger('created_by')->nullable();
                $table->unsignedBigInteger('last_reply_by')->nullable();
                $table->timestamp('last_reply_at')->nullable();
                $table->boolean('is_pinned')->default(false);
                $table->boolean('is_closed')->default(false);
                $table->timestamps();
                
                $table->foreign('project_id')->references('id')->on('projects')->onDelete('cascade');
                $table->index('created_by');
                $table->index('is_pinned');
            });
        }

        // ==================== PROJECT DISCUSSION REPLIES ====================
        if (!Schema::hasTable('project_discussion_replies')) {
            Schema::create('project_discussion_replies', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('discussion_id');
                $table->longText('message');
                $table->unsignedBigInteger('created_by')->nullable();
                $table->timestamps();
                
                $table->foreign('discussion_id')->references('id')->on('project_discussions')->onDelete('cascade');
                $table->index('created_by');
            });
        }

        // ==================== PROJECT DISCUSSION ATTACHMENTS ====================
        if (!Schema::hasTable('project_discussion_attachments')) {
            Schema::create('project_discussion_attachments', function (Blueprint $table) {
                $table->id();
                $table->unsignedBigInteger('discussion_id');
                $table->string('file_name', 255);
                $table->string('file_path', 500);
                $table->string('file_type', 100)->nullable();
                $table->unsignedBigInteger('file_size')->default(0);
                $table->unsignedBigInteger('uploaded_by')->nullable();
                $table->timestamps();
                
                $table->foreign('discussion_id')->references('id')->on('project_discussions')->onDelete('cascade');
                $table->index('uploaded_by');
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('project_discussion_attachments');
        Schema::dropIfExists('project_discussion_replies');
        Schema::dropIfExists('project_discussions');
        Schema::dropIfExists('project_milestones');
        Schema::dropIfExists('project_team');
        Schema::dropIfExists('projects');
        Schema::dropIfExists('project_statuses');
    }

    /**
     * Seed default project statuses
     */
    protected function seedDefaultStatuses(): void
    {
        $defaultStatuses = [
            ['label' => 'Not Started', 'color' => '#6b7280', 'show_by_default' => true, 'sort_order' => 1],
            ['label' => 'In Progress', 'color' => '#3b82f6', 'show_by_default' => false, 'sort_order' => 2],
            ['label' => 'On Hold', 'color' => '#f59e0b', 'show_by_default' => false, 'sort_order' => 3],
            ['label' => 'Completed', 'color' => '#10b981', 'show_by_default' => false, 'sort_order' => 4],
            ['label' => 'Cancelled', 'color' => '#ef4444', 'show_by_default' => false, 'sort_order' => 5],
        ];

        foreach ($defaultStatuses as $status) {
            // Use 'label' instead of 'name'
            $exists = DB::table('project_statuses')->where('label', $status['label'])->exists();
            
            if (!$exists) {
                DB::table('project_statuses')->insert([
                    'label' => $status['label'],
                    'color' => $status['color'],
                    'show_by_default' => $status['show_by_default'],
                    'sort_order' => $status['sort_order'],
                    'is_active' => true,
                ]);
            }
        }
    }
};
