<?php

namespace Modules\Projects\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Projects\Models\Project;
use Modules\Projects\Models\Milestone;
use Modules\Projects\Models\Discussion;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Exception;

class DashboardController extends AdminController
{
    /**
     * Display the project management dashboard
     */
    public function index()
    {
        try {
            // Get admin/staff ID safely
            $staffId = $this->admin->id ?? 0;
            
            // Overview Stats - with safe checks
            $projectsTotal = 0;
            $projectsActive = 0;
            $projectsPinned = 0;
            $myProjects = 0;
            
            try {
                $projectsTotal = Project::count();
                
                // Active projects - only if status_id column exists
                if (Schema::hasColumn('projects', 'status_id') && Schema::hasTable('project_statuses')) {
                    $projectsActive = Project::active()->count();
                } else {
                    $projectsActive = $projectsTotal;
                }
                
                // Pinned projects
                if (Schema::hasColumn('projects', 'is_pinned')) {
                    $projectsPinned = Project::where('is_pinned', true)->count();
                }
                
                // My projects - only if project_team table exists with proper columns
                if (Schema::hasTable('project_team')) {
                    // Priority: staff_id > admin_id > user_id
                    $teamColumn = null;
                    if (Schema::hasColumn('project_team', 'staff_id')) {
                        $teamColumn = 'staff_id';
                    } elseif (Schema::hasColumn('project_team', 'admin_id')) {
                        $teamColumn = 'admin_id';
                    } elseif (Schema::hasColumn('project_team', 'user_id')) {
                        $teamColumn = 'user_id';
                    }
                    
                    if ($teamColumn) {
                        $myProjects = DB::table('project_team')
                            ->where($teamColumn, $staffId)
                            ->distinct('project_id')
                            ->count('project_id');
                    }
                }
            } catch (\Exception $e) {
                // Silently fail for project stats
            }
            
            // Tasks stats - completely safe
            $tasksTotal = 0;
            $myTasks = 0;
            
            try {
                // Check for module_tasks or tasks table
                $taskTable = null;
                if (Schema::hasTable('module_tasks')) {
                    $taskTable = 'module_tasks';
                } elseif (Schema::hasTable('tasks')) {
                    $taskTable = 'tasks';
                }
                
                if ($taskTable) {
                    $query = DB::table($taskTable);
                    if (Schema::hasColumn($taskTable, 'deleted_at')) {
                        $query->whereNull('deleted_at');
                    }
                    $tasksTotal = $query->count();
                }
            } catch (\Exception $e) {
                // Silently fail
            }
            
            $stats = [
                'projects' => [
                    'total' => $projectsTotal,
                    'active' => $projectsActive,
                    'pinned' => $projectsPinned,
                    'my_projects' => $myProjects,
                ],
                'tasks' => [
                    'total' => $tasksTotal,
                    'my_tasks' => $myTasks,
                    'completed' => 0,
                    'overdue' => 0,
                ],
                'time' => [
                    'total_hours' => 0,
                    'billable_hours' => 0,
                    'my_hours' => 0,
                    'running_timers' => 0,
                ],
                'milestones' => [
                    'total' => Schema::hasTable('project_milestones') ? Milestone::count() : 0,
                    'completed' => Schema::hasTable('project_milestones') ? Milestone::completed()->count() : 0,
                    'incomplete' => Schema::hasTable('project_milestones') ? Milestone::incomplete()->count() : 0,
                    'overdue' => 0,
                ],
                'discussions' => [
                    'total' => Schema::hasTable('project_discussions') ? Discussion::count() : 0,
                    'my_discussions' => Schema::hasTable('project_discussions') ? Discussion::where('created_by', $staffId)->count() : 0,
                ],
            ];

            // Recent Activity - safe
            $recentProjects = collect([]);
            try {
                if (Schema::hasTable('projects')) {
                    $query = Project::query();
                    if (Schema::hasColumn('projects', 'status_id') && Schema::hasTable('project_statuses')) {
                        $query->with('status');
                    }
                    if (Schema::hasColumn('projects', 'created_at')) {
                        $query->orderBy('created_at', 'desc');
                    }
                    $recentProjects = $query->limit(5)->get();
                }
            } catch (\Exception $e) {
                $recentProjects = collect([]);
            }

            $myActiveTasks = collect([]);
            $runningTimers = collect([]);

            return $this->View('projects::dashboard.index', compact(
                'stats',
                'recentProjects',
                'myActiveTasks',
                'runningTimers'
            ));

        } catch (Exception $e) {
            Log::error('DashboardController@index Error: ' . $e->getMessage());
            
            // Return empty stats on error
            $stats = [
                'projects' => ['total' => 0, 'active' => 0, 'pinned' => 0, 'my_projects' => 0],
                'tasks' => ['total' => 0, 'my_tasks' => 0, 'completed' => 0, 'overdue' => 0],
                'time' => ['total_hours' => 0, 'billable_hours' => 0, 'my_hours' => 0, 'running_timers' => 0],
                'milestones' => ['total' => 0, 'completed' => 0, 'incomplete' => 0, 'overdue' => 0],
                'discussions' => ['total' => 0, 'my_discussions' => 0],
            ];
            $recentProjects = collect([]);
            $myActiveTasks = collect([]);
            $runningTimers = collect([]);
            
            return $this->View('projects::dashboard.index', compact(
                'stats', 'recentProjects', 'myActiveTasks', 'runningTimers'
            ));
        }
    }

    /**
     * Get chart data for AJAX
     */
    public function chartData(Request $request): JsonResponse
    {
        try {
            $chartType = $request->input('chart');

            switch ($chartType) {
                case 'project-status':
                    return $this->getProjectStatusChart();
                
                case 'task-priority':
                    return $this->getTaskPriorityChart();
                
                case 'task-status':
                    return $this->getTaskStatusChart();
                
                case 'time-tracking':
                    return $this->getTimeTrackingChart();
                
                case 'project-progress':
                    return $this->getProjectProgressChart();
                
                case 'monthly-time':
                    return $this->getMonthlyTimeChart();
                
                default:
                    return response()->json(['error' => 'Invalid chart type'], 400);
            }

        } catch (Exception $e) {
            Log::error('DashboardController@chartData Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to load chart data'], 500);
        }
    }

    /**
     * Project Status Distribution (Pie Chart)
     */
    private function getProjectStatusChart(): JsonResponse
    {
        $data = Project::select('project_statuses.label', DB::raw('count(*) as count'))
            ->join('project_statuses', 'projects.status_id', '=', 'project_statuses.id')
            ->groupBy('project_statuses.label', 'project_statuses.color')
            ->get();

        return response()->json([
            'labels' => $data->pluck('label'),
            'data' => $data->pluck('count'),
            'colors' => ['#3b82f6', '#10b981', '#f59e0b', '#ef4444', '#8b5cf6'],
        ]);
    }

    /**
     * Task Priority Distribution (Pie Chart)
     */
    private function getTaskPriorityChart(): JsonResponse
    {
        $priorities = ['low', 'medium', 'high', 'critical'];
        $data = [];
        
        foreach ($priorities as $priority) {
            $data[] = Task::where('priority', $priority)->count();
        }

        return response()->json([
            'labels' => ['Low', 'Medium', 'High', 'Critical'],
            'data' => $data,
            'colors' => ['#10b981', '#3b82f6', '#f59e0b', '#ef4444'],
        ]);
    }

    /**
     * Task Status Distribution (Donut Chart)
     */
    private function getTaskStatusChart(): JsonResponse
    {
        $data = Task::select('task_statuses.label', DB::raw('count(*) as count'))
            ->join('task_statuses', 'tasks.status_id', '=', 'task_statuses.id')
            ->groupBy('task_statuses.label')
            ->get();

        return response()->json([
            'labels' => $data->pluck('label'),
            'data' => $data->pluck('count'),
            'colors' => ['#6366f1', '#8b5cf6', '#ec4899', '#f43f5e', '#10b981'],
        ]);
    }

    /**
     * Time Tracking - Billable vs Non-billable (Bar Chart)
     */
    private function getTimeTrackingChart(): JsonResponse
    {
        $billable = round(TimeLog::where('is_billable', true)->sum('duration_minutes') / 60, 1);
        $nonBillable = round(TimeLog::where('is_billable', false)->sum('duration_minutes') / 60, 1);

        return response()->json([
            'labels' => ['Billable', 'Non-Billable'],
            'data' => [$billable, $nonBillable],
            'colors' => ['#10b981', '#94a3b8'],
        ]);
    }

    /**
     * Project Progress - Top 10 Projects (Horizontal Bar Chart)
     */
    private function getProjectProgressChart(): JsonResponse
    {
        $projects = Project::with('tasks')
            ->active()
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($project) {
                $totalTasks = $project->tasks->count();
                $completedTasks = $project->tasks->where('status.is_completed', true)->count();
                $progress = $totalTasks > 0 ? round(($completedTasks / $totalTasks) * 100, 1) : 0;

                return [
                    'name' => $project->title,
                    'progress' => $progress,
                ];
            });

        return response()->json([
            'labels' => $projects->pluck('name'),
            'data' => $projects->pluck('progress'),
            'colors' => array_fill(0, 10, '#3b82f6'),
        ]);
    }

    /**
     * Monthly Time Tracking Trend (Line Chart)
     */
    private function getMonthlyTimeChart(): JsonResponse
    {
        $monthlyData = TimeLog::select(
                DB::raw('DATE_FORMAT(start_time, "%Y-%m") as month'),
                DB::raw('ROUND(SUM(duration_minutes) / 60, 1) as hours')
            )
            ->where('start_time', '>=', now()->subMonths(6))
            ->groupBy('month')
            ->orderBy('month', 'asc')
            ->get();

        return response()->json([
            'labels' => $monthlyData->pluck('month')->map(function($month) {
                return \Carbon\Carbon::parse($month . '-01')->format('M Y');
            }),
            'data' => $monthlyData->pluck('hours'),
            'colors' => ['#3b82f6'],
        ]);
    }

    /**
     * Refresh dashboard stats via AJAX
     */
    public function refreshStats(): JsonResponse
    {
        try {
            $staffId = $this->admin->id ?? 0;
            
            // Projects stats
            $projectStats = [
                'total' => 0,
                'active' => 0,
                'pinned' => 0,
                'my_projects' => 0,
            ];
            
            try {
                $projectStats['total'] = Project::count();
                
                if (Schema::hasColumn('projects', 'status_id') && Schema::hasTable('project_statuses')) {
                    $projectStats['active'] = Project::active()->count();
                }
                
                if (Schema::hasColumn('projects', 'is_pinned')) {
                    $projectStats['pinned'] = Project::where('is_pinned', true)->count();
                }
                
                if (Schema::hasTable('project_team')) {
                    $projectStats['my_projects'] = Project::byTeamMember($staffId)->count();
                }
            } catch (\Exception $e) {
                // Silent fail
            }
            
            // Milestones stats
            $milestoneStats = [
                'total' => 0,
                'completed' => 0,
                'incomplete' => 0,
                'overdue' => 0,
            ];
            
            try {
                if (Schema::hasTable('project_milestones')) {
                    $milestoneStats['total'] = Milestone::count();
                    $milestoneStats['completed'] = Milestone::where('is_completed', true)->count();
                    $milestoneStats['incomplete'] = Milestone::where('is_completed', false)->count();
                    $milestoneStats['overdue'] = Milestone::where('is_completed', false)
                        ->whereNotNull('target_date')
                        ->where('target_date', '<', now())
                        ->count();
                }
            } catch (\Exception $e) {
                // Silent fail
            }
            
            // Discussions stats
            $discussionStats = [
                'total' => 0,
                'my_discussions' => 0,
            ];
            
            try {
                if (Schema::hasTable('project_discussions')) {
                    $discussionStats['total'] = Discussion::count();
                    $discussionStats['my_discussions'] = Discussion::where('created_by', $staffId)->count();
                }
            } catch (\Exception $e) {
                // Silent fail
            }
            
            return response()->json([
                'projects' => $projectStats,
                'milestones' => $milestoneStats,
                'discussions' => $discussionStats,
            ]);

        } catch (Exception $e) {
            Log::error('DashboardController@refreshStats Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to refresh stats'], 500);
        }
    }
}