<?php

namespace Modules\Projects\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Projects\Models\Discussion;
use Modules\Projects\Models\Project;
use Modules\Projects\Models\DiscussionReply;
use Modules\Projects\Models\DiscussionAttachment;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Exception;

class DiscussionController extends AdminController
{
    /**
     * Allowed file types for uploads
     */
    protected $allowedMimes = [
        'jpg', 'jpeg', 'png', 'gif', 'webp', 'svg',
        'pdf', 'doc', 'docx', 'xls', 'xlsx', 'csv', 'ppt', 'pptx',
        'txt', 'rtf', 'zip', 'rar', '7z',
        'html', 'css', 'js', 'json', 'xml'
    ];

    /**
     * Max file size in KB (10MB)
     */
    protected $maxFileSize = 10240;

    public function index()
    {
        $staffId = $this->admin->id ?? 0;
        
        $stats = [
            'total' => 0,
            'portal_visible' => 0,
            'my_discussions' => 0,
        ];
        
        try {
            if (\Schema::hasTable('project_discussions')) {
                $stats['total'] = Discussion::count();
                
                if (\Schema::hasColumn('project_discussions', 'portal_visible')) {
                    $stats['portal_visible'] = Discussion::where('portal_visible', true)->count();
                }
                
                if (\Schema::hasColumn('project_discussions', 'created_by')) {
                    $stats['my_discussions'] = Discussion::where('created_by', $staffId)->count();
                }
            }
        } catch (\Exception $e) {
            // Silently fail
        }

        return $this->View('projects::discussions.index', compact('stats'));
    }

    public function dataTable(Request $request): JsonResponse
    {
        try {
            if (!\Schema::hasTable('project_discussions')) {
                return response()->json([
                    'data' => [],
                    'total' => 0,
                    'current_page' => 1,
                    'last_page' => 1,
                ]);
            }
            
            $query = Discussion::query();
            
            // Only load relations if tables exist
            $with = [];
            if (\Schema::hasTable('projects')) {
                $with[] = 'project';
            }
            // Check for staffs or admins table
            if (\Schema::hasTable('staffs') || \Schema::hasTable('admins')) {
                $with[] = 'creator';
            }
            
            if (!empty($with)) {
                $query->with($with);
            }

            if ($search = $request->input('search')) {
                $query->where(function($q) use ($search) {
                    $q->where('title', 'LIKE', "%{$search}%");
                    if (\Schema::hasColumn('project_discussions', 'message')) {
                        $q->orWhere('message', 'LIKE', "%{$search}%");
                    }
                });
            }

            if ($filters = $request->input('filters')) {
                $decoded = is_array($filters) ? $filters : json_decode($filters, true);
                foreach ($decoded ?? [] as $key => $value) {
                    if ($value !== '' && $value !== null) {
                        if ($key === 'portal_visible' && \Schema::hasColumn('project_discussions', 'portal_visible')) {
                            $query->where('portal_visible', $value);
                        } elseif ($key === 'project_id') {
                            $query->where('project_id', $value);
                        }
                    }
                }
            }

            $sortColumn = $request->input('sort', 'created_at');
            if (\Schema::hasColumn('project_discussions', $sortColumn)) {
                $query->orderBy($sortColumn, $request->input('dir', 'desc'));
            }

            $data = $query->paginate($request->input('per_page', 15));

            $items = collect($data->items())->map(function ($item) {
                $item->_show_url = route('admin.discussions.show', $item->id);
                
                // Project display - handle title/name variants
                $item->project_display = '-';
                if ($item->project) {
                    $item->project_display = $item->project->title ?? $item->project->name ?? 'Project #' . $item->project->id;
                }
                
                // Creator display
                $item->creator_display = $item->creator ? ($item->creator->name ?? $item->creator->firstname ?? 'Unknown') : 'Unknown';
                
                $item->created_at_display = $item->created_at 
                    ? $item->created_at->format('M d, Y H:i')
                    : '-';
                    
                $item->portal_display = isset($item->portal_visible) ? ($item->portal_visible ? 'Visible' : 'Hidden') : '-';
                
                // Safe counts
                $item->replies_count = 0;
                $item->attachments_count = 0;
                try {
                    if (\Schema::hasTable('project_discussion_replies')) {
                        $item->replies_count = $item->replies()->count();
                    }
                    if (\Schema::hasTable('project_discussion_attachments')) {
                        $item->attachments_count = $item->attachments()->count();
                    }
                } catch (\Exception $e) {
                    // Silently fail
                }
                
                return $item;
            });

            return response()->json([
                'data' => $items,
                'total' => $data->total(),
                'current_page' => $data->currentPage(),
                'last_page' => $data->lastPage(),
            ]);
            
        } catch (\Exception $e) {
            Log::error('DiscussionController@dataTable Error: ' . $e->getMessage());
            return response()->json([
                'data' => [],
                'total' => 0,
                'current_page' => 1,
                'last_page' => 1,
                'error' => 'Failed to load discussions'
            ], 500);
        }
    }

    public function stats(): JsonResponse
    {
        try {
            $staffId = $this->admin->id ?? 0;
            
            $stats = [
                'total' => 0,
                'portal_visible' => 0,
                'my_discussions' => 0,
            ];
            
            if (\Schema::hasTable('project_discussions')) {
                $stats['total'] = Discussion::count();
                
                if (\Schema::hasColumn('project_discussions', 'portal_visible')) {
                    $stats['portal_visible'] = Discussion::where('portal_visible', true)->count();
                }
                
                if (\Schema::hasColumn('project_discussions', 'created_by')) {
                    $stats['my_discussions'] = Discussion::where('created_by', $staffId)->count();
                }
            }

            return response()->json($stats);

        } catch (Exception $e) {
            Log::error('DiscussionController@stats Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to load stats'
            ], 500);
        }
    }

    public function create()
    {
        $projects = collect([]);
        
        try {
            if (\Schema::hasTable('projects')) {
                $query = Project::query();
                
                // Order by title or name
                if (\Schema::hasColumn('projects', 'title')) {
                    $query->orderBy('title');
                } elseif (\Schema::hasColumn('projects', 'name')) {
                    $query->orderBy('name');
                }
                
                $projects = $query->get();
            }
        } catch (\Exception $e) {
            // Return empty collection
        }

        return $this->View('projects::discussions.create', compact('projects'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'title' => 'required|string|max:191',
            'message' => 'required|string',
            'portal_visible' => 'boolean',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:' . $this->maxFileSize . '|mimes:' . implode(',', $this->allowedMimes),
        ]);

        $validated['created_by'] = $this->admin->id;

        // Remove attachments from validated data before creating discussion
        $attachmentFiles = $request->file('attachments', []);
        unset($validated['attachments']);

        $discussion = Discussion::create($validated);

        // Handle file uploads
        if (!empty($attachmentFiles)) {
            $this->handleFileUploads($discussion, $attachmentFiles);
        }

        return redirect()->route('admin.discussions.show', $discussion->id)
            ->with('success', 'Discussion created successfully!');
    }

    public function show($id)
    {
        $discussion = Discussion::with(['project', 'creator', 'replies.creator', 'attachments'])
            ->findOrFail($id);

        return $this->View('projects::discussions.show', compact('discussion'));
    }

    public function destroy($id)
    {
        $discussion = Discussion::findOrFail($id);
        
        // Attachments will be deleted via model events
        $discussion->attachments()->delete();
        $discussion->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'Discussion deleted successfully!']);
        }

        return redirect()->route('admin.discussions.index')
            ->with('success', 'Discussion deleted successfully!');
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        $ids = $request->input('ids', []);
        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No discussions selected'], 400);
        }

        // Delete attachments first
        DiscussionAttachment::whereIn('discussion_id', $ids)->delete();
        
        $deleted = Discussion::whereIn('id', $ids)->delete();
        return response()->json(['success' => true, 'message' => "{$deleted} discussions deleted!"]);
    }

    public function addReply(Request $request, $id)
    {
        // Check if replies table exists
        if (!\Schema::hasTable('project_discussion_replies')) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Replies feature not available. Please run migrations.'
                ], 500);
            }
            return back()->with('error', 'Replies feature not available. Please run migrations.');
        }
        
        $discussion = Discussion::findOrFail($id);
        
        $validated = $request->validate([
            'message' => 'required|string',
        ]);

        $staffId = $this->admin->id ?? 0;

        $reply = DiscussionReply::create([
            'discussion_id' => $discussion->id,
            'message' => $validated['message'],
            'created_by' => $staffId,
        ]);

        if ($request->ajax()) {
            // Try to load creator relationship safely
            try {
                $reply->load('creator');
            } catch (\Exception $e) {
                // Ignore if creator can't be loaded
            }
            
            return response()->json([
                'success' => true,
                'message' => 'Reply added!',
                'reply' => $reply
            ]);
        }

        return back()->with('success', 'Reply added successfully!');
    }

    public function deleteReply($id, $replyId)
    {
        $discussion = Discussion::findOrFail($id);
        $reply = DiscussionReply::where('discussion_id', $discussion->id)
            ->findOrFail($replyId);
            
        $reply->delete();

        return response()->json([
            'success' => true,
            'message' => 'Reply deleted!'
        ]);
    }

    // ==================== ATTACHMENT METHODS ====================

    /**
     * Handle file uploads for discussion
     */
    protected function handleFileUploads(Discussion $discussion, array $files): void
    {
        foreach ($files as $file) {
            if (!$file || !$file->isValid()) {
                continue;
            }

            try {
                // Generate unique filename
                $originalName = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension();
                $fileName = pathinfo($originalName, PATHINFO_FILENAME);
                $uniqueName = $fileName . '_' . time() . '_' . uniqid() . '.' . $extension;

                // Store file
                $path = $file->storeAs(
                    'discussions/' . $discussion->id,
                    $uniqueName,
                    'public'
                );

                // Create attachment record
                DiscussionAttachment::create([
                    'discussion_id' => $discussion->id,
                    'file_name' => $originalName,
                    'file_path' => $path,
                    'file_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                    'uploaded_by' => $this->admin->id,
                ]);

            } catch (Exception $e) {
                Log::error('Discussion file upload error: ' . $e->getMessage());
            }
        }
    }

    /**
     * Download attachment
     */
    public function downloadAttachment($attachmentId)
    {
        $attachment = DiscussionAttachment::findOrFail($attachmentId);

        if (!Storage::disk('public')->exists($attachment->file_path)) {
            abort(404, 'File not found');
        }

        return Storage::disk('public')->download(
            $attachment->file_path,
            $attachment->file_name
        );
    }

    /**
     * Preview attachment (for images)
     */
    public function previewAttachment($attachmentId)
    {
        $attachment = DiscussionAttachment::findOrFail($attachmentId);

        if (!Storage::disk('public')->exists($attachment->file_path)) {
            abort(404, 'File not found');
        }

        $file = Storage::disk('public')->get($attachment->file_path);
        $mimeType = $attachment->file_type ?? 'application/octet-stream';

        return response($file, 200)
            ->header('Content-Type', $mimeType)
            ->header('Content-Disposition', 'inline; filename="' . $attachment->file_name . '"');
    }

    /**
     * Delete single attachment
     */
    public function deleteAttachment($attachmentId): JsonResponse
    {
        try {
            $attachment = DiscussionAttachment::findOrFail($attachmentId);
            $attachment->delete(); // This will also delete the file via model boot method

            return response()->json([
                'success' => true,
                'message' => 'Attachment deleted successfully!'
            ]);

        } catch (Exception $e) {
            Log::error('Delete attachment error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete attachment'
            ], 500);
        }
    }

    /**
     * Upload additional attachments to existing discussion
     */
    public function uploadAttachments(Request $request, $id): JsonResponse
    {
        $discussion = Discussion::findOrFail($id);

        $request->validate([
            'attachments' => 'required|array',
            'attachments.*' => 'file|max:' . $this->maxFileSize . '|mimes:' . implode(',', $this->allowedMimes),
        ]);

        $files = $request->file('attachments', []);
        $uploaded = [];

        foreach ($files as $file) {
            if (!$file || !$file->isValid()) {
                continue;
            }

            try {
                $originalName = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension();
                $fileName = pathinfo($originalName, PATHINFO_FILENAME);
                $uniqueName = $fileName . '_' . time() . '_' . uniqid() . '.' . $extension;

                $path = $file->storeAs(
                    'discussions/' . $discussion->id,
                    $uniqueName,
                    'public'
                );

                $attachment = DiscussionAttachment::create([
                    'discussion_id' => $discussion->id,
                    'file_name' => $originalName,
                    'file_path' => $path,
                    'file_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                    'uploaded_by' => $this->admin->id,
                ]);

                $uploaded[] = [
                    'id' => $attachment->id,
                    'file_name' => $attachment->file_name,
                    'formatted_size' => $attachment->formatted_size,
                    'icon' => $attachment->icon,
                    'is_image' => $attachment->is_image,
                    'download_url' => $attachment->download_url,
                    'preview_url' => $attachment->preview_url,
                ];

            } catch (Exception $e) {
                Log::error('Additional attachment upload error: ' . $e->getMessage());
            }
        }

        return response()->json([
            'success' => true,
            'message' => count($uploaded) . ' file(s) uploaded successfully!',
            'attachments' => $uploaded,
        ]);
    }
}