<?php

namespace Modules\Projects\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Projects\Models\Milestone;
use Modules\Projects\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Exception;

class MilestoneController extends AdminController
{
    public function index()
    {
        $stats = [
            'total' => 0,
            'completed' => 0,
            'incomplete' => 0,
            'overdue' => 0,
        ];
        
        try {
            if (\Schema::hasTable('project_milestones')) {
                $stats['total'] = Milestone::count();
                
                if (\Schema::hasColumn('project_milestones', 'is_completed')) {
                    $stats['completed'] = Milestone::where('is_completed', true)->count();
                    $stats['incomplete'] = Milestone::where('is_completed', false)->count();
                    
                    if (\Schema::hasColumn('project_milestones', 'target_date')) {
                        $stats['overdue'] = Milestone::where('is_completed', false)
                            ->where('target_date', '<', now())
                            ->count();
                    }
                }
            }
        } catch (\Exception $e) {
            // Silently fail
        }

        return $this->View('projects::milestones.index', compact('stats'));
    }

public function dataTable(Request $request): JsonResponse
{
    try {
        if (!\Schema::hasTable('project_milestones')) {
            return response()->json([
                'data' => [],
                'total' => 0,
                'current_page' => 1,
                'last_page' => 1,
            ]);
        }
        
        $query = Milestone::query();
        
        // Only load relations if tables exist
        $with = [];
        if (\Schema::hasTable('projects')) {
            $with[] = 'project';
        }
        if (\Schema::hasTable('staffs') || \Schema::hasTable('admins')) {
            $with[] = 'creator';
        }
        
        if (!empty($with)) {
            $query->with($with);
        }

        // Search filter
        if ($search = $request->input('search')) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        // Handle filter_type from quick filters (All, Completed, In Progress, Overdue)
        $filterType = $request->input('filter_type', 'all');
        
        if (\Schema::hasColumn('project_milestones', 'is_completed')) {
            if ($filterType === 'completed') {
                $query->where('is_completed', true);
            } elseif ($filterType === 'incomplete') {
                $query->where('is_completed', false);
            } elseif ($filterType === 'overdue') {
                $query->where('is_completed', false);
                if (\Schema::hasColumn('project_milestones', 'target_date')) {
                    $query->where('target_date', '<', now());
                }
            }
        }

        // Handle advanced filters from DataTable filter dropdowns
        if ($filters = $request->input('filters')) {
            $decoded = is_array($filters) ? $filters : json_decode($filters, true);
            foreach ($decoded ?? [] as $key => $value) {
                if ($value !== '' && $value !== null) {
                    if ($key === 'is_completed' && \Schema::hasColumn('project_milestones', 'is_completed')) {
                        $query->where('is_completed', $value);
                    } elseif ($key === 'project_id') {
                        $query->where('project_id', $value);
                    }
                }
            }
        }

        // Sorting
        $sortColumn = $request->input('sort', 'sort_order');
        if (!\Schema::hasColumn('project_milestones', $sortColumn)) {
            $sortColumn = 'created_at';
            if (!\Schema::hasColumn('project_milestones', 'created_at')) {
                $sortColumn = 'id';
            }
        }
        $query->orderBy($sortColumn, $request->input('dir', 'asc'));

        // Pagination
        $data = $query->paginate($request->input('per_page', 15));

        // Map items with display fields and action URLs
        $items = collect($data->items())->map(function ($item) {
            // Add action URLs
            $item->_edit_url = route('admin.milestones.edit', $item->id);
            $item->_show_url = route('admin.milestones.show', $item->id);
            
            // Project display - handle title/name variants
            $item->project_display = '-';
            if ($item->project) {
                $item->project_display = $item->project->title ?? $item->project->name ?? 'Project #' . $item->project->id;
            }
            
            // Format target date
            $item->target_date_display = '-';
            if (isset($item->target_date) && $item->target_date) {
                try {
                    $item->target_date_display = \Carbon\Carbon::parse($item->target_date)->format('M d, Y');
                } catch (\Exception $e) {
                    $item->target_date_display = '-';
                }
            }
            
            // Status display
            $item->status_display = isset($item->is_completed) ? ($item->is_completed ? 'Completed' : 'Incomplete') : '-';
            
            // Check if overdue
            $item->is_overdue = false;
            if (isset($item->is_completed) && !$item->is_completed && isset($item->target_date) && $item->target_date) {
                try {
                    $item->is_overdue = \Carbon\Carbon::parse($item->target_date)->isPast();
                } catch (\Exception $e) {
                    // Ignore
                }
            }
            
            // Days remaining/overdue
            $item->days_status = '-';
            if (isset($item->target_date) && $item->target_date && isset($item->is_completed) && !$item->is_completed) {
                try {
                    $targetDate = \Carbon\Carbon::parse($item->target_date);
                    $now = \Carbon\Carbon::now();
                    
                    if ($targetDate->isPast()) {
                        $item->days_status = 'Overdue by ' . $now->diffInDays($targetDate) . ' days';
                    } else {
                        $item->days_status = $targetDate->diffInDays($now) . ' days remaining';
                    }
                } catch (\Exception $e) {
                    $item->days_status = '-';
                }
            }
            
            return $item;
        });

        return response()->json([
            'data' => $items,
            'total' => $data->total(),
            'current_page' => $data->currentPage(),
            'last_page' => $data->lastPage(),
        ]);
        
    } catch (\Exception $e) {
        Log::error('MilestoneController@dataTable Error: ' . $e->getMessage());
        return response()->json([
            'data' => [],
            'total' => 0,
            'current_page' => 1,
            'last_page' => 1,
            'error' => 'Failed to load milestones'
        ], 500);
    }
}

/**
 * Get milestone statistics for AJAX refresh
 */
public function stats(): JsonResponse
{
    try {
        $stats = [
            'total' => 0,
            'completed' => 0,
            'incomplete' => 0,
            'overdue' => 0,
        ];
        
        if (\Schema::hasTable('project_milestones')) {
            $stats['total'] = Milestone::count();
            
            if (\Schema::hasColumn('project_milestones', 'is_completed')) {
                $stats['completed'] = Milestone::where('is_completed', true)->count();
                $stats['incomplete'] = Milestone::where('is_completed', false)->count();
                
                if (\Schema::hasColumn('project_milestones', 'target_date')) {
                    $stats['overdue'] = Milestone::where('is_completed', false)
                        ->where('target_date', '<', now())
                        ->count();
                }
            }
        }

        return response()->json($stats);

    } catch (\Exception $e) {
        \Log::error('MilestoneController@stats Error: ' . $e->getMessage());
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to load stats'
        ], 500);
    }
}

    public function create()
    {
        $projects = collect([]);
        
        try {
            if (\Schema::hasTable('projects')) {
                $query = Project::query();
                
                // Order by title or name
                if (\Schema::hasColumn('projects', 'title')) {
                    $query->orderBy('title');
                } elseif (\Schema::hasColumn('projects', 'name')) {
                    $query->orderBy('name');
                }
                
                $projects = $query->get();
            }
        } catch (\Exception $e) {
            // Return empty collection
        }

        return $this->View('projects::milestones.create', compact('projects'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'name' => 'required|string|max:191',
            'description' => 'nullable|string',
            'target_date' => 'nullable|date',
            'portal_visible' => 'boolean',
            'sort_order' => 'nullable|integer',
        ]);

        $staffId = $this->admin->id ?? 0;
        $validated['created_by'] = $staffId;
        
        if (!isset($validated['sort_order'])) {
            $maxOrder = Milestone::where('project_id', $validated['project_id'])->max('sort_order') ?? 0;
            $validated['sort_order'] = $maxOrder + 1;
        }

        $milestone = Milestone::create($validated);

        return redirect()->route('admin.milestones.index')
            ->with('success', 'Milestone created successfully!');
    }

    /**
 * Display the specified milestone details
 */
    public function show($id)
    {
        $with = [];
        if (\Schema::hasTable('projects')) {
            $with[] = 'project';
        }
        if (\Schema::hasTable('staffs') || \Schema::hasTable('admins')) {
            $with[] = 'creator';
        }
        
        $milestone = Milestone::with($with)->findOrFail($id);

        return $this->View('projects::milestones.show', compact('milestone'));
    }

    public function edit($id)
    {
        $milestone = Milestone::findOrFail($id);
        
        $projects = collect([]);
        try {
            if (\Schema::hasTable('projects')) {
                $query = Project::query();
                if (\Schema::hasColumn('projects', 'title')) {
                    $query->orderBy('title');
                } elseif (\Schema::hasColumn('projects', 'name')) {
                    $query->orderBy('name');
                }
                $projects = $query->get();
            }
        } catch (\Exception $e) {
            // Return empty collection
        }

        return $this->View('projects::milestones.edit', compact('milestone', 'projects'));
    }

    public function update(Request $request, $id)
    {
        $milestone = Milestone::findOrFail($id);

        $validated = $request->validate([
            'project_id' => 'required|exists:projects,id',
            'name' => 'required|string|max:191',
            'description' => 'nullable|string',
            'target_date' => 'nullable|date',
            'portal_visible' => 'boolean',
            'sort_order' => 'nullable|integer',
            'is_completed' => 'boolean',
        ]);

        if (isset($validated['is_completed']) && $validated['is_completed'] && !$milestone->is_completed) {
            $validated['completed_at'] = now();
        } elseif (isset($validated['is_completed']) && !$validated['is_completed']) {
            $validated['completed_at'] = null;
        }

        $milestone->update($validated);

        return redirect()->route('admin.milestones.index')
            ->with('success', 'Milestone updated successfully!');
    }

    public function destroy($id)
    {
        $milestone = Milestone::findOrFail($id);
        $milestone->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'Milestone deleted successfully!']);
        }

        return redirect()->route('admin.milestones.index')
            ->with('success', 'Milestone deleted successfully!');
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        $ids = $request->input('ids', []);
        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No milestones selected'], 400);
        }

        $deleted = Milestone::whereIn('id', $ids)->delete();
        return response()->json(['success' => true, 'message' => "{$deleted} milestones deleted!"]);
    }
}
