<?php

namespace Modules\Projects\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Projects\Models\Project;
use Modules\Projects\Models\ProjectStatus;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class ProjectController extends AdminController
{
    public function index()
    {
        $stats = [
            'total' => Project::count(),
            'active' => \Schema::hasColumn('projects', 'status_id') ? Project::active()->count() : Project::count(),
            'pinned' => \Schema::hasColumn('projects', 'is_pinned') ? Project::pinned()->count() : 0,
            'my_projects' => \Schema::hasTable('project_team') ? Project::byTeamMember($this->admin->id)->count() : 0,
        ];

        return $this->View('projects::projects.index', compact('stats'));
    }

public function dataTable(Request $request): JsonResponse
{
    try {
        // Detect column names
        $hasTitle = \Schema::hasColumn('projects', 'title');
        $hasName = \Schema::hasColumn('projects', 'name');
        $hasIsPinned = \Schema::hasColumn('projects', 'is_pinned');
        $hasStatusId = \Schema::hasColumn('projects', 'status_id');
        $hasStatus = \Schema::hasColumn('projects', 'status');
        $hasAccountId = \Schema::hasColumn('projects', 'account_id');
        $hasClientId = \Schema::hasColumn('projects', 'client_id');
        
        $nameColumn = $hasTitle ? 'title' : ($hasName ? 'name' : 'id');
        $hasCustomerId = \Schema::hasColumn('projects', 'customer_id');
        
        // Query PROJECTS - only include relations that exist
        $relations = [];
        if ($hasStatusId && \Schema::hasTable('project_statuses')) {
            $relations[] = 'status';
        }
        // Load account/client relation if any customer FK column exists
        if (($hasAccountId || $hasClientId || $hasCustomerId) && \Schema::hasTable('customers')) {
            $relations[] = 'account';
        }
        if (\Schema::hasColumn('projects', 'currency_id') && \Schema::hasTable('currencies')) {
            $relations[] = 'currency';
        }
        
        $query = Project::query();
        
        // Only load relations if they exist
        if (!empty($relations)) {
            $query->with($relations);
        }

        // Search filter
        if ($search = $request->input('search')) {
            $query->where(function($q) use ($search, $nameColumn) {
                $q->where($nameColumn, 'LIKE', "%{$search}%");
                if (\Schema::hasColumn('projects', 'scope_notes')) {
                    $q->orWhere('scope_notes', 'LIKE', "%{$search}%");
                }
                if (\Schema::hasColumn('projects', 'description')) {
                    $q->orWhere('description', 'LIKE', "%{$search}%");
                }
            });
        }

        // Handle filter_type from quick filters (All, Active, Pinned, My Projects)
        $filterType = $request->input('filter_type', 'all');
        
        if ($filterType === 'active') {
            if ($hasStatusId) {
                $query->active();
            } elseif ($hasStatus) {
                $query->where('status', 'active');
            }
        } elseif ($filterType === 'pinned' && $hasIsPinned) {
            $query->pinned();
        } elseif ($filterType === 'my' && \Schema::hasTable('project_team')) {
            $query->byTeamMember($this->admin->id);
        }
        // 'all' - no additional filter

        // Advanced filters
        if ($filters = $request->input('filters')) {
            $decoded = is_array($filters) ? $filters : json_decode($filters, true);
            foreach ($decoded ?? [] as $key => $value) {
                if ($value !== '' && $value !== null) {
                    if ($key === 'status_id' && $hasStatusId) {
                        $query->where('status_id', $value);
                    } elseif ($key === 'status' && $hasStatus) {
                        $query->where('status', $value);
                    } elseif ($key === 'account_id' && $hasAccountId) {
                        $query->where('account_id', $value);
                    } elseif ($key === 'client_id' && $hasClientId) {
                        $query->where('client_id', $value);
                    } elseif ($key === 'billing_method' && \Schema::hasColumn('projects', 'billing_method')) {
                        $query->where('billing_method', $value);
                    }
                }
            }
        }

        // Sorting - Pinned projects go first (if column exists)
        $sortColumn = $request->input('sort', 'created_at');
        $sortDirection = $request->input('dir', 'desc');
        
        // Validate sort column exists
        if (!\Schema::hasColumn('projects', $sortColumn)) {
            $sortColumn = 'id';
        }
        
        if ($hasIsPinned) {
            $query->orderBy('is_pinned', 'desc'); // Pinned first
        }
        $query->orderBy($sortColumn, $sortDirection);

        // Pagination
        $data = $query->paginate($request->input('per_page', 15));

        // Map items with display fields and action URLs
        $items = collect($data->items())->map(function ($item) use ($nameColumn, $hasStatusId, $hasStatus, $hasIsPinned) {
            // Add action URLs
            $item->_show_url = route('admin.projects.show', $item->id);
            $item->_edit_url = route('admin.projects.edit', $item->id);
            $item->_finance_url = route('admin.projects.finance', $item->id);
            
            // Add display fields for frontend rendering
            $item->_title = $item->{$nameColumn} ?? 'Project #' . $item->id;
            $item->title = $item->_title; // For JS compatibility
            $item->is_pinned = $hasIsPinned ? (bool)($item->is_pinned ?? false) : false;
            
            // Status display
            if ($hasStatusId && $item->relationLoaded('status') && $item->status) {
                $item->_status_label = $item->status->label;
                $item->_status_color = $item->status->color;
            } elseif ($hasStatus && isset($item->status)) {
                // Handle both string and integer status values
                $statusLabels = [
                    'planned' => 'Planned', '1' => 'Planned', 1 => 'Planned',
                    'active' => 'Active', '2' => 'Active', 2 => 'Active',
                    'on_hold' => 'On Hold', '3' => 'On Hold', 3 => 'On Hold',
                    'completed' => 'Completed', '4' => 'Completed', 4 => 'Completed',
                    'cancelled' => 'Cancelled', '5' => 'Cancelled', 5 => 'Cancelled',
                ];
                $statusColors = [
                    'planned' => '#3b82f6', '1' => '#3b82f6', 1 => '#3b82f6',
                    'active' => '#10b981', '2' => '#10b981', 2 => '#10b981',
                    'on_hold' => '#f59e0b', '3' => '#f59e0b', 3 => '#f59e0b',
                    'completed' => '#22c55e', '4' => '#22c55e', 4 => '#22c55e',
                    'cancelled' => '#ef4444', '5' => '#ef4444', 5 => '#ef4444',
                ];
                $item->_status_label = $statusLabels[$item->status] ?? ucfirst(str_replace('_', ' ', $item->status));
                $item->_status_color = $statusColors[$item->status] ?? '#9ca3af';
            } else {
                $item->_status_label = 'Unknown';
                $item->_status_color = '#9ca3af';
            }
            
            $item->_is_pinned = $item->is_pinned;
            
            // Account/Client display
            if ($item->relationLoaded('account') && $item->account) {
                $item->account_display = $item->account->company ?? $item->account->name ?? '-';
            } else {
                $item->account_display = '-';
            }
            
            $item->status_display = $item->_status_label;
            $item->status_color = $item->_status_color;
            
            // Billing method display
            $billingMethods = [
                'fixed' => 'Fixed Amount',
                'project_hours' => 'Project Hours',
                'task_hours' => 'Task Hours',
            ];
            $item->billing_method_display = isset($item->billing_method) ? ($billingMethods[$item->billing_method] ?? ucfirst($item->billing_method ?? '')) : '-';
            
            return $item;
        });

        return response()->json([
            'data' => $items,
            'total' => $data->total(),
            'current_page' => $data->currentPage(),
            'last_page' => $data->lastPage(),
        ]);

    } catch (Exception $e) {
        Log::error('ProjectController@dataTable Error: ' . $e->getMessage(), [
            'trace' => $e->getTraceAsString()
        ]);

        return response()->json([
            'success' => false,
            'message' => 'Failed to load projects data',
            'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
        ], 500);
    }
}



    /**
 * Get project statistics for AJAX updates
 */
public function stats(): JsonResponse
{
    return response()->json([
        'total' => Project::count(),
        'active' => \Schema::hasColumn('projects', 'status_id') ? Project::active()->count() : Project::count(),
        'pinned' => \Schema::hasColumn('projects', 'is_pinned') ? Project::pinned()->count() : 0,
        'my_projects' => \Schema::hasTable('project_team') ? Project::byTeamMember($this->admin->id)->count() : 0,
    ]);
}

    public function create()
    {
        // Get statuses - from table or from column enum
        $statuses = collect([]);
        $statusOptions = []; // For simple status column
        
        if (\Schema::hasTable('project_statuses')) {
            $statuses = ProjectStatus::where('is_active', true)->orderBy('sort_order')->get();
        } elseif (\Schema::hasColumn('projects', 'status')) {
            // Try to get enum values from column
            try {
                $columnType = DB::selectOne("SHOW COLUMNS FROM projects WHERE Field = 'status'")->Type ?? '';
                
                if (strpos($columnType, 'enum') !== false) {
                    preg_match("/^enum\(\'(.*)\'\)$/", $columnType, $matches);
                    if (isset($matches[1])) {
                        $enumValues = explode("','", $matches[1]);
                        foreach ($enumValues as $value) {
                            $statusOptions[] = [
                                'value' => $value,
                                'label' => ucfirst(str_replace('_', ' ', $value)),
                            ];
                        }
                    }
                }
            } catch (\Exception $e) {
                // If can't get enum values, use defaults
            }
            
            // If no enum values found, use defaults
            if (empty($statusOptions)) {
                $statusOptions = [
                    ['value' => 'planned', 'label' => 'Planned'],
                    ['value' => 'active', 'label' => 'Active'],
                    ['value' => 'on_hold', 'label' => 'On Hold'],
                    ['value' => 'completed', 'label' => 'Completed'],
                    ['value' => 'cancelled', 'label' => 'Cancelled'],
                ];
            }
        }
        
        // Fetch customers
        $customers = collect([]);
        if (\Schema::hasTable('customers')) {
            $query = \DB::table('customers');
            if (\Schema::hasColumn('customers', 'active')) {
                $query->where('active', true);
            }
            if (\Schema::hasColumn('customers', 'name')) {
                $query->orderBy('name');
            } elseif (\Schema::hasColumn('customers', 'company')) {
                $query->orderBy('company');
            }
            $customers = $query->get();
        }
        
        // Check if Currency table exists
        $currencies = collect([]);
        if (\Schema::hasTable('currencies')) {
            $currencies = \DB::table('currencies')->where('is_active', true)->get();
        }
        
        // Get staff/admins for team assignment - check staffs first, then admins
        $admins = collect([]);
        if (\Schema::hasTable('staffs')) {
            $query = \DB::table('staffs');
            
            // Apply active filter
            if (\Schema::hasColumn('staffs', 'active')) {
                $query->where('active', 1);
            } elseif (\Schema::hasColumn('staffs', 'is_active')) {
                $query->where('is_active', 1);
            }
            
            // Order by available column
            if (\Schema::hasColumn('staffs', 'firstname')) {
                $query->orderBy('firstname');
            } elseif (\Schema::hasColumn('staffs', 'name')) {
                $query->orderBy('name');
            } else {
                $query->orderBy('id');
            }
            
            $admins = $query->get()->map(function($staff) {
                // Build display name from available columns
                if (isset($staff->firstname) || isset($staff->lastname)) {
                    $staff->name = trim(($staff->firstname ?? '') . ' ' . ($staff->lastname ?? ''));
                }
                if (empty($staff->name)) {
                    $staff->name = $staff->first_name ?? 'Staff #' . $staff->id;
                }
                return $staff;
            });
        } elseif (class_exists('App\Models\Admin') && \Schema::hasTable('admins')) {
            $admins = \App\Models\Admin::where('is_active', true)->orderBy('name')->get();
        }

        return $this->View('projects::projects.create', compact('statuses', 'statusOptions', 'customers', 'currencies', 'admins'));
    }

    public function store(Request $request)
    {
        // Detect which columns exist in projects table
        $hasTitle = \Schema::hasColumn('projects', 'title');
        $hasName = \Schema::hasColumn('projects', 'name');
        $hasAccountId = \Schema::hasColumn('projects', 'account_id');
        $hasClientId = \Schema::hasColumn('projects', 'client_id');
        $hasBillingMethod = \Schema::hasColumn('projects', 'billing_method');
        $hasPlannedDates = \Schema::hasColumn('projects', 'planned_start_date');
        $hasScopeNotes = \Schema::hasColumn('projects', 'scope_notes');
        $hasDescription = \Schema::hasColumn('projects', 'description');
        $hasInternalNotes = \Schema::hasColumn('projects', 'internal_notes');
        $hasIsBillable = \Schema::hasColumn('projects', 'is_billable');
        $hasCustomerId = \Schema::hasColumn('projects', 'customer_id');
        
        // Build validation rules dynamically based on existing columns
        // Determine team table for validation
        $teamTable = \Schema::hasTable('staffs') ? 'staffs' : 'admins';
        
        $rules = [
            'title' => 'required|string|max:191', // Always validate title from form
            'team' => 'array',
            'team.*' => "exists:{$teamTable},id",
        ];
        
        // Customer field - always validate account_id from form (maps to account_id, client_id, or customer_id)
        if ($hasAccountId || $hasClientId || $hasCustomerId) {
            $rules['account_id'] = 'nullable|exists:customers,id';
        }
        
        // Optional fields - only validate if column exists
        if ($hasBillingMethod) {
            $rules['billing_method'] = 'nullable|in:fixed,project_hours,task_hours';
            $rules['fixed_amount'] = 'nullable|numeric|min:0';
            $rules['hourly_rate'] = 'nullable|numeric|min:0';
        }
        
        if (\Schema::hasColumn('projects', 'currency_id') && \Schema::hasTable('currencies')) {
            $rules['currency_id'] = 'nullable|exists:currencies,id';
        }
        
        if ($hasPlannedDates) {
            $rules['planned_start_date'] = 'nullable|date';
            $rules['planned_end_date'] = 'nullable|date|after_or_equal:planned_start_date';
        } elseif (\Schema::hasColumn('projects', 'start_date')) {
            $rules['start_date'] = 'nullable|date';
            $rules['end_date'] = 'nullable|date';
        }
        
        if ($hasScopeNotes) {
            $rules['scope_notes'] = 'nullable|string';
        } elseif ($hasDescription) {
            $rules['description'] = 'nullable|string';
        }
        
        if ($hasInternalNotes) {
            $rules['internal_notes'] = 'nullable|string';
        }
        
        if ($hasIsBillable) {
            $rules['is_billable'] = 'boolean';
        }
        
        // Handle status - either status_id (foreign key) or status (enum/string/int)
        if (\Schema::hasTable('project_statuses') && \Schema::hasColumn('projects', 'status_id')) {
            $rules['status_id'] = 'nullable|exists:project_statuses,id';
        } elseif (\Schema::hasColumn('projects', 'status')) {
            $rules['status'] = 'nullable|string|max:50';
        }
        
        $validated = $request->validate($rules);

        DB::beginTransaction();
        try {
            // Map form fields to actual column names
            $data = [];
            
            // Title/Name
            if ($hasTitle && isset($validated['title'])) {
                $data['title'] = $validated['title'];
            } elseif ($hasName && isset($validated['title'])) {
                $data['name'] = $validated['title']; // Map title input to name column
            } elseif ($hasName && isset($validated['name'])) {
                $data['name'] = $validated['name'];
            }
            
            // Customer - check for account_id, client_id, or customer_id columns
            $hasCustomerId = \Schema::hasColumn('projects', 'customer_id');
            if (isset($validated['account_id'])) {
                if ($hasAccountId) {
                    $data['account_id'] = $validated['account_id'];
                } elseif ($hasClientId) {
                    $data['client_id'] = $validated['account_id'];
                } elseif ($hasCustomerId) {
                    $data['customer_id'] = $validated['account_id'];
                }
            }
            
            // Handle status specially - map string values to what table expects
            if (isset($validated['status']) && \Schema::hasColumn('projects', 'status')) {
                $statusValue = $validated['status'];
                
                // Get column type to determine format
                $columnType = DB::selectOne("SHOW COLUMNS FROM projects WHERE Field = 'status'")->Type ?? '';
                
                // If it's an ENUM, check what values are allowed
                if (strpos($columnType, 'enum') !== false) {
                    // Extract enum values
                    preg_match("/^enum\(\'(.*)\'\)$/", $columnType, $matches);
                    if (isset($matches[1])) {
                        $enumValues = explode("','", $matches[1]);
                        // Try to find matching value (case-insensitive)
                        foreach ($enumValues as $enumVal) {
                            if (strtolower($enumVal) === strtolower($statusValue)) {
                                $statusValue = $enumVal; // Use exact enum value
                                break;
                            }
                        }
                    }
                } elseif (strpos($columnType, 'int') !== false || strpos($columnType, 'tinyint') !== false) {
                    // Integer column - map status names to numbers
                    $statusMap = [
                        'planned' => 1,
                        'active' => 2,
                        'on_hold' => 3,
                        'completed' => 4,
                        'cancelled' => 5,
                    ];
                    $statusValue = $statusMap[strtolower($statusValue)] ?? 1;
                }
                
                $data['status'] = $statusValue;
            } elseif (isset($validated['status_id'])) {
                $data['status_id'] = $validated['status_id'];
            }
            
            // Copy other validated fields that exist in table (excluding status which is handled above)
            $copyFields = [
                'billing_method', 'fixed_amount', 'hourly_rate', 'currency_id',
                'planned_start_date', 'planned_end_date', 'start_date', 'end_date',
                'scope_notes', 'description', 'internal_notes', 'is_billable'
            ];
            
            foreach ($copyFields as $field) {
                if (isset($validated[$field]) && \Schema::hasColumn('projects', $field)) {
                    $data[$field] = $validated[$field];
                }
            }
            
            // Add audit fields if they exist
            if (\Schema::hasColumn('projects', 'created_by')) {
                $data['created_by'] = $this->admin->id;
            }
            if (\Schema::hasColumn('projects', 'updated_by')) {
                $data['updated_by'] = $this->admin->id;
            }
            
            $team = $validated['team'] ?? [];
            
            $project = Project::create($data);
            
            // Only sync team if project_team table exists
            if (!empty($team) && \Schema::hasTable('project_team')) {
                $project->team()->sync($team);
            }

            DB::commit();

            return redirect()->route('admin.projects.index')
                ->with('success', 'Project created successfully!');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()->with('error', 'Failed to create project: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        // Build relations array based on what tables exist
        $with = [];
        
        if (\Schema::hasTable('project_statuses') && \Schema::hasColumn('projects', 'status_id')) {
            $with[] = 'status';
        }
        if (\Schema::hasTable('project_team')) {
            $with[] = 'team';
        }
        if (\Schema::hasTable('project_milestones')) {
            $with[] = 'milestones';
        }
        if (\Schema::hasTable('project_discussions')) {
            $with[] = 'discussions';
        }
        if (\Schema::hasTable('customers')) {
            $with[] = 'account';
        }
        if (\Schema::hasTable('currencies')) {
            $with[] = 'currency';
        }
        
        $project = Project::with($with)->findOrFail($id);
        
        // Load tasks separately - check for module_tasks or tasks table
        $tasks = collect([]);
        $hasTasksTable = false;
        
        try {
            if (\Schema::hasTable('module_tasks')) {
                $hasTasksTable = true;
                $query = \DB::table('module_tasks')->where('project_id', $id);
                
                // Check if deleted_at column exists
                if (\Schema::hasColumn('module_tasks', 'deleted_at')) {
                    $query->whereNull('deleted_at');
                }
                
                // Check if created_at exists for ordering
                if (\Schema::hasColumn('module_tasks', 'created_at')) {
                    $query->orderBy('created_at', 'desc');
                }
                
                $tasks = $query->get();
            } elseif (\Schema::hasTable('tasks')) {
                $hasTasksTable = true;
                $query = \DB::table('tasks')->where('project_id', $id);
                
                if (\Schema::hasColumn('tasks', 'deleted_at')) {
                    $query->whereNull('deleted_at');
                }
                
                if (\Schema::hasColumn('tasks', 'created_at')) {
                    $query->orderBy('created_at', 'desc');
                }
                
                $tasks = $query->get();
            }
        } catch (\Exception $e) {
            // If any error occurs, just return empty tasks
            $tasks = collect([]);
            $hasTasksTable = false;
        }

        // Calculate completed tasks safely
        $completedCount = 0;
        if ($tasks->count() > 0) {
            $completedCount = $tasks->filter(function($task) {
                return (isset($task->status) && $task->status === 'completed') 
                    || (isset($task->completed_at) && $task->completed_at !== null);
            })->count();
        }

        $stats = [
            'total_tasks' => $tasks->count(),
            'completed_tasks' => $completedCount,
            'total_hours' => $project->total_time_logged ?? 0,
            'billable_hours' => $project->total_billable_hours ?? 0,
            'billed_hours' => $project->total_billed_hours ?? 0,
        ];

        return $this->View('projects::projects.show', compact('project', 'stats', 'tasks', 'hasTasksTable'));
    }
    
    /**
 * Display project overview with comprehensive statistics
 */
public function overview($id)
{
    try {
        // Build relations based on existing tables
        $with = [];
        if (\Schema::hasTable('project_statuses') && \Schema::hasColumn('projects', 'status_id')) {
            $with[] = 'status';
        }
        if (\Schema::hasTable('project_team')) {
            $with[] = 'team';
        }
        if (\Schema::hasTable('tasks')) {
            $with[] = 'tasks.status';
            if (\Schema::hasTable('task_time_logs')) {
                $with[] = 'tasks.timeLogs';
            }
        }
        
        $project = Project::with($with)->findOrFail($id);

        // Calculate project progress based on completed tasks
        $totalTasks = \Schema::hasTable('tasks') ? $project->tasks->count() : 0;
        $completedTasks = 0;
        if ($totalTasks > 0) {
            $completedTasks = $project->tasks->filter(function($task) {
                return $task->status && $task->status->is_completed;
            })->count();
        }
        $projectProgress = $totalTasks > 0 ? ($completedTasks / $totalTasks) * 100 : 0;

        // Calculate total hours and costs
        $totalHours = 0;
        if (\Schema::hasTable('tasks') && \Schema::hasTable('task_time_logs')) {
            $totalHours = $project->tasks->sum(function ($task) {
                return $task->timeLogs ? $task->timeLogs->sum('hours_spent') : 0;
            });
        }

        $estimatedHours = \Schema::hasTable('tasks') ? ($project->tasks->sum('estimated_hours') ?? 0) : 0;

        // Calculate total cost based on billing method
        $totalCost = 0;
        if (isset($project->billing_method)) {
            if ($project->billing_method === 'fixed') {
                $totalCost = $project->fixed_amount ?? 0;
            } elseif ($project->billing_method === 'project_hours') {
                $totalCost = $totalHours * ($project->hourly_rate ?? 0);
            } else { // task_hours
                if (\Schema::hasTable('tasks')) {
                    $totalCost = $project->tasks->sum(function ($task) use ($project) {
                        $hours = $task->timeLogs ? $task->timeLogs->sum('hours_spent') : 0;
                        return $hours * ($task->hourly_rate ?? $project->hourly_rate ?? 0);
                    });
                }
            }
        }

        // Calculate remaining budget
        $remainingBudget = ($project->fixed_amount ?? 0) - $totalCost;

        // Get team members
        $team = \Schema::hasTable('project_team') ? ($project->team ?? collect()) : collect();
        $teamCount = $team->count();

        // Get recent tasks (limit to 10)
        $tasks = collect();
        if (\Schema::hasTable('tasks')) {
            $taskWith = ['status'];
            if (\Schema::hasTable('task_time_logs')) {
                $taskWith[] = 'timeLogs';
            }
            $tasks = $project->tasks()
                ->with($taskWith)
                ->orderBy('created_at', 'desc')
                ->paginate(10);
        }

        return $this->View('projects::projects.overview', compact(
            'project',
            'projectProgress',
            'totalTasks',
            'completedTasks',
            'totalHours',
            'estimatedHours',
            'totalCost',
            'remainingBudget',
            'team',
            'teamCount',
            'tasks'
        ));
    } catch (\Exception $e) {
        \Log::error('Error loading project overview: ' . $e->getMessage());
        return back()->with('error', 'Failed to load project overview.');
    }
}


    public function edit($id)
    {
        // Build relations based on existing tables
        $with = [];
        if (\Schema::hasTable('project_team')) {
            $with[] = 'team';
        }
        
        $project = !empty($with) ? Project::with($with)->findOrFail($id) : Project::findOrFail($id);
        
        // Get statuses - from table or provide defaults
        $statuses = collect([]);
        $statusOptions = [];
        
        if (\Schema::hasTable('project_statuses')) {
            $statuses = ProjectStatus::where('is_active', true)->orderBy('sort_order')->get();
        } elseif (\Schema::hasColumn('projects', 'status')) {
            // Try to get enum values from column
            try {
                $columnType = DB::selectOne("SHOW COLUMNS FROM projects WHERE Field = 'status'")->Type ?? '';
                
                if (strpos($columnType, 'enum') !== false) {
                    preg_match("/^enum\(\'(.*)\'\)$/", $columnType, $matches);
                    if (isset($matches[1])) {
                        $enumValues = explode("','", $matches[1]);
                        foreach ($enumValues as $value) {
                            $statusOptions[] = [
                                'value' => $value,
                                'label' => ucfirst(str_replace('_', ' ', $value)),
                            ];
                        }
                    }
                }
            } catch (\Exception $e) {
                // If can't get enum values, use defaults
            }
            
            if (empty($statusOptions)) {
                $statusOptions = [
                    ['value' => '1', 'label' => 'Planned'],
                    ['value' => '2', 'label' => 'Active'],
                    ['value' => '3', 'label' => 'On Hold'],
                    ['value' => '4', 'label' => 'Completed'],
                    ['value' => '5', 'label' => 'Cancelled'],
                ];
            }
        }
        
        // Fetch customers
        $customers = collect([]);
        if (\Schema::hasTable('customers')) {
            $query = \DB::table('customers');
            if (\Schema::hasColumn('customers', 'active')) {
                $query->where('active', true);
            }
            if (\Schema::hasColumn('customers', 'name')) {
                $query->orderBy('name');
            } elseif (\Schema::hasColumn('customers', 'company')) {
                $query->orderBy('company');
            }
            $customers = $query->get();
        }
        
        $currencies = collect([]);
        if (\Schema::hasTable('currencies')) {
            $currencies = \DB::table('currencies')->where('is_active', true)->get();
        }
        
        // Get staff/admins for team assignment - check staffs first, then admins
        $admins = collect([]);
        if (\Schema::hasTable('staffs')) {
            $query = \DB::table('staffs');
            
            // Apply active filter
            if (\Schema::hasColumn('staffs', 'active')) {
                $query->where('active', 1);
            } elseif (\Schema::hasColumn('staffs', 'is_active')) {
                $query->where('is_active', 1);
            }
            
            // Order by available column
            if (\Schema::hasColumn('staffs', 'firstname')) {
                $query->orderBy('firstname');
            } elseif (\Schema::hasColumn('staffs', 'name')) {
                $query->orderBy('name');
            } else {
                $query->orderBy('id');
            }
            
            $admins = $query->get()->map(function($staff) {
                // Build display name from available columns
                if (isset($staff->firstname) || isset($staff->lastname)) {
                    $staff->name = trim(($staff->firstname ?? '') . ' ' . ($staff->lastname ?? ''));
                }
                if (empty($staff->name)) {
                    $staff->name = $staff->email ?? 'Staff #' . $staff->id;
                }
                return $staff;
            });
        } elseif (class_exists('App\Models\Admin') && \Schema::hasTable('admins')) {
            $admins = \App\Models\Admin::where('is_active', true)->orderBy('name')->get();
        }

        return $this->View('projects::projects.edit', compact('project', 'statuses', 'statusOptions', 'customers', 'currencies', 'admins'));
    }

    public function update(Request $request, $id)
    {
        $project = Project::findOrFail($id);

        // Detect which columns exist
        $hasTitle = \Schema::hasColumn('projects', 'title');
        $hasName = \Schema::hasColumn('projects', 'name');
        $hasAccountId = \Schema::hasColumn('projects', 'account_id');
        $hasClientId = \Schema::hasColumn('projects', 'client_id');
        $hasCustomerId = \Schema::hasColumn('projects', 'customer_id');
        $hasBillingMethod = \Schema::hasColumn('projects', 'billing_method');
        $hasStatusId = \Schema::hasColumn('projects', 'status_id');
        $hasStatus = \Schema::hasColumn('projects', 'status');
        
        // Build validation rules dynamically
        // Determine team table for validation
        $teamTable = \Schema::hasTable('staffs') ? 'staffs' : 'admins';
        
        $rules = [
            'title' => 'required|string|max:191',
            'team' => 'array',
            'team.*' => "exists:{$teamTable},id",
        ];
        
        if ($hasAccountId || $hasClientId || $hasCustomerId) {
            $rules['account_id'] = 'nullable|exists:customers,id';
        }
        
        if ($hasBillingMethod) {
            $rules['billing_method'] = 'nullable|in:fixed,project_hours,task_hours';
            $rules['fixed_amount'] = 'nullable|numeric|min:0';
            $rules['hourly_rate'] = 'nullable|numeric|min:0';
        }
        
        if (\Schema::hasColumn('projects', 'currency_id') && \Schema::hasTable('currencies')) {
            $rules['currency_id'] = 'nullable|exists:currencies,id';
        }
        
        if ($hasStatusId && \Schema::hasTable('project_statuses')) {
            $rules['status_id'] = 'nullable|exists:project_statuses,id';
        } elseif ($hasStatus) {
            $rules['status'] = 'nullable|string|max:50';
        }
        
        if (\Schema::hasColumn('projects', 'planned_start_date')) {
            $rules['planned_start_date'] = 'nullable|date';
            $rules['planned_end_date'] = 'nullable|date|after_or_equal:planned_start_date';
        } elseif (\Schema::hasColumn('projects', 'start_date')) {
            $rules['start_date'] = 'nullable|date';
            $rules['end_date'] = 'nullable|date';
        }
        
        if (\Schema::hasColumn('projects', 'scope_notes')) {
            $rules['scope_notes'] = 'nullable|string';
        } elseif (\Schema::hasColumn('projects', 'description')) {
            $rules['description'] = 'nullable|string';
        }
        
        if (\Schema::hasColumn('projects', 'internal_notes')) {
            $rules['internal_notes'] = 'nullable|string';
        }
        
        if (\Schema::hasColumn('projects', 'is_billable')) {
            $rules['is_billable'] = 'boolean';
        }
        
        $validated = $request->validate($rules);

        DB::beginTransaction();
        try {
            // Build data array with correct column mappings
            $data = [];
            
            // Title/Name
            if ($hasTitle) {
                $data['title'] = $validated['title'];
            } elseif ($hasName) {
                $data['name'] = $validated['title'];
            }
            
            // Customer
            if (isset($validated['account_id'])) {
                if ($hasAccountId) {
                    $data['account_id'] = $validated['account_id'];
                } elseif ($hasClientId) {
                    $data['client_id'] = $validated['account_id'];
                } elseif ($hasCustomerId) {
                    $data['customer_id'] = $validated['account_id'];
                }
            }
            
            // Status
            if (isset($validated['status']) && $hasStatus) {
                $data['status'] = $validated['status'];
            } elseif (isset($validated['status_id']) && $hasStatusId) {
                $data['status_id'] = $validated['status_id'];
            }
            
            // Copy other fields that exist
            $copyFields = [
                'billing_method', 'fixed_amount', 'hourly_rate', 'currency_id',
                'planned_start_date', 'planned_end_date', 'start_date', 'end_date',
                'scope_notes', 'description', 'internal_notes', 'is_billable'
            ];
            
            foreach ($copyFields as $field) {
                if (isset($validated[$field]) && \Schema::hasColumn('projects', $field)) {
                    $data[$field] = $validated[$field];
                }
            }
            
            if (\Schema::hasColumn('projects', 'updated_by')) {
                $data['updated_by'] = $this->admin->id;
            }
            
            $team = $validated['team'] ?? [];
            
            $project->update($data);
            
            // Only sync team if table exists
            if (!empty($team) && \Schema::hasTable('project_team')) {
                $project->team()->sync($team);
            }

            DB::commit();

            return redirect()->route('admin.projects.index')
                ->with('success', 'Project updated successfully!');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withInput()->with('error', 'Failed to update project: ' . $e->getMessage());
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $project = Project::with(['tasks', 'milestones', 'discussions'])->findOrFail($id);
            
            \Log::info('Deleting project', [
                'project_id' => $id,
                'title' => $project->title,
                'deleted_by' => $this->admin->id
            ]);
            
            // Count relationships for logging
            $stats = [
                'tasks' => $project->tasks()->count(),
                'milestones' => $project->milestones()->count(),
                'discussions' => $project->discussions()->count(),
                'time_logs' => $project->timeLogs()->count(),
            ];
            
            // Delete all tasks (will cascade delete time logs, comments, attachments via model events)
            $project->tasks()->each(function($task) {
                // Delete task comments
                $task->comments()->delete();
                // Delete task attachments (also delete physical files)
                $task->attachments()->each(function($attachment) {
                    \Storage::disk('public')->delete($attachment->file_path);
                    $attachment->delete();
                });
                // Delete time logs
                $task->timeLogs()->delete();
                // Detach owners and followers
                $task->owners()->detach();
                $task->followers()->detach();
                // Delete the task
                $task->delete();
            });
            
            // Delete milestones
            $project->milestones()->delete();
            
            // Delete discussions and their replies
            $project->discussions()->each(function($discussion) {
                $discussion->replies()->delete();
                $discussion->delete();
            });
            
            // Detach team members (pivot table)
            $project->team()->detach();
            
            // Finally, delete the project itself
            $project->delete();
            
            DB::commit();
            
            \Log::info('Project deleted successfully', [
                'project_id' => $id,
                'stats' => $stats
            ]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => true, 
                    'message' => 'Project and all related data deleted successfully!',
                    'stats' => $stats
                ]);
            }
            
            return redirect()->route('admin.projects.index')
                ->with('success', 'Project deleted successfully!');
                
        } catch (\Exception $e) {
            DB::rollback();
            
            \Log::error('Project deletion failed', [
                'project_id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false, 
                    'message' => 'Failed to delete project: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Failed to delete project');
        }
    }

    public function bulkDelete(Request $request): JsonResponse
    {
        $ids = $request->input('ids', []);
        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No projects selected'], 400);
        }

        $deleted = Project::whereIn('id', $ids)->delete();
        return response()->json(['success' => true, 'message' => "{$deleted} projects deleted!"]);
    }

    public function pin($id)
    {
        $project = Project::findOrFail($id);
        $project->is_pinned = !$project->is_pinned;
        $project->save();

        return response()->json([
            'success' => true,
            'is_pinned' => $project->is_pinned,
            'message' => $project->is_pinned ? 'Project pinned!' : 'Project unpinned!'
        ]);
    }

    public function duplicate($id)
    {
        $original = Project::with(['milestones', 'tasks'])->findOrFail($id);

        DB::beginTransaction();
        try {
            $project = $original->replicate();
            $project->title = $original->title . ' (Copy)';
            $project->created_by = $this->admin->id;
            $project->updated_by = $this->admin->id;
            $project->is_pinned = false;
            $project->save();

            // Copy team members
            $project->team()->sync($original->team->pluck('id'));

            // Copy milestones
            foreach ($original->milestones as $milestone) {
                $newMilestone = $milestone->replicate();
                $newMilestone->project_id = $project->id;
                $newMilestone->is_completed = false;
                $newMilestone->completed_at = null;
                $newMilestone->save();
            }

            // Copy tasks
            foreach ($original->tasks as $task) {
                $newTask = $task->replicate();
                $newTask->project_id = $project->id;
                $newTask->completed_at = null;
                $newTask->save();

                // Copy task owners and followers
                $newTask->owners()->sync($task->owners->pluck('id'));
                $newTask->followers()->sync($task->followers->pluck('id'));
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Project duplicated successfully!',
                'project_id' => $project->id,
                'url' => route('admin.projects.show', $project->id)
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['success' => false, 'message' => 'Failed to duplicate project'], 500);
        }
    }

    public function finance($id)
    {
        $project = Project::with(['tasks.timeLogs', 'currency'])->findOrFail($id);

        $timeLogs = $project->timeLogs()->with(['task', 'admin'])->get();
        
        $unbilledLogs = $timeLogs->where('is_billed', false)->where('is_billable', true);
        $billedLogs = $timeLogs->where('is_billed', true);

        $stats = [
            'total_hours' => $timeLogs->sum('duration_minutes') / 60,
            'billable_hours' => $timeLogs->where('is_billable', true)->sum('duration_minutes') / 60,
            'unbilled_hours' => $unbilledLogs->sum('duration_minutes') / 60,
            'billed_hours' => $billedLogs->sum('duration_minutes') / 60,
            'unbilled_amount' => $unbilledLogs->sum(function($log) {
                return ($log->duration_minutes / 60) * ($log->hourly_rate ?? 0);
            }),
            'billed_amount' => $billedLogs->sum(function($log) {
                return ($log->duration_minutes / 60) * ($log->hourly_rate ?? 0);
            }),
        ];

        return $this->View('projects::projects.finance', compact('project', 'stats', 'unbilledLogs', 'billedLogs'));
    }

    public function generateInvoice(Request $request, $id)
    {
        // This would integrate with your invoice module
        // For now, just return a placeholder
        return response()->json([
            'success' => true,
            'message' => 'Invoice generation functionality to be integrated with Invoice module'
        ]);
    }
}