<?php

namespace Modules\Projects\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Schema;

class Discussion extends Model
{
    protected $table = 'project_discussions';

    protected $fillable = [
        'project_id',
        'title',
        'message',
        'portal_visible',
        'created_by',
    ];

    protected $casts = [
        'portal_visible' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    public function creator()
    {
        // Try Staff model first, then Admin
        if (class_exists('App\Models\Staff')) {
            return $this->belongsTo('App\Models\Staff', 'created_by');
        }
        if (class_exists('App\Models\Admin')) {
            return $this->belongsTo('App\Models\Admin', 'created_by');
        }
        // Fallback - return empty relation
        return $this->belongsTo(Discussion::class, 'id', 'id')->whereRaw('1 = 0');
    }

    public function replies()
    {
        if (!Schema::hasTable('project_discussion_replies')) {
            return $this->hasMany(Discussion::class, 'id', 'id')->whereRaw('1 = 0');
        }
        return $this->hasMany(DiscussionReply::class, 'discussion_id')->orderBy('created_at', 'asc');
    }

    /**
     * Get all attachments for this discussion
     */
    public function attachments()
    {
        if (!Schema::hasTable('project_discussion_attachments')) {
            return $this->hasMany(Discussion::class, 'id', 'id')->whereRaw('1 = 0');
        }
        return $this->hasMany(DiscussionAttachment::class, 'discussion_id');
    }

    // ==================== SCOPES ====================

    public function scopePortalVisible(Builder $query): Builder
    {
        if (!Schema::hasColumn('project_discussions', 'portal_visible')) {
            return $query;
        }
        return $query->where('portal_visible', true);
    }

    public function scopeByProject(Builder $query, $projectId): Builder
    {
        return $query->where('project_id', $projectId);
    }

    // ==================== ACCESSORS ====================

    public function getRepliesCountAttribute(): int
    {
        try {
            return $this->replies()->count();
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getAttachmentsCountAttribute(): int
    {
        try {
            return $this->attachments()->count();
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getHasAttachmentsAttribute(): bool
    {
        try {
            return $this->attachments()->exists();
        } catch (\Exception $e) {
            return false;
        }
    }
}