<?php

namespace Modules\Projects\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Schema;

class Milestone extends Model
{
    protected $table = 'project_milestones';

    protected $fillable = [
        'project_id',
        'name',
        'description',
        'target_date',
        'completed_at',
        'is_completed',
        'portal_visible',
        'sort_order',
        'created_by',
    ];

    protected $casts = [
        'target_date' => 'date',
        'completed_at' => 'datetime',
        'is_completed' => 'boolean',
        'portal_visible' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

    public function project()
    {
        return $this->belongsTo(Project::class, 'project_id');
    }

    public function creator()
    {
        // Try Staff model first, then Admin
        if (class_exists('App\Models\Staff')) {
            return $this->belongsTo('App\Models\Staff', 'created_by');
        }
        if (class_exists('App\Models\Admin')) {
            return $this->belongsTo('App\Models\Admin', 'created_by');
        }
        return $this->belongsTo(Milestone::class, 'id', 'id')->whereRaw('1 = 0');
    }

    public function tasks()
    {
        // Check for module_tasks or tasks table
        if (Schema::hasTable('module_tasks')) {
            $taskClass = 'Modules\\Tasks\\Models\\Task';
            if (class_exists($taskClass)) {
                return $this->hasMany($taskClass, 'milestone_id');
            }
        }
        if (Schema::hasTable('tasks')) {
            $taskClass = 'Modules\\Tasks\\Models\\Task';
            if (class_exists($taskClass)) {
                return $this->hasMany($taskClass, 'milestone_id');
            }
        }
        return $this->hasMany(Milestone::class, 'id', 'id')->whereRaw('1 = 0');
    }

    // ==================== SCOPES ====================

    public function scopeCompleted(Builder $query): Builder
    {
        if (!Schema::hasColumn('project_milestones', 'is_completed')) {
            return $query->whereRaw('1 = 0');
        }
        return $query->where('is_completed', true);
    }

    public function scopeIncomplete(Builder $query): Builder
    {
        if (!Schema::hasColumn('project_milestones', 'is_completed')) {
            return $query;
        }
        return $query->where('is_completed', false);
    }

    public function scopePortalVisible(Builder $query): Builder
    {
        if (!Schema::hasColumn('project_milestones', 'portal_visible')) {
            return $query;
        }
        return $query->where('portal_visible', true);
    }

    public function scopeByProject(Builder $query, $projectId): Builder
    {
        return $query->where('project_id', $projectId);
    }

    // ==================== ACCESSORS ====================

    public function getIsOverdueAttribute(): bool
    {
        if (!$this->target_date || $this->is_completed) {
            return false;
        }
        try {
            return $this->target_date->isPast();
        } catch (\Exception $e) {
            return false;
        }
    }

    public function getCompletionPercentageAttribute(): float
    {
        try {
            $total = $this->tasks()->count();
            if ($total == 0) {
                return 0;
            }
            $completed = $this->tasks()->whereHas('status', function ($q) {
                $q->where('is_completed', true);
            })->count();
            return round(($completed / $total) * 100, 2);
        } catch (\Exception $e) {
            return 0;
        }
    }
}
