<?php

namespace Modules\Projects\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;

class Project extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'name',  // Alternative column name
        'title', // Standard column name
        'account_id',
        'client_id', // Alternative to account_id
        'customer_id', // Another alternative to account_id
        'billing_method',
        'billing_type', // Alternative
        'fixed_amount',
        'hourly_rate',
        'currency_id',
        'status_id',
        'status', // Simple status column fallback
        'progress', // Progress percentage
        'planned_start_date',
        'planned_end_date',
        'actual_start_date',
        'actual_end_date',
        'start_date', // Alternative column names
        'end_date',
        'due_date',
        'deadline',
        'scope_notes',
        'description', // Alternative to scope_notes
        'internal_notes',
        'notes',
        'code', // Project code
        'is_pinned',
        'is_billable',
        // Portal visibility flags
        'portal_show_tasks',
        'portal_allow_task_comments',
        'portal_show_task_conversations',
        'portal_show_task_files',
        'portal_show_checklist',
        'portal_allow_task_uploads',
        'portal_show_time_logs',
        'portal_show_finance',
        'portal_allow_file_uploads',
        'portal_show_discussions',
        'portal_show_milestones',
        'portal_show_gantt',
        'portal_show_activity',
        'portal_show_team',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'fixed_amount' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'planned_start_date' => 'date',
        'planned_end_date' => 'date',
        'actual_start_date' => 'date',
        'actual_end_date' => 'date',
        'is_pinned' => 'boolean',
        'is_billable' => 'boolean',
        'portal_show_tasks' => 'boolean',
        'portal_allow_task_comments' => 'boolean',
        'portal_show_task_conversations' => 'boolean',
        'portal_show_task_files' => 'boolean',
        'portal_show_checklist' => 'boolean',
        'portal_allow_task_uploads' => 'boolean',
        'portal_show_time_logs' => 'boolean',
        'portal_show_finance' => 'boolean',
        'portal_allow_file_uploads' => 'boolean',
        'portal_show_discussions' => 'boolean',
        'portal_show_milestones' => 'boolean',
        'portal_show_gantt' => 'boolean',
        'portal_show_activity' => 'boolean',
        'portal_show_team' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'deleted_at' => 'datetime',
    ];

    // ==================== RELATIONSHIPS ====================

public function account()
{
    // Determine which foreign key column exists and has value
    // Priority: customer_id > client_id > account_id (customer_id is most common)
    $foreignKey = null;
    
    if (\Schema::hasColumn('projects', 'customer_id')) {
        $foreignKey = 'customer_id';
    } elseif (\Schema::hasColumn('projects', 'client_id')) {
        $foreignKey = 'client_id';
    } elseif (\Schema::hasColumn('projects', 'account_id')) {
        $foreignKey = 'account_id';
    } else {
        $foreignKey = 'customer_id'; // Default fallback
    }
    
    // Only define relationship if Customer model exists
    if (class_exists('App\Models\Customer')) {
        return $this->belongsTo('App\Models\Customer', $foreignKey);
    }
    // Return empty relation if model doesn't exist
    return $this->belongsTo(Project::class, 'id', 'id')->whereRaw('1 = 0');
}

// Alias for customer relationship
public function customer()
{
    return $this->account();
}

// Alias for client relationship
public function client()
{
    return $this->account();
}

public function currency()
{
    // Only define relationship if Currency model exists
    if (class_exists('App\Models\Currency')) {
        return $this->belongsTo('App\Models\Currency', 'currency_id');
    }
    // Return empty relation if model doesn't exist
    return $this->belongsTo(Project::class, 'currency_id')->whereRaw('1 = 0');
}

public function status()
{
    if (!\Schema::hasTable('project_statuses')) {
        return $this->belongsTo(Project::class, 'id', 'id')->whereRaw('1 = 0');
    }
    return $this->belongsTo(ProjectStatus::class, 'status_id');
}

public function creator()
{
    return $this->belongsTo('App\Models\Admin', 'created_by');
}

public function updater()
{
    return $this->belongsTo('App\Models\Admin', 'updated_by');
}

    public function tasks()
    {
        // Check for different possible task table names
        $taskTable = null;
        if (\Schema::hasTable('module_tasks')) {
            $taskTable = 'module_tasks';
        } elseif (\Schema::hasTable('tasks')) {
            $taskTable = 'tasks';
        }
        
        if (!$taskTable) {
            return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
        }
        
        // Use string class name to avoid errors if Tasks module not installed
        $taskClass = 'Modules\\Tasks\\Models\\Task';
        if (!class_exists($taskClass)) {
            // Try App\Models\Task as fallback
            if (class_exists('App\\Models\\Task')) {
                return $this->hasMany('App\\Models\\Task', 'project_id');
            }
            // Return direct DB query as last resort
            return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
        }
        return $this->hasMany($taskClass, 'project_id');
    }
    
    // Direct relationship to module_tasks table
    public function moduleTasks()
    {
        if (!\Schema::hasTable('module_tasks')) {
            return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
        }
        
        // Use dynamic model or direct query
        $taskClass = 'Modules\\Tasks\\Models\\Task';
        if (class_exists($taskClass)) {
            return $this->hasMany($taskClass, 'project_id');
        }
        
        // Fallback to App\Models\Task
        if (class_exists('App\\Models\\Task')) {
            return $this->hasMany('App\\Models\\Task', 'project_id');
        }
        
        return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
    }

    public function milestones()
    {
        if (!\Schema::hasTable('project_milestones')) {
            return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
        }
        return $this->hasMany(Milestone::class, 'project_id')->orderBy('sort_order');
    }

    public function timeLogs()
    {
        if (!\Schema::hasTable('task_time_logs') || !\Schema::hasTable('tasks')) {
            return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
        }
        $taskClass = 'Modules\\Tasks\\Models\\Task';
        $timeLogClass = 'Modules\\Tasks\\Models\\TimeLog';
        if (!class_exists($taskClass) || !class_exists($timeLogClass)) {
            return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
        }
        return $this->hasManyThrough($timeLogClass, $taskClass, 'project_id', 'task_id');
    }

    public function discussions()
    {
        if (!\Schema::hasTable('project_discussions')) {
            return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
        }
        return $this->hasMany(Discussion::class, 'project_id')->orderBy('created_at', 'desc');
    }

    public function team()
    {
        if (!\Schema::hasTable('project_team')) {
            // Return a hasMany relation that returns empty - avoids belongsToMany alias issues
            return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
        }
        
        // Determine the foreign key in project_team table
        // Priority: staff_id > admin_id > user_id
        $foreignKey = 'staff_id';
        if (!\Schema::hasColumn('project_team', 'staff_id')) {
            if (\Schema::hasColumn('project_team', 'admin_id')) {
                $foreignKey = 'admin_id';
            } elseif (\Schema::hasColumn('project_team', 'user_id')) {
                $foreignKey = 'user_id';
            }
        }
        
        // Determine which model/table to use for team members
        // Priority: staffs table > admins table
        $staffModel = null;
        $staffTable = null;
        
        if (\Schema::hasTable('staffs')) {
            $staffTable = 'staffs';
            if (class_exists('App\Models\Staff')) {
                $staffModel = 'App\Models\Staff';
            }
        }
        
        if (!$staffModel && \Schema::hasTable('admins')) {
            $staffTable = 'admins';
            if (class_exists('App\Models\Admin')) {
                $staffModel = 'App\Models\Admin';
            }
        }
        
        // If we have a model, use it for the relationship
        if ($staffModel) {
            return $this->belongsToMany($staffModel, 'project_team', 'project_id', $foreignKey)
                ->withTimestamps();
        }
        
        // Fallback: return empty relation
        return $this->hasMany(Project::class, 'id', 'id')->whereRaw('1 = 0');
    }

    // ==================== SCOPES ====================

    public function scopeActive(Builder $query): Builder
    {
        // Check if status_id column exists
        if (\Schema::hasColumn('projects', 'status_id')) {
            return $query->whereHas('status', function ($q) {
                $q->where('is_active', true);
            });
        }
        // Fallback - return all if no status column
        return $query;
    }

    public function scopePinned(Builder $query): Builder
    {
        if (\Schema::hasColumn('projects', 'is_pinned')) {
            return $query->where('is_pinned', true);
        }
        return $query;
    }

    public function scopeBillable(Builder $query): Builder
    {
        if (\Schema::hasColumn('projects', 'is_billable')) {
            return $query->where('is_billable', true);
        }
        return $query;
    }

    public function scopeSearch(Builder $query, ?string $search): Builder
    {
        if (empty($search)) {
            return $query;
        }

        return $query->where(function ($q) use ($search) {
            $q->where('title', 'LIKE', "%{$search}%")
                ->orWhere('scope_notes', 'LIKE', "%{$search}%")
                ->orWhere('internal_notes', 'LIKE', "%{$search}%");
        });
    }

    public function scopeByAccount(Builder $query, $accountId): Builder
    {
        return $query->where('account_id', $accountId);
    }

    public function scopeByStatus(Builder $query, $statusId): Builder
    {
        return $query->where('status_id', $statusId);
    }

    public function scopeByTeamMember(Builder $query, $staffId): Builder
    {
        // Only apply filter if project_team table exists
        if (!\Schema::hasTable('project_team')) {
            return $query->whereRaw('1 = 0'); // Return empty if no team table
        }
        
        // Determine which column to use - priority: staff_id > admin_id > user_id
        $foreignKey = 'staff_id';
        if (!\Schema::hasColumn('project_team', 'staff_id')) {
            if (\Schema::hasColumn('project_team', 'admin_id')) {
                $foreignKey = 'admin_id';
            } elseif (\Schema::hasColumn('project_team', 'user_id')) {
                $foreignKey = 'user_id';
            } else {
                return $query->whereRaw('1 = 0'); // Return empty if no valid column
            }
        }
        
        return $query->whereHas('team', function ($q) use ($staffId, $foreignKey) {
            $q->where($foreignKey, $staffId);
        });
    }

    // ==================== ACCESSORS ====================

    public function getStatusLabelAttribute(): string
    {
        // If using status_id with relation
        if ($this->status_id) {
            // Try to get from loaded relation first
            if ($this->relationLoaded('status') && $this->getRelation('status')) {
                return $this->getRelation('status')->label ?? 'Unknown';
            }
            // Try to load the status
            $status = $this->status;
            if ($status) {
                return $status->label ?? 'Unknown';
            }
        }
        
        // If using simple status column (supports both string and integer)
        if (isset($this->attributes['status']) && $this->attributes['status'] !== null) {
            $labels = [
                'planned' => 'Planned', '1' => 'Planned', 1 => 'Planned',
                'active' => 'Active', '2' => 'Active', 2 => 'Active',
                'on_hold' => 'On Hold', '3' => 'On Hold', 3 => 'On Hold',
                'completed' => 'Completed', '4' => 'Completed', 4 => 'Completed',
                'cancelled' => 'Cancelled', '5' => 'Cancelled', 5 => 'Cancelled',
            ];
            return $labels[$this->attributes['status']] ?? ucfirst(str_replace('_', ' ', $this->attributes['status']));
        }
        
        return 'Unknown';
    }

    public function getStatusColorAttribute(): string
    {
        // If using status_id with relation
        if ($this->status_id) {
            // Try to get from loaded relation first
            if ($this->relationLoaded('status') && $this->getRelation('status')) {
                return $this->getRelation('status')->color ?? '#9ca3af';
            }
            // Try to load the status
            $status = $this->status;
            if ($status) {
                return $status->color ?? '#9ca3af';
            }
        }
        
        // If using simple status column (supports both string and integer)
        if (isset($this->attributes['status']) && $this->attributes['status'] !== null) {
            $colors = [
                'planned' => '#3b82f6', '1' => '#3b82f6', 1 => '#3b82f6',
                'active' => '#10b981', '2' => '#10b981', 2 => '#10b981',
                'on_hold' => '#f59e0b', '3' => '#f59e0b', 3 => '#f59e0b',
                'completed' => '#22c55e', '4' => '#22c55e', 4 => '#22c55e',
                'cancelled' => '#ef4444', '5' => '#ef4444', 5 => '#ef4444',
            ];
            return $colors[$this->attributes['status']] ?? '#9ca3af';
        }
        
        return '#9ca3af';
    }

    public function getBillingMethodLabelAttribute(): string
    {
        $methods = config('projectmanagement.billing_methods');
        return $methods[$this->billing_method] ?? ucfirst($this->billing_method ?? 'Unknown');
    }

    public function getTotalTimeLoggedAttribute(): float
    {
        try {
            if (!\Schema::hasTable('task_time_logs') || !\Schema::hasTable('tasks')) {
                return 0;
            }
            return $this->timeLogs()->sum('duration_minutes') / 60;
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getTotalBillableHoursAttribute(): float
    {
        try {
            if (!\Schema::hasTable('task_time_logs') || !\Schema::hasTable('tasks')) {
                return 0;
            }
            // Check if is_billable column exists
            if (\Schema::hasColumn('task_time_logs', 'is_billable')) {
                return $this->timeLogs()->where('task_time_logs.is_billable', true)->sum('duration_minutes') / 60;
            }
            return 0;
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getTotalBilledHoursAttribute(): float
    {
        try {
            if (!\Schema::hasTable('task_time_logs') || !\Schema::hasTable('tasks')) {
                return 0;
            }
            // Check if is_billed column exists
            if (\Schema::hasColumn('task_time_logs', 'is_billed')) {
                return $this->timeLogs()->where('task_time_logs.is_billed', true)->sum('duration_minutes') / 60;
            }
            return 0;
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getCompletedTasksCountAttribute(): int
    {
        try {
            if (!\Schema::hasTable('tasks')) {
                return 0;
            }
            $taskClass = 'Modules\\Tasks\\Models\\Task';
            if (!class_exists($taskClass)) {
                return 0;
            }
            if (\Schema::hasTable('task_statuses')) {
                return $this->tasks()->whereHas('status', function ($q) {
                    $q->where('is_completed', true);
                })->count();
            }
            return 0;
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getTotalTasksCountAttribute(): int
    {
        try {
            if (!\Schema::hasTable('tasks')) {
                return 0;
            }
            $taskClass = 'Modules\\Tasks\\Models\\Task';
            if (!class_exists($taskClass)) {
                return 0;
            }
            return $this->tasks()->count();
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function getProgressPercentageAttribute(): float
    {
        $total = $this->total_tasks_count;
        if ($total == 0) {
            return 0;
        }
        return round(($this->completed_tasks_count / $total) * 100, 2);
    }
}
