<style>
/* Modern Projects Index - Theme-Aware Version */
:root {
    --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    --success-gradient: linear-gradient(135deg, #56ab2f 0%, #a8e063 100%);
    --warning-gradient: linear-gradient(135deg, #f2994a 0%, #f2c94c 100%);
    --danger-gradient: linear-gradient(135deg, #eb3349 0%, #f45c43 100%);
    --card-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    --hover-shadow: 0 12px 40px rgba(102, 126, 234, 0.2);
}

.projects-container {
    padding: 24px;
    background: var(--body-bg);
    min-height: 100vh;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Page Header - COMPACT VERSION */
.page-header {
    background: var(--primary-gradient);
    padding: 18px 24px;
    border-radius: 12px;
    margin-bottom: 20px;
    box-shadow: var(--card-shadow);
    animation: slideDown 0.5s ease-out;
}

@keyframes slideDown {
    from { opacity: 0; transform: translateY(-20px); }
    to { opacity: 1; transform: translateY(0); }
}

.header-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 16px;
}

.page-header h1 {
    color: white;
    font-size: 20px;
    font-weight: 700;
    margin-bottom: 4px;
    display: flex;
    align-items: center;
    gap: 10px;
    letter-spacing: -0.02em;
}

.page-header h1 svg {
    width: 22px;
    height: 22px;
    animation: iconPulse 2s ease-in-out infinite;
}

@keyframes iconPulse {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.08); }
}

.page-header p {
    color: rgba(255, 255, 255, 0.95);
    font-size: 12px;
    font-weight: 400;
    letter-spacing: 0.01em;
}

/* Search Bar in Header - COMPACT */
.header-search {
    flex: 1;
    max-width: 350px;
}

.search-input-wrapper {
    position: relative;
}

.search-input {
    width: 100%;
    padding: 8px 14px 8px 36px;
    border-radius: 8px;
    border: 2px solid rgba(255, 255, 255, 0.3);
    font-size: 13px;
    background: rgba(255, 255, 255, 0.15);
    color: white;
    transition: all 0.3s ease;
}

.search-input::placeholder {
    color: rgba(255, 255, 255, 0.7);
}

.search-input:focus {
    outline: none;
    background: white;
    color: #1a202c;
    border-color: white;
}

.search-input:focus::placeholder {
    color: #9ca3af;
}

.search-icon {
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    width: 16px;
    height: 16px;
    color: rgba(255, 255, 255, 0.7);
    pointer-events: none;
    transition: color 0.3s ease;
}

.search-input:focus + .search-icon {
    color: #667eea;
}

.header-actions {
    display: flex;
    gap: 10px;
}

.btn {
    padding: 8px 16px;
    border-radius: 8px;
    font-weight: 600;
    font-size: 12px;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 6px;
    text-decoration: none;
}

.btn svg {
    width: 14px;
    height: 14px;
}

.btn-primary {
    background: white;
    color: #667eea;
    box-shadow: 0 4px 15px rgba(255, 255, 255, 0.3);
}

.btn-primary:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(255, 255, 255, 0.5);
}

.btn-success {
    background: var(--success-gradient);
    color: white;
    box-shadow: 0 4px 15px rgba(86, 171, 47, 0.3);
}

.btn-success:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(86, 171, 47, 0.4);
}

.btn-danger {
    background: var(--danger-gradient);
    color: white;
}

.btn-danger:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(235, 51, 73, 0.4);
}

.btn-sm {
    padding: 6px 12px;
    font-size: 11px;
}

/* Stats Bar - COMPACT VERSION */
.stats-bar {
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 12px;
    margin-bottom: 20px;
    animation: fadeInUp 0.5s ease-out 0.15s backwards;
}

@keyframes fadeInUp {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

.stat-card {
    background: var(--card-bg);
    padding: 14px 16px;
    border-radius: 12px;
    box-shadow: var(--card-shadow);
    display: flex;
    align-items: center;
    gap: 12px;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    border: 1px solid var(--card-border);
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: var(--primary-gradient);
    transform: scaleY(0);
    transform-origin: top;
    transition: transform 0.3s ease;
}

.stat-card:hover::before {
    transform: scaleY(1);
}

.stat-card:hover {
    transform: translateY(-6px) scale(1.02);
    box-shadow: var(--hover-shadow);
}

.stat-icon {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: var(--primary-gradient);
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.25);
    transition: transform 0.3s ease;
    flex-shrink: 0;
}

.stat-card:hover .stat-icon {
    transform: rotate(5deg) scale(1.1);
}

.stat-icon svg {
    width: 18px;
    height: 18px;
    color: white;
}

.stat-content h3 {
    font-size: 22px;
    font-weight: 700;
    color: var(--text-primary);
    margin: 0;
    letter-spacing: -0.02em;
}

.stat-content p {
    font-size: 11px;
    color: var(--text-muted);
    margin: 2px 0 0 0;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    font-weight: 600;
}

/* Filters Bar */
.filters-bar {
    background: var(--card-bg);
    padding: 20px;
    border-radius: 14px;
    box-shadow: var(--card-shadow);
    margin-bottom: 28px;
    animation: fadeInUp 0.5s ease-out 0.3s backwards;
    border: 1px solid var(--card-border);
}

.filters-row {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
    align-items: center;
}

.quick-filters {
    display: flex;
    gap: 8px;
    flex-wrap: wrap;
}

.quick-filter-btn {
    padding: 8px 16px;
    border-radius: 8px;
    font-size: 13px;
    font-weight: 600;
    border: 2px solid var(--card-border);
    background: var(--card-bg);
    color: var(--text-secondary);
    cursor: pointer;
    transition: all 0.3s ease;
    white-space: nowrap;
}

.quick-filter-btn:hover {
    border-color: #667eea;
    color: #667eea;
    transform: translateY(-1px);
}

.quick-filter-btn.active {
    background: var(--primary-gradient);
    color: white;
    border-color: transparent;
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
}

/* Per Page Selector */
.per-page-selector {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-left: auto;
}

.per-page-selector label {
    font-size: 13px;
    color: var(--text-secondary);
    font-weight: 600;
    white-space: nowrap;
}

.per-page-selector select {
    padding: 8px 32px 8px 12px;
    border-radius: 8px;
    border: 2px solid var(--card-border);
    font-size: 13px;
    font-weight: 600;
    color: var(--text-secondary);
    background: var(--card-bg);
    cursor: pointer;
    transition: all 0.3s ease;
    appearance: none;
    background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 24 24' stroke='%234a5568'%3E%3Cpath stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M19 9l-7 7-7-7'%3E%3C/path%3E%3C/svg%3E");
    background-repeat: no-repeat;
    background-position: right 8px center;
    background-size: 16px;
}

.per-page-selector select:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

/* Table Card - IMPROVED */
.table-card {
    background: var(--card-bg);
    padding: 28px;
    border-radius: 14px;
    box-shadow: var(--card-shadow);
    animation: fadeInUp 0.5s ease-out 0.45s backwards;
    border: 1px solid var(--card-border);
}

.table-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding-bottom: 16px;
    border-bottom: 2px solid var(--card-border);
}

.table-title {
    font-size: 19px;
    font-weight: 700;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: 10px;
    letter-spacing: -0.01em;
}

.table-title svg {
    width: 20px;
    height: 20px;
    color: #667eea;
    animation: iconFloat 3s ease-in-out infinite;
}

@keyframes iconFloat {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-4px); }
}

.pagination-info {
    font-size: 13px;
    color: var(--text-muted);
    font-weight: 600;
}

.bulk-actions {
    display: flex;
    gap: 10px;
    align-items: center;
}

.bulk-actions-dropdown {
    display: none;
    gap: 10px;
}

.bulk-actions-dropdown.active {
    display: flex;
}

/* DataTable - IMPROVED */
.datatable-wrapper {
    overflow-x: auto;
}

table.dt-table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
}

table.dt-table thead th {
    background: var(--input-bg);
    padding: 14px 16px;
    text-align: left;
    font-weight: 600;
    font-size: 13px;
    color: var(--text-secondary);
    border-bottom: 2px solid var(--card-border);
    white-space: nowrap;
    cursor: pointer;
    transition: all 0.3s ease;
    letter-spacing: 0.02em;
}

table.dt-table thead th:hover {
    background: var(--card-bg);
    filter: brightness(0.95);
}

table.dt-table thead th.sortable::after {
    content: '⇅';
    margin-left: 6px;
    opacity: 0.3;
    font-size: 11px;
}

table.dt-table thead th.sorting-asc::after {
    content: '↑';
    opacity: 1;
}

table.dt-table thead th.sorting-desc::after {
    content: '↓';
    opacity: 1;
}

table.dt-table tbody tr {
    transition: all 0.3s ease;
    border-bottom: 1px solid var(--card-border);
}

table.dt-table tbody tr:hover {
    background: var(--input-bg);
    transform: scale(1.005);
}

table.dt-table tbody td {
    padding: 16px;
    vertical-align: middle;
    font-size: 14px;
    color: var(--text-primary);
}

/* Pinned Row Highlight */
tr.pinned-row {
    background: linear-gradient(to right, rgba(239, 68, 68, 0.05) 0%, transparent 100%);
}

tr.pinned-row:hover {
    background: linear-gradient(to right, rgba(239, 68, 68, 0.08) 0%, transparent 100%);
}

/* Project Title with Pin Icon */
.project-title-wrapper {
    display: flex;
    align-items: center;
    gap: 8px;
}

.pin-indicator {
    width: 16px;
    height: 16px;
    color: #ef4444;
    flex-shrink: 0;
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.05); }
}

.project-title-text {
    font-weight: 600;
    color: var(--text-primary);
}

/* Badges */
.badge {
    padding: 5px 11px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    display: inline-flex;
    align-items: center;
    gap: 5px;
    letter-spacing: 0.03em;
}

.badge-success {
    background: rgba(86, 171, 47, 0.12);
    color: #2d7a1f;
}

.badge-warning {
    background: rgba(242, 153, 74, 0.12);
    color: #8b5e00;
}

.badge-danger {
    background: rgba(239, 68, 68, 0.12);
    color: #c53030;
}

.badge-info {
    background: rgba(79, 172, 254, 0.12);
    color: #0c5d9f;
}

.badge-secondary {
    background: rgba(128, 128, 128, 0.12);
    color: var(--text-secondary);
}

/* Action Buttons - IMPROVED */
.action-buttons {
    display: flex;
    gap: 6px;
}

.btn-icon {
    width: 34px;
    height: 34px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-icon svg {
    width: 15px;
    height: 15px;
    transition: transform 0.3s ease;
}

.btn-icon:hover svg {
    transform: scale(1.15);
}

.btn-view {
    background: rgba(25, 118, 210, 0.1);
    color: #1976d2;
}

.btn-view:hover {
    background: #1976d2;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(25, 118, 210, 0.3);
}

.btn-edit {
    background: rgba(242, 153, 74, 0.1);
    color: #f2994a;
}

.btn-edit:hover {
    background: #f2994a;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(242, 153, 74, 0.3);
}

.btn-finance {
    background: rgba(46, 125, 50, 0.1);
    color: #2e7d32;
}

.btn-finance:hover {
    background: #2e7d32;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(46, 125, 50, 0.3);
}

.btn-pin {
    background: rgba(239, 68, 68, 0.1);
    color: #ef4444;
    transition: all 0.3s ease;
}

.btn-pin:hover {
    background: rgba(239, 68, 68, 0.2);
    transform: translateY(-2px) rotate(15deg);
}

.btn-pin.pinned {
    background: #ef4444;
    color: white;
}

.btn-pin.pinned:hover {
    transform: translateY(-2px) rotate(-15deg);
}

/* Pagination Controls */
.pagination-controls {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 12px;
    margin-top: 28px;
    padding-top: 28px;
    border-top: 2px solid var(--card-border);
    flex-wrap: wrap;
}

.pagination-btn {
    padding: 10px 18px;
    border-radius: 8px;
    font-size: 13px;
    font-weight: 600;
    border: 2px solid var(--card-border);
    background: var(--card-bg);
    color: var(--text-secondary);
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 6px;
}

.pagination-btn:hover:not(:disabled) {
    border-color: #667eea;
    color: #667eea;
    transform: translateY(-2px);
}

.pagination-btn:disabled {
    opacity: 0.4;
    cursor: not-allowed;
}

.pagination-btn.active {
    background: var(--primary-gradient);
    color: white;
    border-color: transparent;
    box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);
}

.pagination-dots {
    padding: 0 8px;
    color: var(--text-muted);
    font-weight: 600;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 64px 20px;
}

.empty-state svg {
    width: 80px;
    height: 80px;
    color: var(--text-muted);
    margin-bottom: 18px;
    animation: iconBounce 2s ease-in-out infinite;
}

@keyframes iconBounce {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-8px); }
}

.empty-state h3 {
    font-size: 19px;
    color: var(--text-secondary);
    margin-bottom: 10px;
    font-weight: 700;
    letter-spacing: -0.01em;
}

.empty-state p {
    color: var(--text-muted);
    margin-bottom: 24px;
    font-size: 14px;
    font-weight: 500;
}

/* Responsive */
@media (max-width: 992px) {
    .header-content {
        flex-direction: column;
        align-items: stretch;
    }
    
    .header-search {
        max-width: 100%;
    }
    
    .stats-bar {
        grid-template-columns: repeat(2, 1fr);
        gap: 10px;
    }
}

@media (max-width: 768px) {
    .stats-bar {
        grid-template-columns: repeat(2, 1fr);
        gap: 10px;
    }
    
    .page-header {
        padding: 16px 20px;
    }
    
    .page-header h1 {
        font-size: 18px;
    }
    
    .page-header h1 svg {
        width: 20px;
        height: 20px;
    }
    
    .page-header p {
        font-size: 11px;
    }
    
    .stat-content h3 {
        font-size: 20px;
    }
    
    .stat-content p {
        font-size: 10px;
    }
    
    .projects-container {
        padding: 16px;
    }
}

@media (max-width: 480px) {
    .stats-bar {
        grid-template-columns: 1fr;
    }
    
    .header-actions {
        flex-direction: column;
        width: 100%;
    }
    
    .btn {
        width: 100%;
        justify-content: center;
    }
}

/* ==================== COMPACT DELETE MODAL ==================== */
.delete-modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.6);
    backdrop-filter: blur(6px);
    z-index: 10000;
    align-items: center;
    justify-content: center;
    animation: fadeIn 0.2s ease;
}

.delete-modal.active {
    display: flex;
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

.delete-modal-content {
    background: var(--card-bg);
    border-radius: 16px;
    padding: 32px;
    max-width: 420px;
    width: 90%;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    animation: slideUp 0.3s cubic-bezier(0.34, 1.56, 0.64, 1);
}

@keyframes slideUp {
    from {
        opacity: 0;
        transform: translateY(30px) scale(0.95);
    }
    to {
        opacity: 1;
        transform: translateY(0) scale(1);
    }
}

.delete-icon-wrapper {
    width: 70px;
    height: 70px;
    margin: 0 auto 20px;
    position: relative;
    display: flex;
    align-items: center;
    justify-content: center;
}

.delete-warning-icon {
    width: 70px;
    height: 70px;
    color: #ef4444;
    animation: shake 0.5s ease-in-out;
}

@keyframes shake {
    0%, 100% { transform: translateX(0); }
    10%, 30%, 50%, 70%, 90% { transform: translateX(-8px); }
    20%, 40%, 60%, 80% { transform: translateX(8px); }
}

.delete-circle-bg {
    position: absolute;
    width: 90px;
    height: 90px;
    background: rgba(239, 68, 68, 0.15);
    border-radius: 50%;
    animation: pulse 2s ease-in-out infinite;
    z-index: -1;
}

@keyframes pulse {
    0%, 100% { transform: scale(1); opacity: 0.5; }
    50% { transform: scale(1.2); opacity: 0.3; }
}

.delete-modal-content h2 {
    font-size: 22px;
    font-weight: 700;
    color: var(--text-primary);
    margin-bottom: 12px;
    text-align: center;
}

.delete-project-name {
    background: rgba(239, 68, 68, 0.1);
    padding: 10px 16px;
    border-radius: 8px;
    margin-bottom: 16px;
    text-align: center;
}

.delete-project-name strong {
    font-size: 15px;
    color: #ef4444;
    font-weight: 700;
}

.delete-warning-box {
    background: var(--input-bg);
    border-left: 3px solid #ef4444;
    border-radius: 8px;
    padding: 12px 16px;
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 12px;
}

.delete-warning-box svg {
    width: 20px;
    height: 20px;
    color: #ef4444;
    flex-shrink: 0;
}

.delete-warning-box p {
    color: var(--text-secondary);
    font-size: 13px;
    margin: 0;
    line-height: 1.5;
}

.delete-modal-actions {
    display: flex;
    gap: 10px;
    margin-top: 20px;
}

.delete-modal-actions button {
    flex: 1;
    padding: 12px 20px;
    border-radius: 10px;
    font-weight: 700;
    font-size: 14px;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.btn-cancel-delete {
    background: var(--input-bg);
    color: var(--text-secondary);
    border: 2px solid var(--card-border);
}

.btn-cancel-delete:hover {
    background: var(--card-bg);
    border-color: #667eea;
    color: #667eea;
    transform: translateY(-2px);
}

.btn-confirm-delete {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(239, 68, 68, 0.3);
}

.btn-confirm-delete:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(239, 68, 68, 0.4);
}

.btn-confirm-delete svg,
.btn-cancel-delete svg {
    width: 16px;
    height: 16px;
}

/* ==================== DELETING PROGRESS MODAL ==================== */
.deleting-modal-content {
    background: var(--card-bg);
    border-radius: 16px;
    padding: 40px;
    max-width: 320px;
    width: 90%;
    text-align: center;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    animation: slideUp 0.3s cubic-bezier(0.34, 1.56, 0.64, 1);
}

.deleting-spinner-wrapper {
    width: 70px;
    height: 70px;
    margin: 0 auto 20px;
    position: relative;
}

.deleting-spinner {
    width: 70px;
    height: 70px;
    border: 5px solid rgba(239, 68, 68, 0.2);
    border-top: 5px solid #ef4444;
    border-radius: 50%;
    animation: spin 0.8s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.deleting-modal-content h3 {
    font-size: 18px;
    font-weight: 700;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.deleting-modal-content p {
    color: var(--text-secondary);
    font-size: 13px;
}

/* Delete Button Style */
.btn-delete {
    background: rgba(239, 68, 68, 0.1);
    color: #ef4444;
}

.btn-delete:hover {
    background: #ef4444;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
}
</style>

<!-- ==================== COMPACT DELETE CONFIRMATION MODAL ==================== -->
<div class="delete-modal" id="deleteModal">
    <div class="delete-modal-content">
        <div class="delete-icon-wrapper">
            <div class="delete-circle-bg"></div>
            <svg class="delete-warning-icon" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
            </svg>
        </div>
        
        <h2>Delete Project?</h2>
        
        <div class="delete-project-name">
            <strong id="deleteProjectTitle"></strong>
        </div>
        
        <div class="delete-warning-box">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
            </svg>
            <p>This will permanently delete all tasks, time logs, files, discussions, and team assignments. This action cannot be undone.</p>
        </div>
        
        <div class="delete-modal-actions">
            <button class="btn-cancel-delete" onclick="closeDeleteModal()">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12"/>
                </svg>
                Cancel
            </button>
            <button class="btn-confirm-delete" onclick="confirmDeleteProject()">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                </svg>
                Delete
            </button>
        </div>
    </div>
</div>

<!-- ==================== DELETING PROGRESS MODAL ==================== -->
<div class="delete-modal" id="deletingModal">
    <div class="deleting-modal-content">
        <div class="deleting-spinner-wrapper">
            <div class="deleting-spinner"></div>
        </div>
        <h3>Deleting Project...</h3>
        <p>Please wait</p>
    </div>
</div>

<div class="projects-container">
    <!-- Page Header with Integrated Search -->
    <div class="page-header">
        <div class="header-content">
            <div>
                <h1>
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
                        <polyline points="9 22 9 12 15 12 15 22"/>
                    </svg>
                    Projects Management
                </h1>
                <p>Manage and track all your projects</p>
            </div>
            
            <div class="header-search">
                <div class="search-input-wrapper">
                    <input type="text" id="searchInput" class="search-input" placeholder="Search projects...">
                    <svg class="search-icon" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                    </svg>
                </div>
            </div>
            
            <div class="header-actions">
                <a href="{{ route('admin.projects.create') }}" class="btn btn-primary">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 4v16m8-8H4"/>
                    </svg>
                    Create Project
                </a>
            </div>
        </div>
    </div>

    <!-- Stats Bar -->
    <div class="stats-bar">
        <div class="stat-card">
            <div class="stat-icon">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="totalProjects">{{ $stats['total'] ?? 0 }}</h3>
                <p>Total Projects</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="background: var(--success-gradient);">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="activeProjects">{{ $stats['active'] ?? 0 }}</h3>
                <p>Active Projects</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="background: var(--danger-gradient);">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <line x1="12" y1="17" x2="12" y2="22"/>
                    <path d="M5 17h14v-1.76a2 2 0 0 0-1.11-1.79l-1.78-.9A2 2 0 0 1 15 10.76V6h1a2 2 0 0 0 0-4H8a2 2 0 0 0 0 4h1v4.76a2 2 0 0 1-1.11 1.79l-1.78.9A2 2 0 0 0 5 15.24Z"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="pinnedProjects">{{ $stats['pinned'] ?? 0 }}</h3>
                <p>Pinned Projects</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="background: var(--warning-gradient);">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/>
                    <circle cx="12" cy="7" r="4"/>
                </svg>
            </div>
            <div class="stat-content">
                <h3 id="myProjects">{{ $stats['my_projects'] ?? 0 }}</h3>
                <p>My Projects</p>
            </div>
        </div>
    </div>

    <!-- Filters Bar -->
    <div class="filters-bar">
        <div class="filters-row">
            <div class="quick-filters">
                <button class="quick-filter-btn active" onclick="applyQuickFilter('all')">All</button>
                <button class="quick-filter-btn" onclick="applyQuickFilter('active')">Active</button>
                <button class="quick-filter-btn" onclick="applyQuickFilter('pinned')">Pinned</button>
                <button class="quick-filter-btn" onclick="applyQuickFilter('my')">My Projects</button>
            </div>

            <div class="per-page-selector">
                <label>Show:</label>
                <select id="perPageSelect" onchange="changePerPage(this.value)">
                    <option value="15" selected>15</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
            </div>
        </div>
    </div>

    <!-- Projects Table -->
    <div class="table-card">
        <div class="table-header">
            <div>
                <h3 class="table-title">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
                    </svg>
                    All Projects
                </h3>
                <div class="pagination-info" id="paginationInfo">Showing 0-0 of 0 projects</div>
            </div>
            <div class="bulk-actions">
                <div class="bulk-actions-dropdown" id="bulkActionsDropdown">
                    <button class="btn btn-sm btn-danger" onclick="bulkDelete()">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                        </svg>
                        Delete Selected
                    </button>
                </div>
                {{-- <button class="btn btn-sm btn-success" onclick="exportProjects()">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
                    </svg>
                    Export CSV
                </button> --}}
            </div>
        </div>

        <div class="datatable-wrapper">
            <table class="dt-table" id="projectsTable">
                <thead>
                    <tr>
                        <th style="width: 40px;">
                            <input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)">
                        </th>
                        <th class="sortable" data-column="id">ID</th>
                        <th class="sortable" data-column="title">Project Title</th>
                        <th class="sortable" data-column="account_display">Customer</th>
                        <th class="sortable" data-column="status_display">Status</th>
                        <th class="sortable" data-column="billing_method">Billing</th>
                        <th class="sortable" data-column="planned_end_date">Target Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody id="projectsTableBody">
                    <!-- Data populated via AJAX -->
                </tbody>
            </table>
        </div>

        <!-- Pagination Controls -->
        <div class="pagination-controls" id="paginationControls" style="display: none;">
            <!-- Populated dynamically -->
        </div>
    </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
// Global state
let currentPage = 1;
let perPage = 15;
let totalRecords = 0;
let lastPage = 1;
let sortColumn = 'created_at';
let sortDirection = 'desc';
let selectedIds = [];
let currentFilter = 'all';

$(document).ready(function() {
    loadProjects();

    // Search with debounce
    let searchTimeout;
    $('#searchInput').on('keyup', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(() => {
            currentPage = 1;
            loadProjects();
        }, 500);
    });

    // Column sorting
    $('.sortable').on('click', function() {
        const column = $(this).data('column');
        
        if (sortColumn === column) {
            sortDirection = sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            sortColumn = column;
            sortDirection = 'asc';
        }

        $('.sortable').removeClass('sorting-asc sorting-desc');
        $(this).addClass('sorting-' + sortDirection);

        loadProjects();
    });
});

function loadProjects() {
    const filters = {
        search: $('#searchInput').val(),
        filter_type: currentFilter,
        page: currentPage,
        per_page: perPage,
        sort: sortColumn,
        dir: sortDirection
    };

    $.ajax({
        url: '{{ route("admin.projects.data") }}',
        type: 'GET',
        data: filters,
        success: function(response) {
            renderProjects(response.data);
            updatePagination(response);
        },
        error: function(xhr) {
            alert('Error loading projects: ' + (xhr.responseJSON?.message || 'Unknown error'));
        }
    });
}

function renderProjects(projects) {
    const tbody = $('#projectsTableBody');
    tbody.empty();

    if (projects.length === 0) {
        tbody.append(`
            <tr>
                <td colspan="8">
                    <div class="empty-state">
                        <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
                            <polyline points="9 22 9 12 15 12 15 22"/>
                        </svg>
                        <h3>No Projects Found</h3>
                        <p>No projects match your search criteria. Try adjusting your filters.</p>
                    </div>
                </td>
            </tr>
        `);
        return;
    }

    projects.forEach(project => {
        const isPinned = project.is_pinned || false;
        const rowClass = isPinned ? 'pinned-row' : '';
        
        const titleHtml = isPinned ? `
            <div class="project-title-wrapper">
                <svg class="pin-indicator" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
                    <path d="M16 12V4H17V2H7V4H8V12L6 14V16H11.2V22H12.8V16H18V14L16 12Z"/>
                </svg>
                <span class="project-title-text">${project.title}</span>
            </div>
        ` : `<span class="project-title-text">${project.title}</span>`;

        const row = `
            <tr class="${rowClass}">
                <td>
                    <input type="checkbox" class="project-checkbox" value="${project.id}" onchange="toggleSelection(this)">
                </td>
                <td><strong>#${project.id}</strong></td>
                <td>${titleHtml}</td>
                <td>${project.account_display || '-'}</td>
                <td>
                    <span class="badge badge-info">${project.status_display}</span>
                </td>
                <td>${project.billing_method_display}</td>
                <td>${project.planned_end_date ? new Date(project.planned_end_date).toLocaleDateString('en-US', {month: 'short', day: 'numeric', year: 'numeric'}) : '-'}</td>
                <td>
                    <div class="action-buttons">
                        <a href="${project._show_url}" class="btn-icon btn-view" title="View Project">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                                <path stroke-linecap="round" stroke-linejoin="round" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                            </svg>
                        </a>
                        <a href="${project._edit_url}" class="btn-icon btn-edit" title="Edit">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
                            </svg>
                        </a>
                        <a href="${project._finance_url}" class="btn-icon btn-finance" title="Finance">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        </a>
                        <button class="btn-icon btn-pin ${isPinned ? 'pinned' : ''}" 
                                onclick="togglePin(${project.id}, this)" 
                                title="${isPinned ? 'Unpin' : 'Pin'}">
                            <svg fill="${isPinned ? 'currentColor' : 'none'}" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <line x1="12" y1="17" x2="12" y2="22"/>
                                <path d="M5 17h14v-1.76a2 2 0 0 0-1.11-1.79l-1.78-.9A2 2 0 0 1 15 10.76V6h1a2 2 0 0 0 0-4H8a2 2 0 0 0 0 4h1v4.76a2 2 0 0 1-1.11 1.79l-1.78.9A2 2 0 0 0 5 15.24Z"/>
                            </svg>
                        </button>
                        <button class="btn-icon btn-delete" 
                                onclick="openDeleteModal(${project.id}, '${project.title.replace(/'/g, "\\'")}', event)" 
                                title="Delete Project">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                            </svg>
                        </button>
                    </div>
                </td>
            </tr>
        `;
        tbody.append(row);
    });
}

function updatePagination(response) {
    totalRecords = response.total || 0;
    currentPage = response.current_page || 1;
    lastPage = response.last_page || 1;

    // Update pagination info
    const startRecord = totalRecords === 0 ? 0 : ((currentPage - 1) * perPage) + 1;
    const endRecord = Math.min(currentPage * perPage, totalRecords);
    $('#paginationInfo').text(`Showing ${startRecord}-${endRecord} of ${totalRecords} projects`);

    // Render pagination controls
    const paginationContainer = $('#paginationControls');
    
    if (lastPage <= 1) {
        paginationContainer.hide();
        return;
    }

    paginationContainer.show();
    paginationContainer.empty();

    let html = '';

    // Previous button
    html += `
        <button class="pagination-btn" onclick="goToPage(${currentPage - 1})" ${currentPage === 1 ? 'disabled' : ''}>
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 16px; height: 16px;">
                <path stroke-linecap="round" stroke-linejoin="round" d="M15 19l-7-7 7-7"/>
            </svg>
            Previous
        </button>
    `;

    // Page numbers
    const maxPagesToShow = 7;
    let startPage = Math.max(1, currentPage - Math.floor(maxPagesToShow / 2));
    let endPage = Math.min(lastPage, startPage + maxPagesToShow - 1);

    if (endPage - startPage < maxPagesToShow - 1) {
        startPage = Math.max(1, endPage - maxPagesToShow + 1);
    }

    // First page
    if (startPage > 1) {
        html += `<button class="pagination-btn" onclick="goToPage(1)">1</button>`;
        if (startPage > 2) {
            html += `<span class="pagination-dots">...</span>`;
        }
    }

    // Page numbers
    for (let i = startPage; i <= endPage; i++) {
        html += `
            <button class="pagination-btn ${i === currentPage ? 'active' : ''}" onclick="goToPage(${i})">
                ${i}
            </button>
        `;
    }

    // Last page
    if (endPage < lastPage) {
        if (endPage < lastPage - 1) {
            html += `<span class="pagination-dots">...</span>`;
        }
        html += `<button class="pagination-btn" onclick="goToPage(${lastPage})">${lastPage}</button>`;
    }

    // Next button
    html += `
        <button class="pagination-btn" onclick="goToPage(${currentPage + 1})" ${currentPage === lastPage ? 'disabled' : ''}>
            Next
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24" style="width: 16px; height: 16px;">
                <path stroke-linecap="round" stroke-linejoin="round" d="M9 5l7 7-7 7"/>
            </svg>
        </button>
    `;

    paginationContainer.html(html);
}

function goToPage(page) {
    if (page < 1 || page > lastPage) return;
    currentPage = page;
    loadProjects();
    
    // Scroll to top of table
    $('html, body').animate({
        scrollTop: $('.table-card').offset().top - 100
    }, 300);
}

function changePerPage(value) {
    perPage = parseInt(value);
    currentPage = 1;
    loadProjects();
}

function applyQuickFilter(type) {
    $('.quick-filter-btn').removeClass('active');
    event.target.classList.add('active');
    
    currentFilter = type;
    currentPage = 1;
    loadProjects();
}

function toggleSelection(checkbox) {
    const id = parseInt(checkbox.value);
    if (checkbox.checked) {
        selectedIds.push(id);
    } else {
        selectedIds = selectedIds.filter(item => item !== id);
    }
    $('#bulkActionsDropdown').toggleClass('active', selectedIds.length > 0);
}

function toggleSelectAll(checkbox) {
    $('.project-checkbox').prop('checked', checkbox.checked);
    selectedIds = checkbox.checked ? 
        $('.project-checkbox').map(function() { return parseInt(this.value); }).get() : [];
    $('#bulkActionsDropdown').toggleClass('active', selectedIds.length > 0);
}

function togglePin(projectId, button) {
    const isPinned = button.classList.contains('pinned');
    
    // Add loading state
    button.style.opacity = '0.5';
    button.style.pointerEvents = 'none';
    
    $.ajax({
        url: `/admin/projects/${projectId}/pin`,
        type: 'POST',
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        success: function(response) {
            if (response.success) {
                // Update button state
                if (response.is_pinned) {
                    button.classList.add('pinned');
                    button.title = 'Unpin Project';
                    button.querySelector('svg').setAttribute('fill', 'currentColor');
                } else {
                    button.classList.remove('pinned');
                    button.title = 'Pin Project';
                    button.querySelector('svg').setAttribute('fill', 'none');
                }
                
                // Reload table to resort (pinned projects go to top)
                loadProjects();
            }
        },
        error: function(xhr) {
            alert('Error: ' + (xhr.responseJSON?.message || 'Failed to toggle pin'));
        },
        complete: function() {
            button.style.opacity = '1';
            button.style.pointerEvents = 'auto';
        }
    });
}

function bulkDelete() {
    if (selectedIds.length === 0) {
        alert('Please select projects to delete');
        return;
    }

    if (!confirm(`Are you sure you want to delete ${selectedIds.length} project(s)?`)) {
        return;
    }

    $.ajax({
        url: '{{ route("admin.projects.bulk-delete") }}',
        type: 'POST',
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        data: {
            ids: selectedIds
        },
        success: function(response) {
            if (response.success) {
                alert(response.message);
                selectedIds = [];
                $('#selectAll').prop('checked', false);
                $('#bulkActionsDropdown').removeClass('active');
                loadProjects();
            } else {
                alert('Error: ' + response.message);
            }
        },
        error: function(xhr) {
            alert('Error: ' + (xhr.responseJSON?.message || 'Failed to delete projects'));
        }
    });
}

// ✅ ADD THIS: Function to update stats via AJAX
function updateStats() {
    $.ajax({
        url: '{{ route("admin.projects.stats") }}',
        type: 'GET',
        success: function(stats) {
            $('#totalProjects').text(stats.total);
            $('#activeProjects').text(stats.active);
            $('#pinnedProjects').text(stats.pinned);
            $('#myProjects').text(stats.my_projects);
        },
        error: function() {
            console.error('Failed to update stats');
        }
    });
}

// ✅ MODIFY: Update togglePin to refresh stats
function togglePin(projectId, button) {
    const isPinned = button.classList.contains('pinned');
    
    button.style.opacity = '0.5';
    button.style.pointerEvents = 'none';
    
    $.ajax({
        url: `/admin/projects/${projectId}/pin`,
        type: 'POST',
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        success: function(response) {
            if (response.success) {
                // Update button state
                if (response.is_pinned) {
                    button.classList.add('pinned');
                    button.title = 'Unpin Project';
                    button.querySelector('svg').setAttribute('fill', 'currentColor');
                } else {
                    button.classList.remove('pinned');
                    button.title = 'Pin Project';
                    button.querySelector('svg').setAttribute('fill', 'none');
                }
                
                // ✅ UPDATE STATS WITHOUT RELOAD
                updateStats();
                
                // Reload table to resort
                loadProjects();
            }
        },
        error: function(xhr) {
            alert('Error: ' + (xhr.responseJSON?.message || 'Failed to toggle pin'));
        },
        complete: function() {
            button.style.opacity = '1';
            button.style.pointerEvents = 'auto';
        }
    });
}

function exportProjects() {
    const search = $('#searchInput').val();
    const queryString = search ? `?search=${encodeURIComponent(search)}` : '';
    window.location.href = `{{ route('admin.projects.data') }}${queryString}&export=1`;
}


// ==================== DELETE PROJECT FUNCTIONS ====================
let projectToDelete = null;

function openDeleteModal(projectId, projectTitle, event) {
    if (event) {
        event.preventDefault();
        event.stopPropagation();
    }
    
    projectToDelete = projectId;
    $('#deleteProjectTitle').text(projectTitle);
    $('#deleteModal').addClass('active');
    
    // Prevent body scroll
    $('body').css('overflow', 'hidden');
}

function closeDeleteModal() {
    $('#deleteModal').removeClass('active');
    projectToDelete = null;
    
    // Restore body scroll
    $('body').css('overflow', 'auto');
}

function confirmDeleteProject() {
    if (!projectToDelete) {
        alert('No project selected for deletion');
        return;
    }
    
    // Hide confirmation modal
    $('#deleteModal').removeClass('active');
    
    // Show deleting progress modal
    $('#deletingModal').addClass('active');
    
    // Make DELETE request
    $.ajax({
        url: `/admin/projects/${projectToDelete}`,
        type: 'DELETE',
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        success: function(response) {
            // Hide deleting modal
            $('#deletingModal').removeClass('active');
            $('body').css('overflow', 'auto');
            
            if (response.success) {
                // Show success notification (you can customize this)
                showSuccessNotification(response.message, response.stats);
                
                // Update stats without reload
                updateStats();
                
                // Reload projects table
                loadProjects();
                
                // Reset selected IDs if project was selected
                selectedIds = selectedIds.filter(id => id !== projectToDelete);
                if (selectedIds.length === 0) {
                    $('#selectAll').prop('checked', false);
                    $('#bulkActionsDropdown').removeClass('active');
                }
            } else {
                alert('Error: ' + response.message);
            }
            
            projectToDelete = null;
        },
        error: function(xhr) {
            // Hide deleting modal
            $('#deletingModal').removeClass('active');
            $('body').css('overflow', 'auto');
            
            console.error('Delete error:', xhr);
            
            const errorMsg = xhr.responseJSON?.message || 'Failed to delete project. Please try again.';
            alert('Error: ' + errorMsg);
            
            projectToDelete = null;
        }
    });
}

// Success notification function
function showSuccessNotification(message, stats) {
    // Create notification element
    const notification = $(`
        <div class="success-notification" style="
            position: fixed;
            top: 20px;
            right: 20px;
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
            color: white;
            padding: 20px 30px;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(16, 185, 129, 0.4);
            z-index: 10001;
            animation: slideInRight 0.5s ease;
            max-width: 400px;
        ">
            <div style="display: flex; align-items: center; gap: 15px;">
                <svg style="width: 24px; height: 24px; flex-shrink: 0;" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>
                <div>
                    <div style="font-weight: 700; font-size: 16px; margin-bottom: 4px;">Project Deleted!</div>
                    <div style="font-size: 13px; opacity: 0.9;">${message}</div>
                    ${stats ? `
                        <div style="font-size: 12px; margin-top: 8px; opacity: 0.8;">
                            Removed: ${stats.tasks || 0} tasks, ${stats.time_logs || 0} time logs, 
                            ${stats.milestones || 0} milestones, ${stats.discussions || 0} discussions
                        </div>
                    ` : ''}
                </div>
            </div>
        </div>
    `);
    
    $('body').append(notification);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        notification.fadeOut(300, function() {
            $(this).remove();
        });
    }, 5000);
}

// Close modals on ESC key
$(document).on('keydown', function(e) {
    if (e.key === 'Escape') {
        if ($('#deleteModal').hasClass('active')) {
            closeDeleteModal();
        }
    }
});

// Close modal on backdrop click
$('#deleteModal').on('click', function(e) {
    if (e.target === this) {
        closeDeleteModal();
    }
});

</script>