<?php

namespace Modules\Purchase\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Purchase\Models\GoodsReceiptNote;
use Modules\Purchase\Models\GoodsReceiptNoteItem;
use Modules\Purchase\Models\PurchaseOrder;
use Modules\Purchase\Models\PurchaseSetting;
use App\Traits\DataTable;

class GoodsReceiptNoteController extends AdminController
{
    use DataTable;

    public function __construct()
    {
        // Check if Inventory module is available
        if (!class_exists('\Modules\Inventory\Models\Warehouse')) {
            abort(503, 'GRN functionality requires the Inventory module to be installed.');
        }
    }
    
    // DataTable Configuration
    protected $model = GoodsReceiptNote::class;
    protected $with = ['vendor', 'purchaseOrder', 'warehouse', 'creator'];
    protected $searchable = ['grn_number', 'invoice_number', 'vendor.name', 'purchaseOrder.po_number'];
    protected $sortable = ['id', 'grn_number', 'grn_date', 'status', 'created_at'];
    protected $filterable = ['status', 'warehouse_id', 'vendor_id'];
    protected $exportTitle = 'Goods Receipt Notes Export';

    // Permission Configuration
    protected $viewPermission = 'purchase.grn.read';
    protected $createPermission = 'purchase.grn.create';
    protected $editPermission = 'purchase.grn.edit';
    protected $deletePermission = 'purchase.grn.delete';
    protected $approvePermission = 'purchase.grn.approve';

    /**
     * Display GRN listing
     */
    public function index()
    {
        $this->authorize('purchase.grn.read');
        
        $stats = [
            'total' => GoodsReceiptNote::count(),
            'draft' => GoodsReceiptNote::where('status', 'DRAFT')->count(),
            'inspecting' => GoodsReceiptNote::where('status', 'INSPECTING')->count(),
            'approved' => GoodsReceiptNote::where('status', 'APPROVED')->count(),
        ];

        return view('purchase::grn.index', compact('stats'));
    }

    /**
     * DataTable row mapping for list view
     */
    protected function mapRow($item)
    {
        $row = [
            'id' => $item->id,
            'grn_number' => $item->grn_number,
            'grn_date' => $item->grn_date->format('d M Y'),
            'po_number' => $item->purchaseOrder->po_number ?? '-',
            'po_id' => $item->purchase_order_id,
            'vendor_name' => $item->vendor->name ?? '-',
            'vendor_id' => $item->vendor_id,
            'warehouse_name' => $item->warehouse->name ?? '-',
            'invoice_number' => $item->invoice_number ?? '-',
            'total_qty' => number_format($item->total_qty, 2),
            'accepted_qty' => number_format($item->accepted_qty, 2),
            'rejected_qty' => number_format($item->rejected_qty, 2),
            'status' => $item->status,
            'status_badge' => $item->status_badge ?? $item->status,
            'stock_updated' => $item->stock_updated,
            'created_by' => $item->creator->name ?? '-',
            'created_at' => $item->created_at->format('d M Y'),
            '_show_url' => route('admin.purchase.grn.show', $item->id),
            '_edit_url' => route('admin.purchase.grn.edit', $item->id),
        ];
        
        // Quick action URLs for list view
        if (in_array($item->status, ['DRAFT', 'INSPECTING'])) {
            $row['_approve_url'] = route('admin.purchase.grn.approve', $item->id);
            $row['_reject_url'] = route('admin.purchase.grn.reject', $item->id);
        }
        if ($item->status === 'APPROVED') {
            $row['_bill_url'] = route('admin.purchase.bills.create') . '?grn_id=' . $item->id;
        }
        
        return $row;
    }

    /**
     * DataTable row mapping for export
     */
    protected function mapExportRow($item)
    {
        return [
            'ID' => $item->id,
            'GRN Number' => $item->grn_number,
            'Date' => $item->grn_date->format('Y-m-d'),
            'PO Number' => $item->purchaseOrder->po_number ?? '',
            'Vendor' => $item->vendor->name ?? '',
            'Warehouse' => $item->warehouse->name ?? '',
            'Invoice No' => $item->invoice_number ?? '',
            'Total Qty' => $item->total_qty,
            'Accepted Qty' => $item->accepted_qty,
            'Rejected Qty' => $item->rejected_qty,
            'Stock Updated' => $item->stock_updated ? 'Yes' : 'No',
            'Status' => $item->status,
        ];
    }

    /**
     * DataTable endpoint
     */
    public function dataTable(Request $request)
    {
        return $this->handleData($request);
    }

    /**
     * Show create form - Select PO first
     */
    public function create(Request $request)
    {
        $this->authorize('purchase.grn.create');
        
        // Get confirmed POs that are not fully received
        $purchaseOrders = PurchaseOrder::with('vendor')
            ->whereIn('status', ['CONFIRMED', 'PARTIALLY_RECEIVED'])
            ->orderBy('po_date', 'desc')
            ->get();

        $warehouses = \Modules\Inventory\Models\Warehouse::where('is_active', true)->orderBy('name')->get();
        
        $selectedPO = null;
        if ($request->has('po_id')) {
            $selectedPO = PurchaseOrder::with(['vendor', 'items.product', 'items.unit'])
                ->find($request->po_id);
        }

        return view('purchase::grn.create', compact('purchaseOrders', 'warehouses', 'selectedPO'));
    }

    /**
     * Store new GRN
     */
    public function store(Request $request)
    {
        $this->authorize('purchase.grn.create');
        
        $validated = $request->validate([
            'purchase_order_id' => 'required|exists:purchase_orders,id',
            'grn_date' => 'required|date',
            'warehouse_id' => 'required|exists:warehouses,id',
            'rack_id' => 'nullable|exists:racks,id',
            'invoice_number' => 'nullable|string|max:100',
            'invoice_date' => 'nullable|date',
            'lr_number' => 'nullable|string|max:100',
            'vehicle_number' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.po_item_id' => 'required|exists:purchase_order_items,id',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.received_qty' => 'required|numeric|min:0',
            'items.*.accepted_qty' => 'required|numeric|min:0',
            'items.*.rejected_qty' => 'nullable|numeric|min:0',
            'items.*.lot_no' => 'nullable|string|max:100',
            'items.*.batch_no' => 'nullable|string|max:100',
            'items.*.manufacturing_date' => 'nullable|date',
            'items.*.expiry_date' => 'nullable|date',
        ]);

        $po = PurchaseOrder::with('vendor')->findOrFail($validated['purchase_order_id']);

        DB::beginTransaction();
        try {
            // Create GRN
            $grn = GoodsReceiptNote::create([
                'grn_number' => GoodsReceiptNote::generateGrnNumber(),
                'purchase_order_id' => $po->id,
                'vendor_id' => $po->vendor_id,
                'grn_date' => $validated['grn_date'],
                'warehouse_id' => $validated['warehouse_id'],
                'rack_id' => $validated['rack_id'] ?? null,
                'invoice_number' => $validated['invoice_number'] ?? null,
                'invoice_date' => $validated['invoice_date'] ?? null,
                'lr_number' => $validated['lr_number'] ?? null,
                'vehicle_number' => $validated['vehicle_number'] ?? null,
                'notes' => $validated['notes'] ?? null,
                'status' => 'DRAFT',
                'received_by' => auth()->id(),
                'created_by' => auth()->id(),
            ]);

            // Create GRN items
            foreach ($validated['items'] as $itemData) {
                if (($itemData['received_qty'] ?? 0) <= 0) continue;

                $poItem = $po->items()->find($itemData['po_item_id']);
                if (!$poItem) continue;

                // Fix: empty string should use fallback, not just null
                $variationId = !empty($itemData['variation_id']) ? $itemData['variation_id'] : $poItem->variation_id;

                $grn->items()->create([
                    'purchase_order_item_id' => $poItem->id,
                    'product_id' => $itemData['product_id'],
                    'variation_id' => $variationId,
                    'unit_id' => $itemData['unit_id'] ?? $poItem->unit_id,
                    'ordered_qty' => $poItem->qty,
                    'received_qty' => $itemData['received_qty'],
                    'accepted_qty' => $itemData['accepted_qty'],
                    'rejected_qty' => $itemData['rejected_qty'] ?? 0,
                    'rate' => $itemData['rate'] ?? $poItem->rate,
                    'discount_percent' => $poItem->discount_percent ?? 0,
                    // Inherit taxes from PO item
                    'tax_1_id' => $poItem->tax_1_id,
                    'tax_1_name' => $poItem->tax_1_name,
                    'tax_1_rate' => $poItem->tax_1_rate ?? 0,
                    'tax_2_id' => $poItem->tax_2_id,
                    'tax_2_name' => $poItem->tax_2_name,
                    'tax_2_rate' => $poItem->tax_2_rate ?? 0,
                    'lot_no' => $itemData['lot_no'] ?? null,
                    'batch_no' => $itemData['batch_no'] ?? null,
                    'manufacturing_date' => $itemData['manufacturing_date'] ?? null,
                    'expiry_date' => $itemData['expiry_date'] ?? null,
                ]);
            }

            $grn->calculateTotals();

            DB::commit();

            return redirect()
                ->route('admin.purchase.grn.show', $grn->id)
                ->with('success', "GRN {$grn->grn_number} created successfully!");

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error creating GRN: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Show GRN details
     */
    public function show(Request $request, $id)
    {
        $this->authorize('purchase.grn.read');
        
        $grn = GoodsReceiptNote::with([
            'vendor',
            'purchaseOrder',
            'warehouse',
            'rack',
            'items.product',
            'items.variation',
            'items.unit',
            'items.lot',
            'items.purchaseOrderItem',
            'creator',
            'receiver',
            'approver',
        ])->findOrFail($id);

        // Return JSON for AJAX requests
        if ($request->wantsJson() || $request->input('format') === 'json') {
            return response()->json([
                'id' => $grn->id,
                'grn_number' => $grn->grn_number,
                'vendor_id' => $grn->vendor_id,
                'vendor' => $grn->vendor,
                'warehouse_id' => $grn->warehouse_id,
                'items' => $grn->items->map(function($item) {
                    // Get tax - first from GRN item, then fallback to PO item
                    $tax1Id = $item->tax_1_id;
                    $tax1Name = $item->tax_1_name;
                    $tax1Rate = $item->tax_1_rate ?? 0;
                    $tax2Id = $item->tax_2_id;
                    $tax2Name = $item->tax_2_name;
                    $tax2Rate = $item->tax_2_rate ?? 0;
                    
                    // Fallback to PO item if GRN doesn't have tax
                    if (empty($tax1Id) && empty($tax1Rate) && $item->purchaseOrderItem) {
                        $poItem = $item->purchaseOrderItem;
                        $tax1Id = $poItem->tax_1_id;
                        $tax1Name = $poItem->tax_1_name;
                        $tax1Rate = $poItem->tax_1_rate ?? 0;
                        $tax2Id = $poItem->tax_2_id;
                        $tax2Name = $poItem->tax_2_name;
                        $tax2Rate = $poItem->tax_2_rate ?? 0;
                    }
                    
                    return [
                        'id' => $item->id,
                        'product_id' => $item->product_id,
                        'product' => $item->product,
                        'unit_id' => $item->unit_id,
                        'unit' => $item->unit,
                        'variation_id' => $item->variation_id,
                        'ordered_qty' => $item->ordered_qty,
                        'received_qty' => $item->received_qty,
                        'accepted_qty' => $item->accepted_qty,
                        'rejected_qty' => $item->rejected_qty,
                        'rate' => $item->rate,
                        'discount_percent' => $item->discount_percent ?? 0,
                        // Tax fields
                        'tax_1_id' => $tax1Id,
                        'tax_1_name' => $tax1Name,
                        'tax_1_rate' => $tax1Rate,
                        'tax_2_id' => $tax2Id,
                        'tax_2_name' => $tax2Name,
                        'tax_2_rate' => $tax2Rate,
                    ];
                }),
            ]);
        }

        return view('purchase::grn.show', compact('grn'));
    }

    /**
     * Edit GRN (only Draft/Inspecting)
     */
    public function edit($id)
    {
        $this->authorize('purchase.grn.edit');
        
        $grn = GoodsReceiptNote::with([
            'items.product',
            'items.unit',
            'purchaseOrder.items',
        ])->findOrFail($id);

        if (!$grn->can_edit) {
            return redirect()
                ->route('admin.purchase.grn.show', $grn->id)
                ->with('error', 'Cannot edit GRN in current status.');
        }

        $warehouses = \Modules\Inventory\Models\Warehouse::where('is_active', true)->orderBy('name')->get();
        $racks = $grn->warehouse_id 
            ? \Modules\Inventory\Models\Rack::where('warehouse_id', $grn->warehouse_id)->where('is_active', true)->get() 
            : collect();

        return view('purchase::grn.edit', compact('grn', 'warehouses', 'racks'));
    }

    /**
     * Update GRN
     */
    public function update(Request $request, $id)
    {
        $this->authorize('purchase.grn.edit');
        
        $grn = GoodsReceiptNote::findOrFail($id);

        if (!$grn->can_edit) {
            return back()->with('error', 'Cannot update GRN in current status.');
        }

        $validated = $request->validate([
            'grn_date' => 'required|date',
            'warehouse_id' => 'required|exists:warehouses,id',
            'rack_id' => 'nullable|exists:racks,id',
            'invoice_number' => 'nullable|string|max:100',
            'invoice_date' => 'nullable|date',
            'lr_number' => 'nullable|string|max:100',
            'vehicle_number' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.id' => 'required|exists:goods_receipt_note_items,id',
            'items.*.received_qty' => 'nullable|numeric|min:0',
            'items.*.accepted_qty' => 'nullable|numeric|min:0',
            'items.*.rejected_qty' => 'nullable|numeric|min:0',
            'items.*.lot_no' => 'nullable|string|max:100',
            'items.*.batch_no' => 'nullable|string|max:100',
            'items.*.manufacturing_date' => 'nullable|date',
            'items.*.expiry_date' => 'nullable|date',
        ]);

        DB::beginTransaction();
        try {
            $grn->update([
                'grn_date' => $validated['grn_date'],
                'warehouse_id' => $validated['warehouse_id'],
                'rack_id' => $validated['rack_id'] ?? null,
                'invoice_number' => $validated['invoice_number'] ?? null,
                'invoice_date' => $validated['invoice_date'] ?? null,
                'lr_number' => $validated['lr_number'] ?? null,
                'vehicle_number' => $validated['vehicle_number'] ?? null,
                'notes' => $validated['notes'] ?? null,
            ]);

            // Update items
            foreach ($validated['items'] as $itemData) {
                $grnItem = $grn->items()->find($itemData['id']);
                if ($grnItem) {
                    $grnItem->update([
                        'received_qty' => $itemData['received_qty'] ?? $grnItem->received_qty,
                        'accepted_qty' => $itemData['accepted_qty'] ?? $grnItem->accepted_qty,
                        'rejected_qty' => $itemData['rejected_qty'] ?? 0,
                        'lot_no' => $itemData['lot_no'] ?? null,
                        'batch_no' => $itemData['batch_no'] ?? null,
                        'manufacturing_date' => $itemData['manufacturing_date'] ?? null,
                        'expiry_date' => $itemData['expiry_date'] ?? null,
                    ]);
                }
            }

            $grn->calculateTotals();

            DB::commit();

            return redirect()
                ->route('admin.purchase.grn.show', $grn->id)
                ->with('success', 'GRN updated successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error updating GRN: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Submit for inspection (alias: inspect)
     */
    public function submit($id)
    {
        $this->authorize('purchase.grn.approve');
        
        $grn = GoodsReceiptNote::findOrFail($id);

        if ($grn->status !== 'DRAFT') {
            return back()->with('error', 'Only draft GRNs can be submitted.');
        }

        $grn->update(['status' => 'INSPECTING']);

        return back()->with('success', 'GRN submitted for inspection.');
    }

    /**
     * Start inspection (change status from DRAFT to INSPECTING)
     */
    public function inspect($id)
    {
        return $this->submit($id);
    }

    /**
     * Approve GRN and update stock
     */
    public function approve($id)
    {
        $this->authorize('purchase.grn.approve');
        
        $grn = GoodsReceiptNote::with(['items.product'])->findOrFail($id);

        if (!in_array($grn->status, ['DRAFT', 'INSPECTING'])) {
            return back()->with('error', 'Only draft/inspecting GRNs can be approved.');
        }

        if ($grn->stock_updated) {
            return back()->with('error', 'Stock already updated for this GRN.');
        }

        DB::beginTransaction();
        try {
            // Update stock for each item
            foreach ($grn->items as $item) {
                if ($item->accepted_qty <= 0) continue;

                $product = $item->product;
                if (!$product || !$product->track_inventory) continue;

                $lotId = null;

                // Handle lot/batch if product is batch managed
                if ($product->is_batch_managed && $item->lot_no) {
                    $lot = \Modules\Inventory\Models\Lot::firstOrCreate([
                        'product_id' => $product->id,
                        'lot_no' => $item->lot_no,
                    ], [
                        'variation_id' => $item->variation_id,
                        'batch_no' => $item->batch_no,
                        'manufacturing_date' => $item->manufacturing_date,
                        'expiry_date' => $item->expiry_date,
                        'purchase_price' => $item->rate,
                        'initial_qty' => $item->accepted_qty,
                        'status' => 'ACTIVE',
                    ]);

                    $lotId = $lot->id;
                    $item->lot_id = $lotId;
                    $item->save();
                }

                // Get variation_id - handle empty string as null
                $variationId = !empty($item->variation_id) ? $item->variation_id : null;

                // Get current stock before update (filter by variation_id too)
                $stockBeforeQuery = \Modules\Inventory\Models\StockLevel::where('product_id', $product->id)
                    ->where('warehouse_id', $grn->warehouse_id)
                    ->when($grn->rack_id, fn($q) => $q->where('rack_id', $grn->rack_id))
                    ->when($lotId, fn($q) => $q->where('lot_id', $lotId));
                
                if ($variationId) {
                    $stockBeforeQuery->where('variation_id', $variationId);
                } else {
                    $stockBeforeQuery->whereNull('variation_id');
                }
                $stockBefore = $stockBeforeQuery->sum('qty') ?? 0;

                // Update/Create stock level
                $stockLevel = \Modules\Inventory\Models\StockLevel::firstOrNew([
                    'product_id' => $product->id,
                    'variation_id' => $variationId,
                    'warehouse_id' => $grn->warehouse_id,
                    'rack_id' => $grn->rack_id,
                    'lot_id' => $lotId,
                ]);

                $stockLevel->unit_id = $item->unit_id ?? $product->unit_id;
                $stockLevel->qty = ($stockLevel->qty ?? 0) + $item->accepted_qty;
                $stockLevel->save();

                $stockAfter = $stockLevel->qty;
                
                // Also update product_variations.stock_qty if variation exists
                if ($variationId) {
                    if (\Illuminate\Support\Facades\Schema::hasColumn('product_variations', 'stock_qty')) {
                        $updated = \Illuminate\Support\Facades\DB::table('product_variations')
                            ->where('id', $variationId)
                            ->increment('stock_qty', $item->accepted_qty);
                        
                        \Log::info("GRN Stock Update - Variation: {$variationId}, Qty: {$item->accepted_qty}, Updated: {$updated}");
                    }
                } else {
                    // Update product.stock_qty if no variation
                    if (\Illuminate\Support\Facades\Schema::hasColumn('products', 'stock_qty')) {
                        \Illuminate\Support\Facades\DB::table('products')
                            ->where('id', $product->id)
                            ->increment('stock_qty', $item->accepted_qty);
                    }
                }

                // Create stock movement record
                $refNo = 'GRN-' . $grn->grn_number . '-' . str_pad($item->id, 3, '0', STR_PAD_LEFT);
                
                $movement = \Modules\Inventory\Models\StockMovement::create([
                    'reference_no' => $refNo,
                    'product_id' => $product->id,
                    'variation_id' => $variationId,
                    'warehouse_id' => $grn->warehouse_id,
                    'rack_id' => $grn->rack_id,
                    'lot_id' => $lotId,
                    'unit_id' => $item->unit_id ?? $product->unit_id,
                    'qty' => $item->accepted_qty,
                    'base_qty' => $item->accepted_qty, // Assuming same unit, adjust if needed
                    'stock_before' => $stockBefore,
                    'stock_after' => $stockAfter,
                    'purchase_price' => $item->rate,
                    'movement_type' => 'IN',
                    'reference_type' => 'PURCHASE',
                    'reference_id' => $grn->id,
                    'reason' => 'GRN Receipt: ' . $grn->grn_number,
                    'notes' => 'PO: ' . ($grn->purchaseOrder->po_number ?? '-'),
                    'created_by' => auth()->id(),
                ]);

                // Link movement to GRN item
                $item->stock_movement_id = $movement->id;
                $item->save();
            }

            // Update GRN status
            $grn->update([
                'status' => 'APPROVED',
                'stock_updated' => true,
                'approved_at' => now(),
                'approved_by' => auth()->id(),
            ]);

            // Update Purchase Order status
            $grn->updatePurchaseOrder();

            DB::commit();

            return back()->with('success', 'GRN approved and stock updated successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error approving GRN: ' . $e->getMessage());
        }
    }

    /**
     * Reject GRN
     */
    public function reject(Request $request, $id)
    {
        $this->authorize('purchase.grn.approve');
        
        $grn = GoodsReceiptNote::findOrFail($id);

        if (!in_array($grn->status, ['DRAFT', 'INSPECTING'])) {
            return back()->with('error', 'Cannot reject GRN in current status.');
        }

        $request->validate([
            'rejection_reason' => 'required|string|max:500',
        ]);

        $grn->update([
            'status' => 'REJECTED',
            'rejection_reason' => $request->rejection_reason,
        ]);

        return back()->with('success', 'GRN rejected.');
    }

    /**
     * Cancel GRN
     */
    public function cancel($id)
    {
        $grn = GoodsReceiptNote::findOrFail($id);

        if ($grn->stock_updated) {
            return back()->with('error', 'Cannot cancel GRN after stock update.');
        }

        if ($grn->status === 'APPROVED') {
            return back()->with('error', 'Cannot cancel approved GRN.');
        }

        $grn->update(['status' => 'CANCELLED']);

        return back()->with('success', 'GRN cancelled.');
    }

    /**
     * Delete GRN
     */
    public function destroy($id)
    {
        $this->authorize('purchase.grn.delete');
        
        $grn = GoodsReceiptNote::findOrFail($id);

        if ($grn->stock_updated) {
            if (request()->ajax()) {
                return response()->json(['success' => false, 'message' => 'Cannot delete GRN after stock update.'], 422);
            }
            return back()->with('error', 'Cannot delete GRN after stock update.');
        }

        $grn->items()->delete();
        $grn->delete();

        if (request()->ajax()) {
            return response()->json(['success' => true, 'message' => 'GRN deleted successfully.']);
        }

        return redirect()
            ->route('admin.purchase.grn.index')
            ->with('success', 'GRN deleted successfully.');
    }

    /**
     * Bulk delete GRNs
     */
    public function bulkDelete(Request $request)
    {
        $ids = $request->input('ids', []);

        if (empty($ids)) {
            return response()->json(['success' => false, 'message' => 'No items selected.'], 400);
        }

        // Check for stock updated GRNs
        $hasStockUpdated = GoodsReceiptNote::whereIn('id', $ids)->where('stock_updated', true)->exists();
        if ($hasStockUpdated) {
            return response()->json([
                'success' => false, 
                'message' => 'Cannot delete GRNs with updated stock.'
            ], 422);
        }

        GoodsReceiptNoteItem::whereIn('goods_receipt_note_id', $ids)->delete();
        GoodsReceiptNote::whereIn('id', $ids)->delete();

        return response()->json([
            'success' => true,
            'message' => count($ids) . ' GRN(s) deleted.'
        ]);
    }

    /**
     * Get PO items for GRN creation (AJAX)
     */
    public function getPOItems($poId)
    {
        $po = PurchaseOrder::with(['items.product.unit', 'items.variation', 'items.unit', 'vendor'])
            ->findOrFail($poId);

        $items = $po->items->map(function($item) {
            $pendingQty = max(0, $item->qty - $item->received_qty);
            $product = $item->product;
            $variation = $item->variation;
            
            return [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'variation_id' => $item->variation_id,
                'product' => $product ? [
                    'name' => $product->name,
                    'sku' => $product->sku,
                    'is_batch_managed' => $product->is_batch_managed ?? false,
                ] : null,
                'variation' => $variation ? [
                    'id' => $variation->id,
                    'variation_name' => $variation->variation_name,
                    'sku' => $variation->sku,
                ] : null,
                'unit_id' => $item->unit_id,
                'unit' => $item->unit ? [
                    'short_name' => $item->unit->short_name ?? $item->unit->name,
                    'name' => $item->unit->name,
                ] : ($product && $product->unit ? [
                    'short_name' => $product->unit->short_name ?? $product->unit->name,
                    'name' => $product->unit->name,
                ] : null),
                'ordered_qty' => $item->qty,
                'received_qty' => $item->received_qty,
                'pending_qty' => $pendingQty,
                'rate' => $item->rate,
            ];
        })->filter(function($item) {
            return $item['pending_qty'] > 0;
        })->values();

        return response()->json([
            'success' => true,
            'po' => [
                'id' => $po->id,
                'po_number' => $po->po_number,
                'vendor_name' => $po->vendor->name ?? '-',
                'po_date' => $po->po_date->format('d M Y'),
            ],
            'items' => $items,
        ]);
    }

    /**
     * Get racks by warehouse (AJAX)
     */
    public function getRacks($warehouseId)
    {
        $racks = [];
        
        if (class_exists('\Modules\Inventory\Models\Rack')) {
            $racks = \Modules\Inventory\Models\Rack::where('warehouse_id', $warehouseId)
                ->where('is_active', true)
                ->orderBy('code')
                ->get(['id', 'code', 'name']);
        }

        return response()->json(['racks' => $racks]);
    }
}
