<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Service Module Migration - NO FOREIGN KEY CONSTRAINTS
     */
    public function up(): void
    {
        // 1. SERVICES TABLE
        Schema::create('services', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('client_id');
            $table->string('machine_name');
            $table->string('equipment_no')->nullable();
            $table->string('model_no')->nullable();
            $table->string('serial_number')->nullable();
            $table->string('service_frequency')->default('monthly');
            $table->integer('custom_days')->nullable();
            $table->date('first_service_date');
            $table->date('last_service_date')->nullable();
            $table->date('next_service_date')->nullable();
            $table->string('status')->default('active');
            $table->string('service_status')->default('draft');
            $table->text('notes')->nullable();
            $table->integer('reminder_days')->default(15);
            $table->boolean('auto_reminder')->default(true);
            $table->timestamp('last_reminder_sent')->nullable();
            $table->unsignedBigInteger('created_by')->nullable();
            $table->timestamps();

            $table->index('client_id');
            $table->index('status');
            $table->index('service_status');
            $table->index('next_service_date');
        });

        // 2. SERVICE RECORDS TABLE
        Schema::create('service_records', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('service_id');
            $table->string('reference_no')->nullable();
            $table->unsignedBigInteger('engineer_id')->nullable();
            $table->string('service_type')->default('Maintenance');
            $table->date('service_date');
            $table->time('service_time')->nullable();
            $table->integer('time_taken')->nullable();
            $table->string('status')->default('scheduled');
            $table->text('remarks')->nullable();
            $table->text('work_done')->nullable();
            $table->decimal('labor_cost', 10, 2)->default(0);
            $table->decimal('total_cost', 10, 2)->default(0);
            $table->boolean('is_paid')->default(false);
            $table->decimal('service_charge', 10, 2)->default(0);
            $table->unsignedBigInteger('invoice_id')->nullable();
            $table->string('service_reference', 20)->nullable();
            $table->boolean('dates_updated')->default(false);
            $table->unsignedBigInteger('created_by')->nullable();
            $table->timestamps();

            $table->index('service_id');
            $table->index('service_date');
            $table->index('status');
        });

        // 3. SERVICE RECORD MATERIALS TABLE
        Schema::create('service_record_materials', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('service_record_id');
            $table->unsignedBigInteger('product_id')->nullable();
            $table->string('material_name')->nullable();
            $table->decimal('quantity', 10, 2)->default(1);
            $table->decimal('unit_price', 10, 2)->default(0);
            $table->decimal('total', 10, 2)->default(0);
            $table->decimal('tax_amount', 15, 2)->default(0);
            $table->text('tax_ids')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();

            $table->index('service_record_id');
        });

        // 4. SERVICE VISITS TABLE
        Schema::create('service_visits', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('service_id');
            $table->unsignedBigInteger('engineer_id')->nullable();
            $table->date('visit_date');
            $table->time('visit_time')->nullable();
            $table->time('check_in_time')->nullable();
            $table->time('check_out_time')->nullable();
            $table->string('status')->default('scheduled');
            $table->text('purpose')->nullable();
            $table->text('notes')->nullable();
            $table->string('client_signature')->nullable();
            $table->unsignedBigInteger('created_by')->nullable();
            $table->timestamps();

            $table->index('service_id');
            $table->index('visit_date');
        });

        // 5. SERVICE NOTIFICATIONS TABLE
        Schema::create('service_notifications', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('service_id');
            $table->string('type', 50)->default('reminder');
            $table->string('email_to');
            $table->string('subject');
            $table->text('message')->nullable();
            $table->string('status')->default('pending');
            $table->timestamp('sent_at')->nullable();
            $table->timestamps();

            $table->index('service_id');
        });

        // 6. ADD COLUMNS TO INVOICES TABLE
        if (Schema::hasTable('invoices')) {
            if (!Schema::hasColumn('invoices', 'service_reference')) {
                Schema::table('invoices', function (Blueprint $table) {
                    $table->string('service_reference', 255)->nullable();
                });
            }
            if (!Schema::hasColumn('invoices', 'service_id')) {
                Schema::table('invoices', function (Blueprint $table) {
                    $table->unsignedBigInteger('service_id')->nullable();
                });
            }
            if (!Schema::hasColumn('invoices', 'service_record_id')) {
                Schema::table('invoices', function (Blueprint $table) {
                    $table->unsignedBigInteger('service_record_id')->nullable();
                });
            }
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('service_notifications');
        Schema::dropIfExists('service_visits');
        Schema::dropIfExists('service_record_materials');
        Schema::dropIfExists('service_records');
        Schema::dropIfExists('services');
    }
};
