<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('support_tickets', function (Blueprint $table) {
            $table->id();
            
            // Human-readable ticket number
            $table->string('ticket_number', 50)->unique()->nullable()
                  ->comment('Human-readable ticket number: TCK-20250107-0001');
            
            // Ticket Identification
            $table->string('ticket_key', 32)->unique()
                  ->comment('Unique key for public access without login');
            $table->string('subject', 191);
            $table->text('message')->nullable()
                  ->comment('Initial ticket description');
            
            // Customer Information
            $table->foreignId('customer_id')->nullable()
                  ->comment('Reference to customers table')
                  ->constrained('customers')
                  ->nullOnDelete();
            
            // Manual Entry Fields
            $table->string('name', 255)->nullable()
                  ->comment('Submitter name - auto-filled from customer or manual');
            $table->string('email', 255)->nullable()
                  ->comment('Submitter email - auto-filled from customer or manual');
            
            // Contact Person
            $table->integer('contact_id')->default(0)
                  ->comment('Contact person ID within customer organization');
            
            // Categorization
            $table->foreignId('department_id')
                  ->constrained('support_departments')
                  ->restrictOnDelete();
            
            $table->foreignId('priority_id')
                  ->constrained('support_priorities')
                  ->restrictOnDelete();
            
            $table->foreignId('status_id')
                  ->constrained('support_statuses')
                  ->restrictOnDelete();
            
            // Service Association
            $table->integer('service_id')->nullable()
                  ->comment('Related service/product ID if applicable');
            
            // Project Linking
            $table->integer('project_id')->default(0)
                  ->comment('Linked project ID if ticket relates to project');
            
            // Assignment & Tracking
            $table->foreignId('assigned_to')->nullable()
                  ->comment('Staff assigned to ticket')
                  ->constrained('staffs')
                  ->nullOnDelete();
            
            $table->foreignId('admin_replying')->default(0)
                  ->comment('Staff currently typing reply');
            
            $table->foreignId('staff_id_replying')->nullable()
                  ->comment('Last staff who replied')
                  ->constrained('staffs')
                  ->nullOnDelete();
            
            // Ticket Metadata
            $table->foreignId('opened_by')->nullable()
                  ->comment('Staff who opened ticket (NULL = customer opened)')
                  ->constrained('staffs')
                  ->nullOnDelete();
            
            // Merged Ticket Tracking
            $table->foreignId('merged_ticket_id')->nullable()
                  ->comment('Parent ticket ID if this was merged INTO another')
                  ->constrained('support_tickets')
                  ->nullOnDelete();
            
            // Track merge status
            $table->boolean('is_merged')->default(false)
                  ->comment('TRUE if this ticket was merged into another');
            
            // CC Recipients
            $table->string('cc', 191)->nullable()
                  ->comment('Comma-separated email addresses for CC');
            
            // Tags
            $table->string('tags', 191)->nullable()
                  ->comment('Comma-separated tags for ticket categorization');
            
            // Read Status Tracking
            $table->boolean('client_read')->default(false)
                  ->comment('Has customer read latest reply?');
            $table->boolean('admin_read')->default(false)
                  ->comment('Has admin read latest reply?');
            
            // Reply Timestamp
            $table->timestamp('last_reply')->nullable()
                  ->comment('Datetime of most recent reply');
            
            $table->timestamps();
            
            // Indexes for Performance
            $table->index('ticket_number'); 
            $table->index('ticket_key');
            $table->index('customer_id');
            $table->index('department_id');
            $table->index('priority_id');
            $table->index('status_id');
            $table->index('assigned_to');
            $table->index('service_id');
            $table->index('project_id');
            $table->index('contact_id');
            $table->index('client_read');
            $table->index('admin_read');
            $table->index('last_reply');
            $table->index('is_merged'); 
            $table->index('merged_ticket_id'); 
            $table->index(['status_id', 'assigned_to']);
            $table->index(['customer_id', 'status_id']);
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('support_tickets');
    }
};