<?php

namespace Modules\Support\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Support\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Exception;

class DepartmentController extends AdminController
{
    /**
     * Display a listing of departments
     */
    public function index()
    {
        try {
            return view('support::admin.departments.index');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to load departments: ' . $e->getMessage());
        }
    }

    /**
     * DataTable endpoint for departments
     */
    public function dataTable(Request $request)
    {
        try {
            $query = Department::query();

            // Search
            if ($search = $request->input('search')) {
                $query->search($search);
            }

            // Filters
            if ($filters = $request->input('filters')) {
                $filters = json_decode($filters, true);
                
                foreach ($filters as $key => $value) {
                    if (!empty($value)) {
                        if ($key === 'is_active') {
                            $query->where('is_active', $value === '1');
                        }
                    }
                }
            }

            // Sorting
            $sortColumn = $request->input('sort', 'department_order');
            $sortDirection = $request->input('dir', 'asc');
            $query->orderBy($sortColumn, $sortDirection);

            // Pagination
            $perPage = $request->input('per_page', 15);
            $departments = $query->paginate($perPage);

            // Transform data
            $data = $departments->map(function ($dept) {
                return [
                    'id' => $dept->id,
                    'name' => $dept->name,
                    'description' => $dept->description,
                    'tickets_count' => $dept->tickets_count,
                    'active_tickets_count' => $dept->active_tickets_count,
                    'is_active' => $dept->is_active,
                    'show_on_client_portal' => $dept->show_on_client_portal,
                    'status_label' => $dept->status_label,
                    'created_at' => $dept->created_at->format('M d, Y'),
                    '_edit_url' => route('admin.support.departments.edit', $dept->id),
                ];
            });

            return response()->json([
                'data' => $data,
                'total' => $departments->total(),
                'current_page' => $departments->currentPage(),
                'last_page' => $departments->lastPage(),
                'per_page' => $departments->perPage(),
            ]);

        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to load departments: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for creating a new department
     */
    public function create()
    {
        try {
            return view('support::admin.departments.create');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to load create form: ' . $e->getMessage());
        }
    }

    /**
     * Store a newly created department
     */
    public function store(Request $request)
    {
        try {
            // Debug: Log incoming request data
            Log::info('Department Store Request:', $request->all());

            // Validate input - NO CHECKBOX VALIDATION!
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:191|unique:support_departments,name',
                'description' => 'nullable|string',
                'department_order' => 'nullable|integer|min:0',
            ]);

            if ($validator->fails()) {
                Log::error('Validation Failed:', $validator->errors()->toArray());
                
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            // Prepare data with explicit boolean conversion
            $data = [
                'name' => $request->name,
                'description' => $request->description,
                'is_active' => $request->has('is_active') ? true : false,
                'show_on_client_portal' => $request->has('show_on_client_portal') ? true : false,
                'department_order' => $request->department_order ?? 0,
            ];

            Log::info('Creating Department with data:', $data);

            // Create department
            $department = Department::create($data);

            Log::info('Department Created Successfully:', ['id' => $department->id]);

            // Return with success message
            return redirect()
                ->route('admin.support.departments.index')
                ->with('success', 'Department created successfully!');

        } catch (Exception $e) {
            Log::error('Department Creation Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to create department: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified department
     */
    public function edit($id)
    {
        try {
            $department = Department::findOrFail($id);
            return view('support::admin.departments.edit', compact('department'));
        } catch (Exception $e) {
            return back()->with('error', 'Department not found');
        }
    }

    /**
     * Update the specified department
     */
    public function update(Request $request, $id)
    {
        try {
            // Debug: Log incoming request data
            Log::info('Department Update Request:', array_merge(['id' => $id], $request->all()));

            // Validate input - NO CHECKBOX VALIDATION!
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:191|unique:support_departments,name,' . $id,
                'description' => 'nullable|string',
                'department_order' => 'nullable|integer|min:0',
            ]);

            if ($validator->fails()) {
                Log::error('Update Validation Failed:', $validator->errors()->toArray());
                
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            $department = Department::findOrFail($id);
            
            // Prepare data with explicit boolean conversion
            $data = [
                'name' => $request->name,
                'description' => $request->description,
                'is_active' => $request->has('is_active') ? true : false,
                'show_on_client_portal' => $request->has('show_on_client_portal') ? true : false,
                'department_order' => $request->department_order ?? 0,
            ];

            Log::info('Updating Department with data:', $data);

            // Update department
            $department->update($data);

            Log::info('Department Updated Successfully:', ['id' => $department->id]);

            // Return with success message
            return redirect()
                ->route('admin.support.departments.index')
                ->with('success', 'Department updated successfully!');

        } catch (Exception $e) {
            Log::error('Department Update Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to update department: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified department
     */
    public function destroy($id)
    {
        try {
            $department = Department::findOrFail($id);
            
            // Check if department has tickets
            if ($department->tickets()->count() > 0) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot delete department with existing tickets'
                    ], 400);
                }
                
                return back()->with('error', 'Cannot delete department with existing tickets');
            }

            $department->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Department deleted successfully'
                ]);
            }

            return redirect()
                ->route('admin.support.departments.index')
                ->with('success', 'Department deleted successfully!');

        } catch (Exception $e) {
            Log::error('Department Delete Failed:', ['error' => $e->getMessage()]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete department: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Failed to delete department: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete departments
     */
    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->input('ids', []);
            
            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No departments selected'
                ], 400);
            }

            // Check for departments with tickets
            $hasTickets = Department::whereIn('id', $ids)
                ->whereHas('tickets')
                ->exists();

            if ($hasTickets) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete departments with existing tickets'
                ], 400);
            }

            $deleted = Department::whereIn('id', $ids)->delete();

            return response()->json([
                'success' => true,
                'message' => 'Deleted ' . $deleted . ' departments successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Bulk Delete Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete departments: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle active status
     */
    public function toggleActive($id)
    {
        try {
            $department = Department::findOrFail($id);
            $department->is_active = !$department->is_active;
            $department->save();

            return response()->json([
                'success' => true,
                'message' => 'Department ' . ($department->is_active ? 'activated' : 'deactivated') . ' successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Toggle Active Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search departments (autocomplete)
     */
    public function search(Request $request)
    {
        try {
            $search = $request->input('q', '');
            
            $departments = Department::active()
                ->where('name', 'LIKE', "%{$search}%")
                ->limit(10)
                ->get()
                ->map(function($dept) {
                    return [
                        'id' => $dept->id,
                        'text' => $dept->name,
                    ];
                });

            return response()->json($departments);

        } catch (Exception $e) {
            return response()->json([]);
        }
    }
}