<?php

namespace Modules\Support\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Support\Models\PredefinedReply;
use Modules\Support\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Exception;

class PredefinedReplyController extends AdminController
{
    /**
     * Display a listing of predefined replies
     */
    public function index()
    {
        try {
            $departments = Department::active()->ordered()->get();
            return view('support::admin.predefined-replies.index', compact('departments'));
        } catch (Exception $e) {
            Log::error('Predefined Replies Index Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Failed to load predefined replies: ' . $e->getMessage());
        }
    }

    /**
     * DataTable endpoint for predefined replies
     */
    public function dataTable(Request $request)
    {
        try {
            $query = PredefinedReply::with('department');

            // Search
            if ($search = $request->input('search')) {
                $query->search($search);
            }

            // Filters
            if ($filters = $request->input('filters')) {
                $filters = json_decode($filters, true);
                
                foreach ($filters as $key => $value) {
                    if (!empty($value)) {
                        if ($key === 'is_active') {
                            $query->where('is_active', $value === '1');
                        } elseif ($key === 'department_id') {
                            if ($value === 'all') {
                                $query->whereNull('department_id');
                            } else {
                                $query->where('department_id', $value);
                            }
                        }
                    }
                }
            }

            // Sorting
            $sortColumn = $request->input('sort', 'name');
            $sortDirection = $request->input('dir', 'asc');
            $query->orderBy($sortColumn, $sortDirection);

            // Pagination
            $perPage = $request->input('per_page', 20);
            $replies = $query->paginate($perPage);

            // Transform data
            $data = $replies->map(function ($reply) {
                return [
                    'id' => $reply->id,
                    'name' => $reply->name,
                    'message_preview' => $reply->message_preview,
                    'department_name' => $reply->department_name,
                    'is_active' => $reply->is_active,
                    'status_label' => $reply->status_label,
                    'created_at' => $reply->created_at->format('M d, Y'),
                    '_edit_url' => route('admin.support.predefined-replies.edit', $reply->id),
                ];
            });

            return response()->json([
                'data' => $data,
                'total' => $replies->total(),
                'current_page' => $replies->currentPage(),
                'last_page' => $replies->lastPage(),
                'per_page' => $replies->perPage(),
            ]);

        } catch (Exception $e) {
            Log::error('Predefined Replies DataTable Failed:', ['error' => $e->getMessage()]);
            return response()->json([
                'error' => 'Failed to load predefined replies: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for creating a new predefined reply
     */
    public function create()
    {
        try {
            $departments = Department::active()->ordered()->get();
            return view('support::admin.predefined-replies.create', compact('departments'));
        } catch (Exception $e) {
            Log::error('Predefined Reply Create Form Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Failed to load create form: ' . $e->getMessage());
        }
    }

    /**
     * Store a newly created predefined reply
     */
    public function store(Request $request)
    {
        try {
            // Debug log
            Log::info('Predefined Reply Store Request:', $request->all());

            // Validate - NO CHECKBOX VALIDATION!
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:191|unique:support_predefined_replies,name',
                'message' => 'required|string',
                'department_id' => 'nullable|exists:support_departments,id',
            ]);

            if ($validator->fails()) {
                Log::error('Validation Failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            // Prepare data with explicit boolean conversion
            $data = [
                'name' => $request->name,
                'message' => $request->message,
                'department_id' => $request->department_id,
                'is_active' => $request->has('is_active') ? true : false,
            ];

            Log::info('Creating Predefined Reply with data:', $data);

            $reply = PredefinedReply::create($data);

            Log::info('Predefined Reply Created Successfully:', ['id' => $reply->id]);

            return redirect()
                ->route('admin.support.predefined-replies.index')
                ->with('success', 'Predefined reply created successfully!');

        } catch (Exception $e) {
            Log::error('Predefined Reply Creation Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to create predefined reply: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified predefined reply
     */
    public function edit($id)
    {
        try {
            $reply = PredefinedReply::with('department')->findOrFail($id);
            $departments = Department::active()->ordered()->get();
            
            return view('support::admin.predefined-replies.edit', compact('reply', 'departments'));
        } catch (Exception $e) {
            Log::error('Predefined Reply Edit Form Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Predefined reply not found');
        }
    }

    /**
     * Update the specified predefined reply
     */
    public function update(Request $request, $id)
    {
        try {
            // Debug log
            Log::info('Predefined Reply Update Request:', array_merge(['id' => $id], $request->all()));

            // Validate - NO CHECKBOX VALIDATION!
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:191|unique:support_predefined_replies,name,' . $id,
                'message' => 'required|string',
                'department_id' => 'nullable|exists:support_departments,id',
            ]);

            if ($validator->fails()) {
                Log::error('Update Validation Failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            $reply = PredefinedReply::findOrFail($id);
            
            // Prepare data with explicit boolean conversion
            $data = [
                'name' => $request->name,
                'message' => $request->message,
                'department_id' => $request->department_id,
                'is_active' => $request->has('is_active') ? true : false,
            ];

            Log::info('Updating Predefined Reply with data:', $data);

            $reply->update($data);

            Log::info('Predefined Reply Updated Successfully:', ['id' => $reply->id]);

            return redirect()
                ->route('admin.support.predefined-replies.index')
                ->with('success', 'Predefined reply updated successfully!');

        } catch (Exception $e) {
            Log::error('Predefined Reply Update Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to update predefined reply: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified predefined reply
     */
    public function destroy($id)
    {
        try {
            $reply = PredefinedReply::findOrFail($id);
            $reply->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Predefined reply deleted successfully'
                ]);
            }

            return redirect()
                ->route('admin.support.predefined-replies.index')
                ->with('success', 'Predefined reply deleted successfully!');

        } catch (Exception $e) {
            Log::error('Predefined Reply Delete Failed:', ['error' => $e->getMessage()]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete predefined reply: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Failed to delete predefined reply: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete predefined replies
     */
    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->input('ids', []);
            
            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No predefined replies selected'
                ], 400);
            }

            $deleted = PredefinedReply::whereIn('id', $ids)->delete();

            return response()->json([
                'success' => true,
                'message' => 'Deleted ' . $deleted . ' predefined replies successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Bulk Delete Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete predefined replies: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle active status
     */
    public function toggleActive($id)
    {
        try {
            $reply = PredefinedReply::findOrFail($id);
            $reply->is_active = !$reply->is_active;
            $reply->save();

            return response()->json([
                'success' => true,
                'message' => 'Predefined reply ' . ($reply->is_active ? 'activated' : 'deactivated') . ' successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Toggle Active Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get reply content (for ticket reply form)
     */
    public function getReply($id)
    {
        try {
            $reply = PredefinedReply::active()->findOrFail($id);

            return response()->json([
                'success' => true,
                'reply' => [
                    'id' => $reply->id,
                    'name' => $reply->name,
                    'message' => $reply->message,
                ]
            ]);

        } catch (Exception $e) {
            Log::error('Get Reply Failed:', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Predefined reply not found'
            ], 404);
        }
    }

    /**
     * Search predefined replies (autocomplete for ticket reply form)
     */
    public function search(Request $request)
    {
        try {
            $search = $request->input('q', '');
            $departmentId = $request->input('department_id');
            
            $query = PredefinedReply::active();
            
            // Filter by department if provided
            if ($departmentId) {
                $query->byDepartment($departmentId);
            }
            
            // Search
            if ($search) {
                $query->where('name', 'LIKE', "%{$search}%");
            }
            
            $replies = $query->limit(10)
                ->get()
                ->map(function($reply) {
                    return [
                        'id' => $reply->id,
                        'text' => $reply->name,
                        'message' => $reply->message,
                    ];
                });

            return response()->json($replies);

        } catch (Exception $e) {
            Log::error('Search Replies Failed:', ['error' => $e->getMessage()]);
            return response()->json([]);
        }
    }
}