<?php

namespace Modules\Support\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Support\Models\Priority;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Exception;

class PriorityController extends AdminController
{
    /**
     * Display a listing of priorities
     */
    public function index()
    {
        try {
            return view('support::admin.priorities.index');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to load priorities: ' . $e->getMessage());
        }
    }

    /**
     * DataTable endpoint for priorities
     */
    public function dataTable(Request $request)
    {
        try {
            $query = Priority::query();

            // Search
            if ($search = $request->input('search')) {
                $query->search($search);
            }

            // Filters
            if ($filters = $request->input('filters')) {
                $filters = json_decode($filters, true);
                
                foreach ($filters as $key => $value) {
                    if (!empty($value)) {
                        if ($key === 'is_active') {
                            $query->where('is_active', $value === '1');
                        }
                    }
                }
            }

            // Sorting
            $sortColumn = $request->input('sort', 'priority_order');
            $sortDirection = $request->input('dir', 'asc');
            $query->orderBy($sortColumn, $sortDirection);

            // Pagination
            $perPage = $request->input('per_page', 15);
            $priorities = $query->paginate($perPage);

            // Transform data
            $data = $priorities->map(function ($priority) {
                return [
                    'id' => $priority->id,
                    'name' => $priority->name,
                    'color' => $priority->color,
                    'badge_html' => $priority->badge_html,
                    'priority_order' => $priority->priority_order,
                    'tickets_count' => $priority->tickets_count,
                    'is_active' => $priority->is_active,
                    'status_label' => $priority->status_label,
                    'created_at' => $priority->created_at->format('M d, Y'),
                    '_edit_url' => route('admin.support.priorities.edit', $priority->id),
                ];
            });

            return response()->json([
                'data' => $data,
                'total' => $priorities->total(),
                'current_page' => $priorities->currentPage(),
                'last_page' => $priorities->lastPage(),
                'per_page' => $priorities->perPage(),
            ]);

        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to load priorities: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for creating a new priority
     */
    public function create()
    {
        try {
            return view('support::admin.priorities.create');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to load create form: ' . $e->getMessage());
        }
    }

    /**
     * Store a newly created priority
     */
    public function store(Request $request)
    {
        try {
            // Debug log
            Log::info('Priority Store Request:', $request->all());

            // Validate - NO CHECKBOX VALIDATION!
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:50|unique:support_priorities,name',
                'color' => 'required|string|max:7|regex:/^#[a-fA-F0-9]{6}$/',
                'priority_order' => 'nullable|integer|min:0',
            ]);

            if ($validator->fails()) {
                Log::error('Validation Failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            // Prepare data with explicit boolean conversion
            $data = [
                'name' => $request->name,
                'color' => $request->color,
                'is_active' => $request->has('is_active') ? true : false,
                'priority_order' => $request->priority_order ?? (Priority::max('priority_order') + 1),
            ];

            Log::info('Creating Priority with data:', $data);

            $priority = Priority::create($data);

            Log::info('Priority Created Successfully:', ['id' => $priority->id]);

            return redirect()
                ->route('admin.support.priorities.index')
                ->with('success', 'Priority created successfully!');

        } catch (Exception $e) {
            Log::error('Priority Creation Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to create priority: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified priority
     */
    public function edit($id)
    {
        try {
            $priority = Priority::findOrFail($id);
            return view('support::admin.priorities.edit', compact('priority'));
        } catch (Exception $e) {
            return back()->with('error', 'Priority not found');
        }
    }

    /**
     * Update the specified priority
     */
    public function update(Request $request, $id)
    {
        try {
            // Debug log
            Log::info('Priority Update Request:', array_merge(['id' => $id], $request->all()));

            // Validate - NO CHECKBOX VALIDATION!
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:50|unique:support_priorities,name,' . $id,
                'color' => 'required|string|max:7|regex:/^#[a-fA-F0-9]{6}$/',
                'priority_order' => 'nullable|integer|min:0',
            ]);

            if ($validator->fails()) {
                Log::error('Update Validation Failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            $priority = Priority::findOrFail($id);

            // Prepare data with explicit boolean conversion
            $data = [
                'name' => $request->name,
                'color' => $request->color,
                'is_active' => $request->has('is_active') ? true : false,
                'priority_order' => $request->priority_order ?? 0,
            ];

            Log::info('Updating Priority with data:', $data);

            $priority->update($data);

            Log::info('Priority Updated Successfully:', ['id' => $priority->id]);

            return redirect()
                ->route('admin.support.priorities.index')
                ->with('success', 'Priority updated successfully!');

        } catch (Exception $e) {
            Log::error('Priority Update Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to update priority: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified priority
     */
    public function destroy($id)
    {
        try {
            $priority = Priority::findOrFail($id);
            
            // Check if priority has tickets
            if ($priority->tickets()->count() > 0) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot delete priority with existing tickets'
                    ], 400);
                }
                return back()->with('error', 'Cannot delete priority with existing tickets');
            }

            $priority->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Priority deleted successfully'
                ]);
            }

            return redirect()
                ->route('admin.support.priorities.index')
                ->with('success', 'Priority deleted successfully!');

        } catch (Exception $e) {
            Log::error('Priority Delete Failed:', ['error' => $e->getMessage()]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete priority: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Failed to delete priority: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete priorities
     */
    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->input('ids', []);
            
            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No priorities selected'
                ], 400);
            }

            // Check for priorities with tickets
            $hasTickets = Priority::whereIn('id', $ids)
                ->whereHas('tickets')
                ->exists();

            if ($hasTickets) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete priorities with existing tickets'
                ], 400);
            }

            $deleted = Priority::whereIn('id', $ids)->delete();

            return response()->json([
                'success' => true,
                'message' => 'Deleted ' . $deleted . ' priorities successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Bulk Delete Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete priorities: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle active status
     */
    public function toggleActive($id)
    {
        try {
            $priority = Priority::findOrFail($id);
            $priority->is_active = !$priority->is_active;
            $priority->save();

            return response()->json([
                'success' => true,
                'message' => 'Priority ' . ($priority->is_active ? 'activated' : 'deactivated') . ' successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Toggle Active Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reorder priorities (drag-drop)
     */
    public function reorder(Request $request)
    {
        try {
            $order = $request->input('order', []);
            
            if (empty($order)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No order data provided'
                ], 400);
            }

            foreach ($order as $item) {
                Priority::where('id', $item['id'])->update(['priority_order' => $item['order']]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Priorities reordered successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Reorder Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to reorder priorities: ' . $e->getMessage()
            ], 500);
        }
    }
}