<?php

namespace Modules\Support\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Support\Models\Status;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Exception;
use DB;

class StatusController extends AdminController
{
    /**
     * Display a listing of statuses
     */
    public function index()
    {
        try {
            return view('support::admin.statuses.index');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to load statuses: ' . $e->getMessage());
        }
    }

    /**
     * DataTable endpoint for statuses
     */
    public function dataTable(Request $request)
    {
        try {
            $query = Status::query();

            // Search
            if ($search = $request->input('search')) {
                $query->search($search);
            }

            // Filters
            if ($filters = $request->input('filters')) {
                $filters = json_decode($filters, true);
                
                foreach ($filters as $key => $value) {
                    if (!empty($value)) {
                        if ($key === 'is_active') {
                            $query->where('is_active', $value === '1');
                        } elseif ($key === 'is_default') {
                            $query->where('is_default', $value === '1');
                        } elseif ($key === 'allow_client_change') {
                            $query->where('allow_client_change', $value === '1');
                        }
                    }
                }
            }

            // Sorting
            $sortColumn = $request->input('sort', 'status_order');
            $sortDirection = $request->input('dir', 'asc');
            $query->orderBy($sortColumn, $sortDirection);

            // Pagination
            $perPage = $request->input('per_page', 15);
            $statuses = $query->paginate($perPage);

            // Transform data
            $data = $statuses->map(function ($status) {
                return [
                    'id' => $status->id,
                    'name' => $status->name,
                    'status_color' => $status->status_color,
                    'badge_html' => $status->badge_html,
                    'status_order' => $status->status_order,
                    'tickets_count' => $status->tickets_count,
                    'is_default' => $status->is_default,
                    'default_label' => $status->default_label,
                    'allow_client_change' => $status->allow_client_change,
                    'is_active' => $status->is_active,
                    'status_label' => $status->status_label,
                    'created_at' => $status->created_at->format('M d, Y'),
                    '_edit_url' => route('admin.support.statuses.edit', $status->id),
                ];
            });

            return response()->json([
                'data' => $data,
                'total' => $statuses->total(),
                'current_page' => $statuses->currentPage(),
                'last_page' => $statuses->lastPage(),
                'per_page' => $statuses->perPage(),
            ]);

        } catch (Exception $e) {
            return response()->json([
                'error' => 'Failed to load statuses: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for creating a new status
     */
    public function create()
    {
        try {
            return view('support::admin.statuses.create');
        } catch (Exception $e) {
            return back()->with('error', 'Failed to load create form: ' . $e->getMessage());
        }
    }

    /**
     * Store a newly created status
     */
    public function store(Request $request)
    {
        try {
            // Debug log
            Log::info('Status Store Request:', $request->all());

            // Validate - NO CHECKBOX VALIDATION!
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:50|unique:support_statuses,name',
                'status_color' => 'nullable|string|max:7|regex:/^#[a-fA-F0-9]{6}$/',
                'status_order' => 'nullable|integer|min:0',
            ]);

            if ($validator->fails()) {
                Log::error('Validation Failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            DB::beginTransaction();

            // If setting as default, unset other defaults
            if ($request->has('is_default')) {
                Status::where('is_default', true)->update(['is_default' => false]);
            }

            // Prepare data with explicit boolean conversion
            $data = [
                'name' => $request->name,
                'status_color' => $request->status_color ?? '#3b82f6',
                'is_active' => $request->has('is_active') ? true : false,
                'is_default' => $request->has('is_default') ? true : false,
                'allow_client_change' => $request->has('allow_client_change') ? true : false,
                'status_order' => $request->status_order ?? (Status::max('status_order') + 1),
            ];

            Log::info('Creating Status with data:', $data);

            $status = Status::create($data);

            Log::info('Status Created Successfully:', ['id' => $status->id]);

            DB::commit();

            return redirect()
                ->route('admin.support.statuses.index')
                ->with('success', 'Status created successfully!');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Status Creation Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to create status: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified status
     */
    public function edit($id)
    {
        try {
            $status = Status::findOrFail($id);
            return view('support::admin.statuses.edit', compact('status'));
        } catch (Exception $e) {
            return back()->with('error', 'Status not found');
        }
    }

    /**
     * Update the specified status
     */
    public function update(Request $request, $id)
    {
        try {
            // Debug log
            Log::info('Status Update Request:', array_merge(['id' => $id], $request->all()));

            // Validate - NO CHECKBOX VALIDATION!
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:50|unique:support_statuses,name,' . $id,
                'status_color' => 'nullable|string|max:7|regex:/^#[a-fA-F0-9]{6}$/',
                'status_order' => 'nullable|integer|min:0',
            ]);

            if ($validator->fails()) {
                Log::error('Update Validation Failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            DB::beginTransaction();

            $status = Status::findOrFail($id);

            // If setting as default, unset other defaults
            if ($request->has('is_default')) {
                Status::where('id', '!=', $id)
                    ->where('is_default', true)
                    ->update(['is_default' => false]);
            }

            // Prepare data with explicit boolean conversion
            $data = [
                'name' => $request->name,
                'status_color' => $request->status_color ?? '#3b82f6',
                'is_active' => $request->has('is_active') ? true : false,
                'is_default' => $request->has('is_default') ? true : false,
                'allow_client_change' => $request->has('allow_client_change') ? true : false,
                'status_order' => $request->status_order ?? 0,
            ];

            Log::info('Updating Status with data:', $data);

            $status->update($data);

            Log::info('Status Updated Successfully:', ['id' => $status->id]);

            DB::commit();

            return redirect()
                ->route('admin.support.statuses.index')
                ->with('success', 'Status updated successfully!');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Status Update Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to update status: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified status
     */
    public function destroy($id)
    {
        try {
            $status = Status::findOrFail($id);
            
            // Prevent deleting default status
            if ($status->is_default) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot delete default status'
                    ], 400);
                }
                return back()->with('error', 'Cannot delete default status');
            }
            
            // Check if status has tickets
            if ($status->tickets()->count() > 0) {
                if (request()->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot delete status with existing tickets'
                    ], 400);
                }
                return back()->with('error', 'Cannot delete status with existing tickets');
            }

            $status->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Status deleted successfully'
                ]);
            }

            return redirect()
                ->route('admin.support.statuses.index')
                ->with('success', 'Status deleted successfully!');

        } catch (Exception $e) {
            Log::error('Status Delete Failed:', ['error' => $e->getMessage()]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete status: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Failed to delete status: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete statuses
     */
    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->input('ids', []);
            
            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No statuses selected'
                ], 400);
            }

            // Check for default status
            $hasDefault = Status::whereIn('id', $ids)
                ->where('is_default', true)
                ->exists();

            if ($hasDefault) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete default status'
                ], 400);
            }

            // Check for statuses with tickets
            $hasTickets = Status::whereIn('id', $ids)
                ->whereHas('tickets')
                ->exists();

            if ($hasTickets) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete statuses with existing tickets'
                ], 400);
            }

            $deleted = Status::whereIn('id', $ids)->delete();

            return response()->json([
                'success' => true,
                'message' => 'Deleted ' . $deleted . ' statuses successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Bulk Delete Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete statuses: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle active status
     */
    public function toggleActive($id)
    {
        try {
            $status = Status::findOrFail($id);
            
            // Prevent deactivating default status
            if ($status->is_default && $status->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot deactivate default status'
                ], 400);
            }
            
            $status->is_active = !$status->is_active;
            $status->save();

            return response()->json([
                'success' => true,
                'message' => 'Status ' . ($status->is_active ? 'activated' : 'deactivated') . ' successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Toggle Active Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Set as default status
     */
    public function setAsDefault($id)
    {
        try {
            DB::beginTransaction();

            // Unset all other defaults
            Status::where('is_default', true)->update(['is_default' => false]);

            // Set this as default
            $status = Status::findOrFail($id);
            $status->is_default = true;
            $status->is_active = true; // Ensure it's active
            $status->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Default status set successfully'
            ]);

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Set Default Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to set default status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reorder statuses (drag-drop)
     */
    public function reorder(Request $request)
    {
        try {
            $order = $request->input('order', []);
            
            if (empty($order)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No order data provided'
                ], 400);
            }

            foreach ($order as $index => $id) {
                Status::where('id', $id)->update(['status_order' => $index + 1]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Statuses reordered successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Reorder Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to reorder statuses: ' . $e->getMessage()
            ], 500);
        }
    }
}