<?php

namespace Modules\Support\Http\Controllers;

use App\Http\Controllers\Controller;
use Modules\Support\Models\Ticket;
use Modules\Support\Models\TicketReply;
use Modules\Support\Models\TicketAttachment;
use Modules\Support\Models\Department;
use Modules\Support\Models\Priority;
use Modules\Support\Models\Status;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Exception;
use DB;

class TicketClientController extends Controller
{
    /**
     * Display list of user's tickets (authenticated)
     */
    public function myTickets()
    {
        try {
            $customer = Auth::guard('web')->user();
            
            $tickets = Ticket::with(['department', 'priority', 'status'])
                ->where('customer_id', $customer->id)
                ->orderBy('created_at', 'desc')
                ->paginate(config('support.pagination.client_tickets', 15));

            $statuses = Status::active()->ordered()->get();

            return view('support::client.my-tickets', compact('tickets', 'statuses'));

        } catch (Exception $e) {
            return back()->with('error', 'Failed to load tickets');
        }
    }

    /**
     * Show form to open a new ticket (authenticated)
     */
    public function openTicket()
    {
        try {
            $customer = Auth::guard('web')->user();
            
            $departments = Department::active()->clientPortal()->ordered()->get();
            $priorities = Priority::active()->ordered()->get();

            return view('support::client.open-ticket', compact('departments', 'priorities', 'customer'));

        } catch (Exception $e) {
            return back()->with('error', 'Failed to load ticket form');
        }
    }

    /**
     * Submit a new ticket (authenticated)
     */
    public function submitTicket(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'subject' => 'required|string|max:191',
                'message' => 'required|string|max:10000',
                'department_id' => 'required|exists:support_departments,id',
                'priority_id' => 'required|exists:support_priorities,id',
                'attachments.*' => 'nullable|file|max:' . config('support.uploads.max_size', 10240),
            ]);

            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            DB::beginTransaction();

            $customer = Auth::guard('web')->user();
            
            // Get default status
            $defaultStatus = Status::where('is_default', true)->first();
            if (!$defaultStatus) {
                $defaultStatus = Status::active()->first();
            }

            // Create ticket
            $ticket = new Ticket();
            $ticket->subject = $request->subject;
            $ticket->message = $request->message;
            $ticket->department_id = $request->department_id;
            $ticket->priority_id = $request->priority_id;
            $ticket->status_id = $defaultStatus->id;
            $ticket->customer_id = $customer->id;
            $ticket->name = $customer->display_name;
            $ticket->email = $customer->email;
            $ticket->client_read = true; // Client created, so mark as read
            $ticket->save();

            // Handle file uploads
            if ($request->hasFile('attachments')) {
                $this->handleAttachments($request->file('attachments'), $ticket->id, null);
            }

            DB::commit();

            return redirect()
                ->route('client.support.view', $ticket->id)
                ->with('success', 'Ticket submitted successfully! Ticket #' . $ticket->id);

        } catch (Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to submit ticket')->withInput();
        }
    }

    /**
     * View ticket details (authenticated)
     */
    public function view($id)
    {
        try {
            $customer = Auth::guard('web')->user();
            
            $ticket = Ticket::with([
                'department',
                'priority',
                'status',
                'replies.staff',
                'replies.customer',
                'replies.attachments',
                'attachments'
            ])
            ->where('customer_id', $customer->id)
            ->findOrFail($id);

            // Mark as read
            if (!$ticket->client_read) {
                $ticket->markAsRead(false);
            }

            // Get allowed statuses for client
            $allowedStatuses = Status::active()
                ->clientChangeable()
                ->ordered()
                ->get();

            return view('support::client.view-ticket', compact('ticket', 'allowedStatuses'));

        } catch (Exception $e) {
            return back()->with('error', 'Ticket not found');
        }
    }

    /**
     * View ticket by key (public - no auth required)
     */
    public function viewByKey($key)
    {
        try {
            $ticket = Ticket::with([
                'department',
                'priority',
                'status',
                'replies.staff',
                'replies.customer',
                'replies.attachments',
                'attachments'
            ])
            ->where('ticket_key', $key)
            ->firstOrFail();

            // Mark as read
            if (!$ticket->client_read) {
                $ticket->markAsRead(false);
            }

            // Get allowed statuses for client
            $allowedStatuses = Status::active()
                ->clientChangeable()
                ->ordered()
                ->get();

            return view('support::client.view-ticket-public', compact('ticket', 'allowedStatuses'));

        } catch (Exception $e) {
            return back()->with('error', 'Ticket not found');
        }
    }

    /**
     * Add reply to ticket (authenticated)
     */
    public function addReply(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'message' => 'required|string|max:10000',
                'attachments.*' => 'nullable|file|max:' . config('support.uploads.max_size', 10240),
            ]);

            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            DB::beginTransaction();

            $customer = Auth::guard('web')->user();
            
            $ticket = Ticket::where('customer_id', $customer->id)
                ->findOrFail($id);

            // Create reply
            $reply = new TicketReply();
            $reply->ticket_id = $ticket->id;
            $reply->message = $request->message;
            $reply->customer_id = $customer->id;
            $reply->name = $customer->display_name;
            $reply->email = $customer->email;
            $reply->replied_at = now();
            $reply->save();

            // Handle file uploads
            if ($request->hasFile('attachments')) {
                $this->handleAttachments($request->file('attachments'), $ticket->id, $reply->id);
                $reply->has_attachment = true;
                $reply->save();
            }

            // Update ticket last_reply and mark admin as unread
            $ticket->last_reply = now();
            $ticket->admin_read = false;
            $ticket->save();

            DB::commit();

            return back()->with('success', 'Reply added successfully!');

        } catch (Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to add reply');
        }
    }

    /**
     * Add reply via ticket key (public)
     */
    public function replyByKey(Request $request, $key)
    {
        try {
            $validator = Validator::make($request->all(), [
                'message' => 'required|string|max:10000',
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'attachments.*' => 'nullable|file|max:' . config('support.uploads.max_size', 10240),
            ]);

            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            DB::beginTransaction();

            $ticket = Ticket::where('ticket_key', $key)->firstOrFail();

            // Create reply
            $reply = new TicketReply();
            $reply->ticket_id = $ticket->id;
            $reply->message = $request->message;
            $reply->name = $request->name;
            $reply->email = $request->email;
            $reply->replied_at = now();
            $reply->save();

            // Handle file uploads
            if ($request->hasFile('attachments')) {
                $this->handleAttachments($request->file('attachments'), $ticket->id, $reply->id);
                $reply->has_attachment = true;
                $reply->save();
            }

            // Update ticket last_reply and mark admin as unread
            $ticket->last_reply = now();
            $ticket->admin_read = false;
            $ticket->save();

            DB::commit();

            return back()->with('success', 'Reply added successfully!');

        } catch (Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to add reply');
        }
    }

    /**
     * Change ticket status (authenticated)
     */
    public function changeStatus(Request $request, $id)
    {
        try {
            $customer = Auth::guard('web')->user();
            
            $ticket = Ticket::where('customer_id', $customer->id)
                ->findOrFail($id);

            // Check if status is allowed for client change
            $status = Status::where('id', $request->status_id)
                ->where('allow_client_change', true)
                ->where('is_active', true)
                ->firstOrFail();

            $ticket->status_id = $status->id;
            $ticket->save();

            return back()->with('success', 'Ticket status changed successfully!');

        } catch (Exception $e) {
            return back()->with('error', 'Failed to change status');
        }
    }

    /**
     * Change status via ticket key (public)
     */
    public function changeStatusByKey(Request $request, $key)
    {
        try {
            $ticket = Ticket::where('ticket_key', $key)->firstOrFail();

            // Check if status is allowed for client change
            $status = Status::where('id', $request->status_id)
                ->where('allow_client_change', true)
                ->where('is_active', true)
                ->firstOrFail();

            $ticket->status_id = $status->id;
            $ticket->save();

            return back()->with('success', 'Ticket status changed successfully!');

        } catch (Exception $e) {
            return back()->with('error', 'Failed to change status');
        }
    }

    /**
     * Download attachment (authenticated)
     */
    public function download($ticketId, $attachmentId)
    {
        try {
            $customer = Auth::guard('web')->user();
            
            $ticket = Ticket::where('customer_id', $customer->id)
                ->findOrFail($ticketId);

            $attachment = TicketAttachment::where('ticket_id', $ticket->id)
                ->findOrFail($attachmentId);

            if (!Storage::disk('public')->exists($attachment->file_path)) {
                return back()->with('error', 'File not found');
            }

            return Storage::disk('public')->download(
                $attachment->file_path,
                $attachment->file_name
            );

        } catch (Exception $e) {
            return back()->with('error', 'Failed to download file');
        }
    }

    /**
     * Download attachment via ticket key (public)
     */
    public function downloadByKey($key, $attachmentId)
    {
        try {
            $ticket = Ticket::where('ticket_key', $key)->firstOrFail();

            $attachment = TicketAttachment::where('ticket_id', $ticket->id)
                ->findOrFail($attachmentId);

            if (!Storage::disk('public')->exists($attachment->file_path)) {
                return back()->with('error', 'File not found');
            }

            return Storage::disk('public')->download(
                $attachment->file_path,
                $attachment->file_name
            );

        } catch (Exception $e) {
            return back()->with('error', 'Failed to download file');
        }
    }

    /**
     * Handle file attachments
     */
    private function handleAttachments($files, $ticketId, $replyId = null)
    {
        foreach ($files as $file) {
            if ($file->isValid()) {
                $fileName = time() . '_' . $file->getClientOriginalName();
                $filePath = 'support_tickets/' . $ticketId . '/' . $fileName;
                
                Storage::disk('public')->putFileAs(
                    'support_tickets/' . $ticketId,
                    $file,
                    $fileName
                );

                TicketAttachment::create([
                    'ticket_id' => $ticketId,
                    'reply_id' => $replyId,
                    'file_name' => $file->getClientOriginalName(),
                    'file_path' => $filePath,
                    'file_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                    'uploaded_at' => now(),
                ]);
            }
        }
    }
}