<?php

namespace Modules\Support\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Support\Models\Ticket;
use Modules\Support\Models\TicketReply;
use Modules\Support\Models\TicketAttachment;
use Modules\Support\Models\Department;
use Modules\Support\Models\Priority;
use Modules\Support\Models\Status;
use App\Models\Customer;
use App\Models\Admin\Staff;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Exception;
use DB;

class TicketController extends AdminController
{
    /**
     * Display a listing of tickets
     */
    public function index()
    {
        try {
            // Get filter options
            $departments = Department::active()->ordered()->get();
            $priorities = Priority::active()->ordered()->get();
            $statuses = Status::active()->ordered()->get();
            $staff = Staff::active()->orderBy('first_name')->get();

            return view('support::admin.tickets.index', compact(
                'departments',
                'priorities',
                'statuses',
                'staff'
            ));

        } catch (Exception $e) {
            Log::error('Tickets Index Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Failed to load tickets: ' . $e->getMessage());
        }
    }

    /**
     * DataTable endpoint for tickets
     */
/**
 * DataTable endpoint for tickets
 */
public function dataTable(Request $request)
{
    try {
        $query = Ticket::with(['department', 'priority', 'status', 'assignedStaff', 'customer', 'mergedIntoTicket'])
            ->notMerged()  //  Hide merged tickets by default
            ->select('support_tickets.*');

        // Search
        if ($search = $request->input('search')) {
            $query->search($search);
        }

        // Handle individual filter parameters
        if ($department = $request->input('department')) {
            $query->where('department_id', $department);
        }

        if ($priority = $request->input('priority')) {
            $query->where('priority_id', $priority);
        }

        if ($status = $request->input('status')) {
            $query->where('status_id', $status);
        }

        if ($assignedTo = $request->input('assigned_to')) {
            if ($assignedTo === 'unassigned') {
                $query->whereNull('assigned_to');
            } elseif ($assignedTo === 'me') {
                $query->where('assigned_to', $this->admin->id);
            } else {
                $query->where('assigned_to', $assignedTo);
            }
        }

        if ($request->input('unread') === '1') {
            $query->where('admin_read', false);
        }

        // Export
        if ($request->has('export')) {
            return $this->exportTickets($query, $request);
        }

        // Sorting
        $sortColumn = $request->input('sort', 'id');
        $sortDirection = $request->input('dir', 'desc');
        
        $query->orderBy($sortColumn, $sortDirection);

        // Pagination
        $perPage = $request->input('per_page', 25);
        $tickets = $query->paginate($perPage);

        //  CALCULATE STATUS COUNTS
        $statusCounts = [];
        $statuses = Status::active()->get();
        foreach ($statuses as $status) {
            $countQuery = Ticket::where('status_id', $status->id);
            
            // Apply same filters for counts
            if ($search = $request->input('search')) {
                $countQuery->search($search);
            }
            if ($department = $request->input('department')) {
                $countQuery->where('department_id', $department);
            }
            if ($priority = $request->input('priority')) {
                $countQuery->where('priority_id', $priority);
            }
            if ($assignedTo = $request->input('assigned_to')) {
                if ($assignedTo === 'unassigned') {
                    $countQuery->whereNull('assigned_to');
                } elseif ($assignedTo === 'me') {
                    $countQuery->where('assigned_to', $this->admin->id);
                } else {
                    $countQuery->where('assigned_to', $assignedTo);
                }
            }
            if ($request->input('unread') === '1') {
                $countQuery->where('admin_read', false);
            }
            
            $statusCounts[$status->id] = $countQuery->count();
        }

        // Transform data
        $data = $tickets->map(function ($ticket) {
            return [
                'id' => $ticket->id,
                'ticket_number' => $ticket->display_number, 
                'merged_badge' => $ticket->merged_badge,
                'subject' => $ticket->subject,
                'submitter_name' => $ticket->submitter_name,
                'department_name' => $ticket->department_name,
                'priority_badge' => $ticket->priority_badge,
                'status_badge' => $ticket->status_badge,
                'assigned_to_name' => $ticket->assigned_to_name,
                'replies_count' => $ticket->replies_count,
                'is_unread' => $ticket->is_unread,
                'last_reply_time' => $ticket->last_reply_time,
                'created_at' => $ticket->created_at->format('M d, Y g:i A'),
                '_show_url' => route('admin.support.tickets.show', $ticket->id),
                '_edit_url' => route('admin.support.tickets.edit', $ticket->id),
            ];
        });

        return response()->json([
            'data' => $data,
            'total' => $tickets->total(),
            'current_page' => $tickets->currentPage(),
            'last_page' => $tickets->lastPage(),
            'per_page' => $tickets->perPage(),
            'status_counts' => $statusCounts, // ✅ RETURN COUNTS
        ]);

    } catch (Exception $e) {
        Log::error('Tickets DataTable Failed:', ['error' => $e->getMessage()]);
        return response()->json([
            'error' => 'Failed to load tickets: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Show the form for creating a new ticket
     */
    public function create()
    {
        try {
            $departments = Department::active()->clientPortal()->ordered()->get();
            $priorities = Priority::active()->ordered()->get();
            $statuses = Status::active()->ordered()->get();
            $staff = Staff::orderBy('first_name')->get();

            return view('support::admin.tickets.create', compact(
                'departments',
                'priorities',
                'statuses',
                'staff'
            ));

        } catch (Exception $e) {
            Log::error('Ticket Create Form Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Failed to load create form: ' . $e->getMessage());
        }
    }

    /**
     * Store a newly created ticket
     */
    public function store(Request $request)
    {
        try {
            // Debug log
            Log::info('Ticket Store Request:', $request->except('attachments'));

            $validator = Validator::make($request->all(), [
                'subject' => 'required|string|max:191',
                'message' => 'required|string|max:10000',
                'department_id' => 'required|exists:support_departments,id',
                'priority_id' => 'required|exists:support_priorities,id',
                'customer_id' => 'nullable|exists:customers,id',
                'name' => 'required_without:customer_id|string|max:255',
                'email' => 'required_without:customer_id|email|max:255',
                'assigned_to' => 'nullable|exists:staffs,id',
                'cc' => 'nullable|string|max:191',
                'attachments.*' => 'nullable|file|max:10240', // 10MB
            ]);

            if ($validator->fails()) {
                Log::error('Ticket Validation Failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            DB::beginTransaction();

            // Get default status
            $defaultStatus = Status::where('is_default', true)->first();
            if (!$defaultStatus) {
                $defaultStatus = Status::active()->first();
            }

            if (!$defaultStatus) {
                throw new Exception('No active status found. Please create at least one status.');
            }

            // Create ticket
            $ticket = new Ticket();
            $ticket->subject = $request->subject;
            $ticket->message = $request->message;
            $ticket->department_id = $request->department_id;
            $ticket->priority_id = $request->priority_id;
            $ticket->status_id = $defaultStatus->id;
            $ticket->customer_id = $request->customer_id;
            
            // If customer selected, auto-fill name and email
            if ($request->customer_id) {
                $customer = Customer::find($request->customer_id);
                $ticket->name = $customer->display_name ?? $customer->name;
                $ticket->email = $customer->email;
            } else {
                $ticket->name = $request->name;
                $ticket->email = $request->email;
            }
            
            $ticket->assigned_to = $request->assigned_to;
            $ticket->cc = $request->cc;
            $ticket->opened_by = $this->admin->id;
            $ticket->admin_read = true; // Admin created, so mark as read
            $ticket->save();

            Log::info('Ticket Created Successfully:', ['id' => $ticket->id]);

            // Handle file uploads
            if ($request->hasFile('attachments')) {
                $this->handleAttachments($request->file('attachments'), $ticket->id, null);
            }

            DB::commit();

            // ✅ FIXED: Use standard Laravel redirect
            return redirect()
                ->route('admin.support.tickets.show', $ticket->id)
                ->with('success', 'Ticket created successfully!');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Ticket Creation Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to create ticket: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified ticket
     */
    public function show($id)
    {
        try {
            $ticket = Ticket::with([
                'department',
                'priority',
                'status',
                'customer',
                'assignedStaff',
                'openedByStaff',
                'replies.staff',
                'replies.customer',
                'replies.attachments',
                'attachments'
            ])->findOrFail($id);

            // Mark as read
            if (!$ticket->admin_read) {
                $ticket->markAsRead(true);
            }

            // Get available options for actions
            $departments = Department::active()->ordered()->get();
            $priorities = Priority::active()->ordered()->get();
            $statuses = Status::active()->ordered()->get();
            $staff = Staff::active()->orderBy('first_name')->get();

            $predefinedReplies = \Modules\Support\Models\PredefinedReply::active()
            ->byDepartment($ticket->department_id)
            ->orderBy('name', 'asc')
            ->get();

            return view('support::admin.tickets.show', compact(
                'ticket',
                'departments',
                'priorities',
                'statuses',
                'staff',
                'predefinedReplies'
            ));

        } catch (Exception $e) {
            Log::error('Ticket Show Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Ticket not found');
        }
    }

    /**
     * Show the form for editing the specified ticket
     */
    public function edit($id)
    {
        try {
            $ticket = Ticket::with(['department', 'priority', 'status'])->findOrFail($id);
            
            $departments = Department::active()->ordered()->get();
            $priorities = Priority::active()->ordered()->get();
            $statuses = Status::active()->ordered()->get();
            $staff = Staff::active()->orderBy('first_name')->get();

            return view('support::admin.tickets.edit', compact(
                'ticket',
                'departments',
                'priorities',
                'statuses',
                'staff'
            ));

        } catch (Exception $e) {
            Log::error('Ticket Edit Form Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Ticket not found');
        }
    }

    /**
     * Update the specified ticket
     */
    public function update(Request $request, $id)
    {
        try {
            Log::info('Ticket Update Request:', array_merge(['id' => $id], $request->all()));

            $validator = Validator::make($request->all(), [
                'subject' => 'required|string|max:191',
                'department_id' => 'required|exists:support_departments,id',
                'priority_id' => 'required|exists:support_priorities,id',
                'status_id' => 'required|exists:support_statuses,id',
                'assigned_to' => 'nullable|exists:staffs,id',
                'cc' => 'nullable|string|max:191',
            ]);

            if ($validator->fails()) {
                Log::error('Ticket Update Validation Failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput()
                    ->with('error', 'Please check the form for errors.');
            }

            $ticket = Ticket::findOrFail($id);
            
            $ticket->subject = $request->subject;
            $ticket->department_id = $request->department_id;
            $ticket->priority_id = $request->priority_id;
            $ticket->status_id = $request->status_id;
            $ticket->assigned_to = $request->assigned_to;
            $ticket->cc = $request->cc;
            $ticket->save();

            Log::info('Ticket Updated Successfully:', ['id' => $ticket->id]);

            // ✅ FIXED: Use standard Laravel redirect
            return redirect()
                ->route('admin.support.tickets.show', $ticket->id)
                ->with('success', 'Ticket updated successfully!');

        } catch (Exception $e) {
            Log::error('Ticket Update Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return back()
                ->withInput()
                ->with('error', 'Failed to update ticket: ' . $e->getMessage());
        }
    }

    /**
 * Update ticket information from sidebar
 */
public function updateTicketInfo(Request $request, $id)
{
    try {
        Log::info('Update Ticket Info Request:', array_merge(['id' => $id], $request->all()));

        $validator = Validator::make($request->all(), [
            'tags' => 'nullable|string|max:191',
            'department_id' => 'nullable|exists:support_departments,id',
            'assigned_to' => 'nullable|exists:staffs,id',
            'priority_id' => 'nullable|exists:support_priorities,id',
            'status_id' => 'nullable|exists:support_statuses,id',
            'cc' => 'nullable|string|max:191',
            'merge_ticket_id' => 'nullable|string|max:191',
        ]);

        if ($validator->fails()) {
            Log::error('Update Ticket Info Validation Failed:', $validator->errors()->toArray());
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        $ticket = Ticket::findOrFail($id);
        
        // Update only provided fields
        if ($request->has('tags')) {
            $ticket->tags = $request->tags;
        }
        
        if ($request->has('department_id')) {
            $ticket->department_id = $request->department_id;
        }
        
        if ($request->has('assigned_to')) {
            $ticket->assigned_to = $request->assigned_to;
        }
        
        if ($request->has('priority_id')) {
            $ticket->priority_id = $request->priority_id;
        }
        
        if ($request->has('status_id')) {
            $ticket->status_id = $request->status_id;
        }
        
        if ($request->has('cc')) {
            $ticket->cc = $request->cc;
        }
        
        //Handle merge ticket IDs
        if ($request->has('merge_ticket_id') && !empty($request->merge_ticket_id)) {
            $mergeIds = array_map('trim', explode(',', $request->merge_ticket_id));
            $mergeIds = array_filter($mergeIds, 'is_numeric');
            
            $mergedCount = 0;
            $errors = [];
            
            foreach ($mergeIds as $mergeId) {
                try {
                    $this->mergeTicketInto((int)$mergeId, $ticket->id);
                    $mergedCount++;
                } catch (Exception $e) {
                    $errors[] = "Ticket #{$mergeId}: " . $e->getMessage();
                }
            }
            
            if ($mergedCount > 0) {
                Log::info("Merged {$mergedCount} tickets into #{$ticket->id}");
            }
            
            if (!empty($errors)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Some tickets could not be merged: ' . implode(', ', $errors)
                ], 400);
            }
        }

        $ticket->save();

        Log::info('Ticket Info Updated Successfully:', ['id' => $ticket->id]);

        return response()->json([
            'success' => true,
            'message' => 'Ticket information updated successfully'
        ]);

    } catch (Exception $e) {
        Log::error('Update Ticket Info Failed:', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to update ticket information: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Remove the specified ticket
     */
    public function destroy($id)
    {
        try {
            $ticket = Ticket::findOrFail($id);
            $ticket->delete();

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Ticket deleted successfully'
                ]);
            }

            return redirect()
                ->route('admin.support.tickets.index')
                ->with('success', 'Ticket deleted successfully!');

        } catch (Exception $e) {
            Log::error('Ticket Delete Failed:', ['error' => $e->getMessage()]);
            
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete ticket: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Failed to delete ticket: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete tickets
     */
    public function bulkDelete(Request $request)
    {
        try {
            $ids = $request->input('ids', []);
            
            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tickets selected'
                ], 400);
            }

            $deleted = Ticket::whereIn('id', $ids)->delete();

            return response()->json([
                'success' => true,
                'message' => 'Deleted ' . $deleted . ' tickets successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Bulk Delete Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete tickets: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add reply to ticket
     */
    public function addReply(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'message' => 'required|string|max:10000',
                'attachments.*' => 'nullable|file|max:10240', // 10MB
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            DB::beginTransaction();

            $ticket = Ticket::findOrFail($id);

            // Create reply
            $reply = new TicketReply();
            $reply->ticket_id = $ticket->id;
            $reply->message = $request->message;
            $reply->staff_id = $this->admin->id;
            $reply->replied_at = now();
            $reply->save();

            // Handle file uploads
            if ($request->hasFile('attachments')) {
                $this->handleAttachments($request->file('attachments'), $ticket->id, $reply->id);
                $reply->has_attachment = true;
                $reply->save();
            }

            // Update ticket last_reply and mark client as unread
            $ticket->last_reply = now();
            $ticket->client_read = false;
            $ticket->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Reply added successfully',
                'reply' => [
                    'id' => $reply->id,
                    'message' => $reply->message,
                    'submitter_name' => $reply->submitter_name,
                    'replied_time' => $reply->replied_time,
                    'is_staff_reply' => $reply->is_staff_reply,
                ]
            ]);

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Add Reply Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to add reply: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Assign ticket to staff
     */
    public function assign(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'staff_id' => 'nullable|exists:staffs,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $ticket = Ticket::findOrFail($id);
            $ticket->assigned_to = $request->staff_id;
            $ticket->save();

            return response()->json([
                'success' => true,
                'message' => 'Ticket assigned successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Assign Ticket Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to assign ticket: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Change ticket status
     */
    public function changeStatus(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'status_id' => 'required|exists:support_statuses,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $ticket = Ticket::findOrFail($id);
            $ticket->status_id = $request->status_id;
            $ticket->save();

            return response()->json([
                'success' => true,
                'message' => 'Status changed successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Change Status Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to change status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Change ticket priority
     */
    public function changePriority(Request $request, $id)
    {
        try {
            $validator = Validator::make($request->all(), [
                'priority_id' => 'required|exists:support_priorities,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $ticket = Ticket::findOrFail($id);
            $ticket->priority_id = $request->priority_id;
            $ticket->save();

            return response()->json([
                'success' => true,
                'message' => 'Priority changed successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Change Priority Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to change priority: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark ticket as read
     */
    public function markAsRead($id)
    {
        try {
            $ticket = Ticket::findOrFail($id);
            $ticket->markAsRead(true);

            return response()->json([
                'success' => true,
                'message' => 'Marked as read'
            ]);

        } catch (Exception $e) {
            Log::error('Mark As Read Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to mark as read: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk assign tickets
     */
    public function bulkAssign(Request $request)
    {
        try {
            $ids = $request->input('ids', []);
            $staffId = $request->input('staff_id');

            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tickets selected'
                ], 400);
            }

            $updated = Ticket::whereIn('id', $ids)->update(['assigned_to' => $staffId]);

            return response()->json([
                'success' => true,
                'message' => 'Assigned ' . $updated . ' tickets successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Bulk Assign Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to assign tickets: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk change status
     */
    public function bulkChangeStatus(Request $request)
    {
        try {
            $ids = $request->input('ids', []);
            $statusId = $request->input('status_id');

            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tickets selected'
                ], 400);
            }

            $updated = Ticket::whereIn('id', $ids)->update(['status_id' => $statusId]);

            return response()->json([
                'success' => true,
                'message' => 'Updated ' . $updated . ' tickets successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Bulk Change Status Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update tickets: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk change priority
     */
    public function bulkChangePriority(Request $request)
    {
        try {
            $ids = $request->input('ids', []);
            $priorityId = $request->input('priority_id');

            if (empty($ids)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tickets selected'
                ], 400);
            }

            $updated = Ticket::whereIn('id', $ids)->update(['priority_id' => $priorityId]);

            return response()->json([
                'success' => true,
                'message' => 'Updated ' . $updated . ' tickets successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Bulk Change Priority Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to update tickets: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search customers (autocomplete)
     */
    public function searchCustomers(Request $request)
    {
        try {
            $search = $request->input('q', '');
            
            $customers = Customer::where('active', true)
                ->where(function($q) use ($search) {
                    $q->where('name', 'LIKE', "%{$search}%")
                      ->orWhere('email', 'LIKE', "%{$search}%")
                      ->orWhere('company', 'LIKE', "%{$search}%");
                })
                ->limit(10)
                ->get()
                ->map(function($customer) {
                    return [
                        'id' => $customer->id,
                        'text' => ($customer->display_name ?? $customer->name) . ' (' . $customer->email . ')',
                        'name' => $customer->display_name ?? $customer->name,
                        'email' => $customer->email,
                    ];
                });

            return response()->json($customers);

        } catch (Exception $e) {
            Log::error('Search Customers Failed:', ['error' => $e->getMessage()]);
            return response()->json([]);
        }
    }

    /**
     * Search staff (autocomplete)
     */
    public function searchStaff(Request $request)
    {
        try {
            $search = $request->input('q', '');
            
            $staff = Staff::where(function($q) use ($search) {
                    $q->where('first_name', 'LIKE', "%{$search}%")
                      ->orWhere('last_name', 'LIKE', "%{$search}%")
                      ->orWhere('email', 'LIKE', "%{$search}%");
                })
                ->limit(10)
                ->get()
                ->map(function($staff) {
                    return [
                        'id' => $staff->id,
                        'text' => $staff->full_name,
                    ];
                });

            return response()->json($staff);

        } catch (Exception $e) {
            Log::error('Search Staff Failed:', ['error' => $e->getMessage()]);
            return response()->json([]);
        }
    }

    /**
     * Get customer info (AJAX)
     */
    public function getCustomerInfo($customerId)
    {
        try {
            $customer = Customer::findOrFail($customerId);

            return response()->json([
                'success' => true,
                'customer' => [
                    'id' => $customer->id,
                    'name' => $customer->display_name ?? $customer->name,
                    'email' => $customer->email,
                    'phone' => $customer->phone,
                    'company' => $customer->company,
                ]
            ]);

        } catch (Exception $e) {
            Log::error('Get Customer Info Failed:', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Customer not found'
            ], 404);
        }
    }

    /**
     * Download attachment
     */
    public function downloadAttachment($attachmentId)
    {
        try {
            $attachment = TicketAttachment::findOrFail($attachmentId);

            if (!Storage::disk('public')->exists($attachment->file_path)) {
                return back()->with('error', 'File not found');
            }

            return Storage::disk('public')->download(
                $attachment->file_path,
                $attachment->file_name
            );

        } catch (Exception $e) {
            Log::error('Download Attachment Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Failed to download file');
        }
    }

    /**
     * Export tickets
     */
    public function export(Request $request)
    {
        try {
            $query = Ticket::with(['department', 'priority', 'status', 'assignedStaff', 'customer']);

            // Apply same filters as dataTable
            if ($search = $request->input('search')) {
                $query->search($search);
            }

            if ($filters = $request->input('filters')) {
                $filters = json_decode($filters, true);
                // Apply filters same as dataTable method
            }

            return $this->exportTickets($query, $request);

        } catch (Exception $e) {
            Log::error('Export Tickets Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Failed to export tickets: ' . $e->getMessage());
        }
    }

    /**
     * Handle ticket export to CSV
     */
    private function exportTickets($query, $request)
    {
        $tickets = $query->get();

        $filename = 'tickets_' . date('Y-m-d_His') . '.csv';

        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($tickets) {
            $file = fopen('php://output', 'w');
            
            // Headers
            fputcsv($file, [
                'ID',
                'Ticket Number',
                'Subject',
                'Submitter',
                'Email',
                'Department',
                'Priority',
                'Status',
                'Assigned To',
                'Created At',
                'Last Reply',
                'Replies Count',
            ]);

            // Data
            foreach ($tickets as $ticket) {
                fputcsv($file, [
                    $ticket->id,
                    $ticket->display_number,
                    $ticket->subject,
                    $ticket->submitter_name,
                    $ticket->submitter_email,
                    $ticket->department_name,
                    $ticket->priority_name,
                    $ticket->status_name,
                    $ticket->assigned_to_name,
                    $ticket->created_at->format('Y-m-d H:i:s'),
                    $ticket->last_reply ? $ticket->last_reply->format('Y-m-d H:i:s') : 'No Reply',
                    $ticket->replies_count,
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Handle file attachments
     */
    private function handleAttachments($files, $ticketId, $replyId = null)
    {
        foreach ($files as $file) {
            if ($file->isValid()) {
                $fileName = time() . '_' . $file->getClientOriginalName();
                $filePath = 'support_tickets/' . $ticketId . '/' . $fileName;
                
                Storage::disk('public')->putFileAs(
                    'support_tickets/' . $ticketId,
                    $file,
                    $fileName
                );

                TicketAttachment::create([
                    'ticket_id' => $ticketId,
                    'reply_id' => $replyId,
                    'file_name' => $file->getClientOriginalName(),
                    'file_path' => $filePath,
                    'file_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                    'uploaded_at' => now(),
                ]);
            }
        }
    }

/**
 * Merge ticket(s) into current ticket
 * Route: POST /admin/support/tickets/{id}/merge
 * Body: { "merge_from": "5" } or { "merge_from": "5,6,7" }
 */
public function merge(Request $request, $id)
{
    try {
        $validator = Validator::make($request->all(), [
            'merge_from' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        DB::beginTransaction();

        $parentTicket = Ticket::findOrFail($id);

        // Check parent ticket is valid
        if ($parentTicket->isMerged()) {
            throw new Exception('Cannot merge into a merged ticket');
        }

        // Parse comma-separated IDs
        $mergeFromIds = array_map('trim', explode(',', $request->merge_from));
        $mergeFromIds = array_filter($mergeFromIds, 'is_numeric');
        $mergeFromIds = array_map('intval', $mergeFromIds);

        if (empty($mergeFromIds)) {
            throw new Exception('No valid ticket IDs provided');
        }

        // Remove current ticket ID if present
        $mergeFromIds = array_diff($mergeFromIds, [$parentTicket->id]);

        if (empty($mergeFromIds)) {
            throw new Exception('Cannot merge ticket into itself');
        }

        $mergedCount = 0;
        $errors = [];

        foreach ($mergeFromIds as $childId) {
            try {
                $childTicket = Ticket::findOrFail($childId);

                // Validate child can be merged
                if ($childTicket->isMerged()) {
                    $errors[] = "Ticket {$childTicket->display_number} is already merged";
                    continue;
                }

                if ($childTicket->isClosed()) {
                    $errors[] = "Ticket {$childTicket->display_number} is closed";
                    continue;
                }

                // Get closed status
                $closedStatus = Status::where('name', 'Closed')->first();
                if (!$closedStatus) {
                    $closedStatus = Status::active()->orderBy('sort_order', 'desc')->first();
                }

                // Move replies to parent ticket
                TicketReply::where('ticket_id', $childTicket->id)
                    ->update(['ticket_id' => $parentTicket->id]);

                // Move attachments to parent ticket
                TicketAttachment::where('ticket_id', $childTicket->id)
                    ->update(['ticket_id' => $parentTicket->id]);

                // Mark child as merged and close it
                $childTicket->is_merged = true;
                $childTicket->merged_ticket_id = $parentTicket->id;
                $childTicket->status_id = $closedStatus->id;
                $childTicket->save();

                $mergedCount++;

                Log::info("Merged ticket {$childTicket->display_number} into {$parentTicket->display_number}");

            } catch (Exception $e) {
                $errors[] = "Ticket #{$childId}: " . $e->getMessage();
                Log::error("Failed to merge ticket #{$childId}: " . $e->getMessage());
            }
        }

        DB::commit();

        // Build response message
        if ($mergedCount === 0) {
            return response()->json([
                'success' => false,
                'message' => 'No tickets were merged. ' . implode(', ', $errors)
            ], 400);
        }

        $message = "Successfully merged {$mergedCount} ticket(s) into {$parentTicket->display_number}";
        
        if (!empty($errors)) {
            $message .= '. Some tickets could not be merged: ' . implode(', ', $errors);
        }

        return response()->json([
            'success' => true,
            'message' => $message,
            'merged_count' => $mergedCount
        ]);

    } catch (Exception $e) {
        DB::rollBack();
        Log::error('Merge Ticket Failed:', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to merge tickets: ' . $e->getMessage()
        ], 500);
    }
}

    /**
 * Merge a ticket into another ticket (reusable helper)
 * 
 * @param int $childId - Ticket to be merged
 * @param int $parentId - Ticket to merge into
 */
private function mergeTicketInto(int $childId, int $parentId): void
{
    $childTicket = Ticket::findOrFail($childId);
    $parentTicket = Ticket::findOrFail($parentId);
    
    // Validation
    if ($childTicket->id === $parentTicket->id) {
        throw new Exception('Cannot merge ticket into itself');
    }
    
    if ($childTicket->isMerged()) {
        throw new Exception("Ticket {$childTicket->display_number} is already merged");
    }
    
    if ($childTicket->isClosed()) {
        throw new Exception("Ticket {$childTicket->display_number} is closed");
    }
    
    if ($parentTicket->isMerged()) {
        throw new Exception("Cannot merge into a merged ticket");
    }
    
    // Move replies
    TicketReply::where('ticket_id', $childTicket->id)
        ->update(['ticket_id' => $parentTicket->id]);
    
    // Move attachments
    TicketAttachment::where('ticket_id', $childTicket->id)
        ->update(['ticket_id' => $parentTicket->id]);
    
    // Get closed status
    $closedStatus = Status::where('name', 'Closed')->first();
    if (!$closedStatus) {
        $closedStatus = Status::active()->first();
    }
    
    // Mark as merged and close
    $childTicket->is_merged = true;
    $childTicket->merged_ticket_id = $parentTicket->id;
    $childTicket->status_id = $closedStatus->id;
    $childTicket->save();
    
    Log::info("Merged ticket {$childTicket->display_number} into {$parentTicket->display_number}");
}

    /**
     * Get other tickets from the same customer
     */
    public function getOtherTickets($id)
    {
        try {
            $ticket = Ticket::findOrFail($id);
            
            // Build query for other tickets from same customer
            $query = Ticket::with(['department', 'priority', 'status'])
                ->where('id', '!=', $ticket->id); // Exclude current ticket
            
            // Filter by customer if exists
            if ($ticket->customer_id) {
                $query->where('customer_id', $ticket->customer_id);
            } else {
                // If no customer_id, match by email
                $query->where('email', $ticket->email);
            }
            
            $otherTickets = $query->orderBy('created_at', 'desc')
                ->limit(10)
                ->get()
                ->map(function($t) {
                    return [
                        'id' => $t->id,
                        'subject' => $t->subject,
                        'department_name' => $t->department_name,
                        'status_name' => $t->status_name,
                        'status_color' => $t->status->color ?? '#6B7280',
                        'priority_name' => $t->priority_name,
                        'priority_color' => $t->priority->color ?? '#6B7280',
                        'last_reply_time' => $t->last_reply_time,
                        'created_at' => $t->created_at->format('M d, Y'),
                        'url' => route('admin.support.tickets.show', $t->id),
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => $otherTickets
            ]);

        } catch (Exception $e) {
            Log::error('Get Other Tickets Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to load other tickets: ' . $e->getMessage()
            ], 500);
        }
    }

}