<?php

namespace Modules\Support\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Support\Models\Ticket;
use Modules\Support\Models\TicketNote;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Exception;

class TicketNoteController extends AdminController
{
    /**
     * Get all notes for a ticket (AJAX)
     */
    public function index($ticketId)
    {
        try {
            $ticket = Ticket::findOrFail($ticketId);
            
            $notes = TicketNote::where('ticket_id', $ticket->id)
                ->with('staff')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function($note) {
                    return [
                        'id' => $note->id,
                        'note' => $note->note,
                        'staff_name' => $note->staff ? $note->staff->full_name : 'Unknown',
                        'created_at' => $note->created_at->format('M d, Y g:i A'),
                    ];
                });

            return response()->json([
                'success' => true,
                'message' => 'Notes loaded successfully',
                'notes' => $notes
            ]);

        } catch (Exception $e) {
            Log::error('Load Notes Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to load notes: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a new note
     */
    public function store(Request $request, $ticketId)
    {
        try {
            Log::info('Add Note Request:', ['ticket_id' => $ticketId, 'note' => $request->note]);

            $validator = Validator::make($request->all(), [
                'note' => 'required|string|max:5000',
            ]);

            if ($validator->fails()) {
                Log::error('Note Validation Failed:', $validator->errors()->toArray());
                
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $ticket = Ticket::findOrFail($ticketId);

            $note = TicketNote::create([
                'ticket_id' => $ticket->id,
                'staff_id' => $this->admin->id,
                'note' => $request->note,
            ]);

            Log::info('Note Added Successfully:', ['note_id' => $note->id]);

            return response()->json([
                'success' => true,
                'message' => 'Note added successfully',
                'note' => [
                    'id' => $note->id,
                    'note' => $note->note,
                    'staff_name' => $this->admin->first_name . ' ' . $this->admin->last_name,
                    'created_at' => $note->created_at->format('M d, Y g:i A'),
                ]
            ]);

        } catch (Exception $e) {
            Log::error('Add Note Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to add note: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a note
     */
    public function destroy($ticketId, $noteId)
    {
        try {
            $note = TicketNote::where('ticket_id', $ticketId)
                ->where('id', $noteId)
                ->firstOrFail();

            // Only allow deletion by note creator or super admin
            if ($note->staff_id !== $this->admin->id && !$this->admin->hasRole('super-admin')) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized to delete this note'
                ], 403);
            }

            $note->delete();

            Log::info('Note Deleted Successfully:', ['note_id' => $noteId]);

            return response()->json([
                'success' => true,
                'message' => 'Note deleted successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Delete Note Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete note: ' . $e->getMessage()
            ], 500);
        }
    }
}