<?php

namespace Modules\Support\Http\Controllers;

use App\Http\Controllers\Admin\AdminController;
use Modules\Support\Models\Ticket;
use Modules\Support\Models\TicketReminder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Exception;

class TicketReminderController extends AdminController
{
    /**
     * Get all reminders for a ticket (View)
     */
    public function index($ticketId)
    {
        try {
            $ticket = Ticket::findOrFail($ticketId);
            
            return view('support::admin.tickets.reminders', compact('ticket'));

        } catch (Exception $e) {
            Log::error('Reminders Index Failed:', ['error' => $e->getMessage()]);
            return back()->with('error', 'Ticket not found');
        }
    }

    /**
     * DataTable endpoint for reminders
     */
    public function dataTable(Request $request, $ticketId)
    {
        try {
            $query = TicketReminder::where('ticket_id', $ticketId)
                ->with('staff');

            // Search
            if ($search = $request->input('search')) {
                $query->where('description', 'LIKE', "%{$search}%");
            }

            // Sorting
            $sortColumn = $request->input('sort', 'remind_at');
            $sortDirection = $request->input('dir', 'asc');
            $query->orderBy($sortColumn, $sortDirection);

            // Pagination
            $perPage = $request->input('per_page', 25);
            $reminders = $query->paginate($perPage);

            // Transform data
            $data = $reminders->map(function ($reminder) {
                return [
                    'id' => $reminder->id,
                    'description' => $reminder->description,
                    'remind_at' => $reminder->remind_at->format('M d, Y g:i A'),
                    'is_notified' => $reminder->is_notified,
                    'staff_name' => $reminder->staff ? $reminder->staff->full_name : 'N/A',
                    'created_at' => $reminder->created_at->format('M d, Y'),
                ];
            });

            return response()->json([
                'data' => $data,
                'total' => $reminders->total(),
                'current_page' => $reminders->currentPage(),
                'last_page' => $reminders->lastPage(),
                'per_page' => $reminders->perPage(),
            ]);

        } catch (Exception $e) {
            Log::error('Reminders DataTable Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'error' => 'Failed to load reminders: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a new reminder
     */
    public function store(Request $request, $ticketId)
    {
        try {
            Log::info('Set Reminder Request:', ['ticket_id' => $ticketId, 'data' => $request->all()]);

            $validator = Validator::make($request->all(), [
                'description' => 'required|string|max:500',
                'remind_at' => 'required|date|after:now',
                'notify_staff_id' => 'nullable|exists:staffs,id',
            ]);

            if ($validator->fails()) {
                Log::error('Reminder Validation Failed:', $validator->errors()->toArray());
                
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $ticket = Ticket::findOrFail($ticketId);

            $reminder = TicketReminder::create([
                'ticket_id' => $ticket->id,
                'staff_id' => $request->notify_staff_id ?? $this->admin->id,
                'description' => $request->description,
                'remind_at' => $request->remind_at,
                'is_notified' => false,
            ]);

            Log::info('Reminder Set Successfully:', ['reminder_id' => $reminder->id]);

            return response()->json([
                'success' => true,
                'message' => 'Reminder set successfully',
                'reminder' => [
                    'id' => $reminder->id,
                    'description' => $reminder->description,
                    'remind_at' => $reminder->remind_at->format('M d, Y g:i A'),
                ]
            ]);

        } catch (Exception $e) {
            Log::error('Set Reminder Failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to set reminder: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a reminder
     */
    public function destroy($ticketId, $reminderId)
    {
        try {
            $reminder = TicketReminder::where('ticket_id', $ticketId)
                ->where('id', $reminderId)
                ->firstOrFail();

            $reminder->delete();

            Log::info('Reminder Deleted Successfully:', ['reminder_id' => $reminderId]);

            return response()->json([
                'success' => true,
                'message' => 'Reminder deleted successfully'
            ]);

        } catch (Exception $e) {
            Log::error('Delete Reminder Failed:', ['error' => $e->getMessage()]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete reminder: ' . $e->getMessage()
            ], 500);
        }
    }
}