<?php

namespace Modules\Support\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Department extends Model
{
    protected $table = 'support_departments';

    protected $fillable = [
        'name',
        'description',
        'is_active',
        'show_on_client_portal',
        'department_order',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'show_on_client_portal' => 'boolean',
        'department_order' => 'integer',
    ];

    /**
     * Relationships
     */
    public function tickets(): HasMany
    {
        return $this->hasMany(Ticket::class, 'department_id');
    }

    public function predefinedReplies(): HasMany
    {
        return $this->hasMany(PredefinedReply::class, 'department_id');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeClientPortal($query)
    {
        return $query->where('show_on_client_portal', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('department_order', 'asc');
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
              ->orWhere('description', 'LIKE', "%{$search}%");
        });
    }

    /**
     * Accessors
     */
    public function getStatusLabelAttribute(): string
    {
        return $this->is_active 
            ? '<span class="badge badge-success">Active</span>' 
            : '<span class="badge badge-danger">Inactive</span>';
    }

    public function getTicketsCountAttribute(): int
    {
        return $this->tickets()->count();
    }

    public function getActiveTicketsCountAttribute(): int
    {
        return $this->tickets()
            ->whereHas('status', function($q) {
                $q->where('name', '!=', 'Closed');
            })
            ->count();
    }
}