<?php

namespace Modules\Support\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PredefinedReply extends Model
{
    protected $table = 'support_predefined_replies';

    protected $fillable = [
        'name',
        'message',
        'department_id',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'department_id' => 'integer',
    ];

    /**
     * Relationships
     */
    public function department(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'department_id');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByDepartment($query, $departmentId)
    {
        if ($departmentId) {
            return $query->where(function($q) use ($departmentId) {
                $q->where('department_id', $departmentId)
                  ->orWhereNull('department_id');
            });
        }
        
        return $query;
    }

    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
              ->orWhere('message', 'LIKE', "%{$search}%");
        });
    }

    /**
     * Accessors
     */
    public function getStatusLabelAttribute(): string
    {
        return $this->is_active 
            ? '<span class="badge badge-success">Active</span>' 
            : '<span class="badge badge-danger">Inactive</span>';
    }

    public function getDepartmentNameAttribute(): string
    {
        return $this->department ? $this->department->name : 'All Departments';
    }

    public function getMessagePreviewAttribute(): string
    {
        return \Illuminate\Support\Str::limit(strip_tags($this->message), 100);
    }
}