<?php

namespace Modules\Support\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Priority extends Model
{
    protected $table = 'support_priorities';

    protected $fillable = [
        'name',
        'color',
        'priority_order',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'priority_order' => 'integer',
    ];

    /**
     * Relationships
     */
    public function tickets(): HasMany
    {
        return $this->hasMany(Ticket::class, 'priority_id');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('priority_order', 'asc');
    }

    public function scopeSearch($query, $search)
    {
        return $query->where('name', 'LIKE', "%{$search}%");
    }

    /**
     * Accessors
     */
    public function getStatusLabelAttribute(): string
    {
        return $this->is_active 
            ? '<span class="badge badge-success">Active</span>' 
            : '<span class="badge badge-danger">Inactive</span>';
    }

    public function getBadgeHtmlAttribute(): string
    {
        return sprintf(
            '<span class="badge" style="background-color: %s; color: #fff;">%s</span>',
            $this->color,
            e($this->name)
        );
    }

    public function getTicketsCountAttribute(): int
    {
        return $this->tickets()->count();
    }
}