<?php

namespace Modules\Support\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Status extends Model
{
    protected $table = 'support_statuses';

    protected $fillable = [
        'name',
        'status_color',
        'status_order',
        'is_default',
        'allow_client_change',
        'is_active',
    ];

    protected $casts = [
        'is_default' => 'boolean',
        'allow_client_change' => 'boolean',
        'is_active' => 'boolean',
        'status_order' => 'integer',
    ];

    /**
     * Relationships
     */
    public function tickets(): HasMany
    {
        return $this->hasMany(Ticket::class, 'status_id');
    }

    /**
     * Scopes
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    public function scopeClientChangeable($query)
    {
        return $query->where('allow_client_change', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('status_order', 'asc');
    }

    public function scopeSearch($query, $search)
    {
        return $query->where('name', 'LIKE', "%{$search}%");
    }

    /**
     * Accessors
     */
    public function getStatusLabelAttribute(): string
    {
        return $this->is_active 
            ? '<span class="badge badge-success">Active</span>' 
            : '<span class="badge badge-danger">Inactive</span>';
    }

    public function getBadgeHtmlAttribute(): string
    {
        return sprintf(
            '<span class="badge" style="background-color: %s; color: #fff;">%s</span>',
            $this->status_color ?? '#6c757d',
            e($this->name)
        );
    }

    public function getTicketsCountAttribute(): int
    {
        return $this->tickets()->count();
    }

    public function getDefaultLabelAttribute(): string
    {
        return $this->is_default 
            ? '<span class="badge badge-primary">Default</span>' 
            : '';
    }
}