<?php

namespace Modules\Support\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Ticket extends Model
{
    protected $table = 'support_tickets';

    protected $fillable = [
        'ticket_number',
        'ticket_key',
        'subject',
        'message',
        'customer_id',
        'name',
        'email',
        'contact_id',
        'department_id',
        'priority_id',
        'status_id',
        'service_id',
        'project_id',
        'assigned_to',
        'admin_replying',
        'staff_id_replying',
        'opened_by',
        'merged_ticket_id',
        'is_merged',
        'cc',
        'client_read',
        'admin_read',
        'last_reply',
        'tags',
    ];

    protected $casts = [
        'customer_id' => 'integer',
        'contact_id' => 'integer',
        'department_id' => 'integer',
        'priority_id' => 'integer',
        'status_id' => 'integer',
        'service_id' => 'integer',
        'project_id' => 'integer',
        'assigned_to' => 'integer',
        'admin_replying' => 'integer',
        'staff_id_replying' => 'integer',
        'opened_by' => 'integer',
        'merged_ticket_id' => 'integer',
        'is_merged' => 'boolean',
        'client_read' => 'boolean',
        'admin_read' => 'boolean',
        'last_reply' => 'datetime',
    ];

    /**
     * Boot method - Auto-generate keys and numbers
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($ticket) {
            // Generate ticket_key if empty
            if (empty($ticket->ticket_key)) {
                $ticket->ticket_key = static::generateUniqueTicketKey();
            }
            
            // Generate ticket_number if empty
            if (empty($ticket->ticket_number)) {
                $ticket->ticket_number = static::generateTicketNumber();
            }
        });
    }

    /**
     * Generate unique ticket key (32 char random - for public access)
     */
    public static function generateUniqueTicketKey(): string
    {
        do {
            $key = Str::random(32);
        } while (static::where('ticket_key', $key)->exists());

        return $key;
    }

    /**
     *Generate ticket number in format: TCK-YYYYMMDD-0001
     */
    public static function generateTicketNumber(): string
    {
        $date = date('Ymd'); // 20250107
        $prefix = 'TCK-' . $date . '-';
        
        // Get last ticket number for today
        $lastTicket = static::where('ticket_number', 'LIKE', $prefix . '%')
            ->orderBy('ticket_number', 'desc')
            ->first();
        
        if ($lastTicket) {
            // Extract sequence number from last ticket
            $lastNumber = (int) substr($lastTicket->ticket_number, -4);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }
        
        // Format: TCK-20250107-0001
        return $prefix . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Relationships
     */
    public function department(): BelongsTo
    {
        return $this->belongsTo(Department::class, 'department_id');
    }

    public function priority(): BelongsTo
    {
        return $this->belongsTo(Priority::class, 'priority_id');
    }

    public function status(): BelongsTo
    {
        return $this->belongsTo(Status::class, 'status_id');
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Customer::class, 'customer_id');
    }

    public function assignedStaff(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Admin\Staff::class, 'assigned_to');
    }

    public function openedByStaff(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Admin\Staff::class, 'opened_by');
    }

    public function replyingStaff(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Admin\Staff::class, 'staff_id_replying');
    }

    //Parent ticket this was merged INTO
    public function mergedIntoTicket(): BelongsTo
    {
        return $this->belongsTo(Ticket::class, 'merged_ticket_id');
    }

    //Tickets that were merged INTO this one
    public function mergedTickets(): HasMany
    {
        return $this->hasMany(Ticket::class, 'merged_ticket_id');
    }

    public function replies(): HasMany
    {
        return $this->hasMany(TicketReply::class, 'ticket_id')
            ->orderBy('replied_at', 'asc');
    }

    public function attachments(): HasMany
    {
        return $this->hasMany(TicketAttachment::class, 'ticket_id')
            ->whereNull('reply_id');
    }

    public function allAttachments(): HasMany
    {
        return $this->hasMany(TicketAttachment::class, 'ticket_id');
    }

    public function notes(): HasMany
    {
        return $this->hasMany(TicketNote::class, 'ticket_id')->orderBy('created_at', 'desc');
    }

    public function reminders(): HasMany
    {
        return $this->hasMany(TicketReminder::class, 'ticket_id')->orderBy('remind_at', 'asc');
    }

    /**
     * Scopes
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('subject', 'LIKE', "%{$search}%")
              ->orWhere('message', 'LIKE', "%{$search}%")
              ->orWhere('ticket_key', 'LIKE', "%{$search}%")
              ->orWhere('ticket_number', 'LIKE', "%{$search}%")
              ->orWhere('id', 'LIKE', "%{$search}%")
              ->orWhere('name', 'LIKE', "%{$search}%")
              ->orWhere('email', 'LIKE', "%{$search}%");
        });
    }

    public function scopeByDepartment($query, $departmentId)
    {
        return $query->where('department_id', $departmentId);
    }

    public function scopeByPriority($query, $priorityId)
    {
        return $query->where('priority_id', $priorityId);
    }

    public function scopeByStatus($query, $statusId)
    {
        return $query->where('status_id', $statusId);
    }

    public function scopeAssignedTo($query, $staffId)
    {
        return $query->where('assigned_to', $staffId);
    }

    public function scopeUnassigned($query)
    {
        return $query->whereNull('assigned_to')->orWhere('assigned_to', 0);
    }

    public function scopeByCustomer($query, $customerId)
    {
        return $query->where('customer_id', $customerId);
    }

    public function scopeUnread($query)
    {
        return $query->where('admin_read', false);
    }

    public function scopeOpen($query)
    {
        return $query->whereHas('status', function($q) {
            $q->where('name', '!=', 'Closed');
        });
    }

    public function scopeClosed($query)
    {
        return $query->whereHas('status', function($q) {
            $q->where('name', 'Closed');
        });
    }

    // Not merged tickets only
    public function scopeNotMerged($query)
    {
        return $query->where('is_merged', false);
    }

    // Merged tickets only
    public function scopeMerged($query)
    {
        return $query->where('is_merged', true);
    }

    /**
     * Accessors
     */
    public function getDepartmentNameAttribute(): string
    {
        return $this->department ? $this->department->name : 'N/A';
    }

    public function getPriorityNameAttribute(): string
    {
        return $this->priority ? $this->priority->name : 'N/A';
    }

    public function getStatusNameAttribute(): string
    {
        return $this->status ? $this->status->name : 'N/A';
    }

    public function getStatusBadgeAttribute(): string
    {
        return $this->status ? $this->status->badge_html : '';
    }

    public function getPriorityBadgeAttribute(): string
    {
        return $this->priority ? $this->priority->badge_html : '';
    }

    public function getAssignedToNameAttribute(): string
    {
        return $this->assignedStaff 
            ? $this->assignedStaff->first_name . ' ' . $this->assignedStaff->last_name
            : 'Unassigned';
    }

    public function getSubmitterNameAttribute(): string
    {
        if ($this->customer) {
            return $this->customer->company ?? $this->customer->name ?? $this->name;
        }
        return $this->name ?? 'Unknown';
    }

    public function getSubmitterEmailAttribute(): string
    {
        return $this->customer ? $this->customer->email : ($this->email ?? 'N/A');
    }

    public function getIsUnreadAttribute(): bool
    {
        return !$this->admin_read;
    }

    public function getIsAssignedAttribute(): bool
    {
        return !empty($this->assigned_to) && $this->assigned_to > 0;
    }

    public function getRepliesCountAttribute(): int
    {
        return $this->replies()->count();
    }

    public function getAttachmentsCountAttribute(): int
    {
        return $this->allAttachments()->count();
    }

    public function getLastReplyTimeAttribute(): ?string
    {
        return $this->last_reply ? $this->last_reply->diffForHumans() : null;
    }

    public function getPublicUrlAttribute(): string
    {
        return route('client.support.view', ['key' => $this->ticket_key]);
    }

    // Display ticket number or fallback
    public function getDisplayNumberAttribute(): string
    {
        return $this->ticket_number ?? '#' . $this->id;
    }

    // Merged status badge
    public function getMergedBadgeAttribute(): string
    {
        if (!$this->is_merged) {
            return '';
        }
        
        $parentNumber = $this->mergedIntoTicket ? $this->mergedIntoTicket->display_number : 'N/A';
        return '<span style="display: inline-flex; align-items: center; gap: 4px; padding: 4px 8px; background: #F59E0B; color: white; border-radius: 4px; font-size: 12px; font-weight: 600;">
            <svg width="12" height="12" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4"/>
            </svg>
            Merged into ' . $parentNumber . '
        </span>';
    }

    /**
     * Helper Methods
     */
    public function markAsRead(bool $isAdmin = true): void
    {
        if ($isAdmin) {
            $this->update(['admin_read' => true]);
        } else {
            $this->update(['client_read' => true]);
        }
    }

    public function updateLastReply(): void
    {
        $this->update(['last_reply' => now()]);
    }

    public function isClosed(): bool
    {
        return $this->status && $this->status->name === 'Closed';
    }

    public function isOpen(): bool
    {
        return !$this->isClosed();
    }

    //  Check if ticket is merged
    public function isMerged(): bool
    {
        return (bool) $this->is_merged;
    }

    // Can this ticket be merged?
    public function canBeMerged(): bool
    {
        return !$this->is_merged && !$this->isClosed();
    }
}