<?php

namespace Modules\Support\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Storage;

class TicketAttachment extends Model
{
    protected $table = 'support_ticket_attachments';

    protected $fillable = [
        'ticket_id',
        'reply_id',
        'file_name',
        'file_path',
        'file_type',
        'file_size',
        'uploaded_at',
    ];

    protected $casts = [
        'ticket_id' => 'integer',
        'reply_id' => 'integer',
        'file_size' => 'integer',
        'uploaded_at' => 'datetime',
    ];

    /**
     * Relationships
     */
    public function ticket(): BelongsTo
    {
        return $this->belongsTo(Ticket::class, 'ticket_id');
    }

    public function reply(): BelongsTo
    {
        return $this->belongsTo(TicketReply::class, 'reply_id');
    }

    /**
     * Scopes
     */
    public function scopeByTicket($query, $ticketId)
    {
        return $query->where('ticket_id', $ticketId);
    }

    public function scopeByReply($query, $replyId)
    {
        return $query->where('reply_id', $replyId);
    }

    public function scopeTicketOnly($query)
    {
        return $query->whereNull('reply_id');
    }

    public function scopeImages($query)
    {
        return $query->where('file_type', 'LIKE', 'image/%');
    }

    /**
     * Accessors
     */
    public function getFileSizeFormattedAttribute(): string
    {
        $bytes = $this->file_size;
        
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        }
        
        return $bytes . ' bytes';
    }

    public function getFileExtensionAttribute(): string
    {
        return pathinfo($this->file_name, PATHINFO_EXTENSION);
    }

    public function getIsImageAttribute(): bool
    {
        return str_starts_with($this->file_type ?? '', 'image/');
    }

    public function getDownloadUrlAttribute(): string
    {
        return route('admin.support.download', ['id' => $this->id]);
    }

    public function getFileIconAttribute(): string
    {
        $ext = strtolower($this->file_extension);
        
        $iconMap = [
            'pdf' => 'fa-file-pdf text-danger',
            'doc' => 'fa-file-word text-primary',
            'docx' => 'fa-file-word text-primary',
            'xls' => 'fa-file-excel text-success',
            'xlsx' => 'fa-file-excel text-success',
            'zip' => 'fa-file-archive text-warning',
            'rar' => 'fa-file-archive text-warning',
            'jpg' => 'fa-file-image text-info',
            'jpeg' => 'fa-file-image text-info',
            'png' => 'fa-file-image text-info',
            'gif' => 'fa-file-image text-info',
            'txt' => 'fa-file-alt',
        ];
        
        return $iconMap[$ext] ?? 'fa-file';
    }

    /**
     * Helper Methods
     */
    public function deleteFile(): bool
    {
        if (Storage::disk('public')->exists($this->file_path)) {
            return Storage::disk('public')->delete($this->file_path);
        }
        
        return false;
    }

    /**
     * Boot method to auto-delete file
     */
    protected static function boot()
    {
        parent::boot();

        static::deleting(function ($attachment) {
            $attachment->deleteFile();
        });
    }
}