<?php

namespace Modules\Support\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class TicketReply extends Model
{
    protected $table = 'support_ticket_replies';

    protected $fillable = [
        'ticket_id',
        'message',
        'customer_id',
        'contact_id',
        'name',
        'email',
        'staff_id',
        'has_attachment',
        'replied_at',
    ];

    protected $casts = [
        'ticket_id' => 'integer',
        'customer_id' => 'integer',
        'contact_id' => 'integer',
        'staff_id' => 'integer',
        'has_attachment' => 'boolean',
        'replied_at' => 'datetime',
    ];

    /**
     * Relationships
     */
    public function ticket(): BelongsTo
    {
        return $this->belongsTo(Ticket::class, 'ticket_id');
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Customer::class, 'customer_id');
    }

    public function staff(): BelongsTo
    {
        return $this->belongsTo(\App\Models\Admin\Staff::class, 'staff_id');
    }

    public function attachments(): HasMany
    {
        return $this->hasMany(TicketAttachment::class, 'reply_id');
    }

    /**
     * Scopes
     */
    public function scopeByTicket($query, $ticketId)
    {
        return $query->where('ticket_id', $ticketId);
    }

    public function scopeByStaff($query)
    {
        return $query->whereNotNull('staff_id');
    }

    public function scopeByCustomer($query)
    {
        return $query->whereNotNull('customer_id');
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('replied_at', 'desc');
    }

    /**
     * Accessors
     */
    public function getSubmitterNameAttribute(): string
    {
        if ($this->staff) {
            return $this->staff->first_name . ' ' . $this->staff->last_name;
        }
        
        if ($this->customer) {
            return $this->customer->company ?? $this->customer->name ?? $this->name;
        }
        
        return $this->name ?? 'Unknown';
    }

    public function getSubmitterEmailAttribute(): string
    {
        if ($this->staff) {
            return $this->staff->email;
        }
        
        return $this->customer ? $this->customer->email : ($this->email ?? 'N/A');
    }

    public function getIsStaffReplyAttribute(): bool
    {
        return !empty($this->staff_id);
    }

    public function getIsCustomerReplyAttribute(): bool
    {
        return !empty($this->customer_id) || (!empty($this->email) && empty($this->staff_id));
    }

    public function getRepliedTimeAttribute(): string
    {
        return $this->replied_at ? $this->replied_at->diffForHumans() : 'N/A';
    }

    public function getAttachmentsCountAttribute(): int
    {
        return $this->attachments()->count();
    }
}