<style>
/* ========== DARK MODE VARIABLES ========== */
:root {
    --space-xl: 32px;
    --space-lg: 24px;
    --space-md: 16px;
    --navbar-height: 60px;
    --primary: #6366f1;
    --card-bg: #ffffff;
    --card-border: #e5e7eb;
    --body-bg: #f9fafb;
    --text-primary: #111827;
    --text-secondary: #6b7280;
    --text-muted: #9ca3af;
    --shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
    --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
}

[data-theme="dark"], 
.dark-mode,
body.dark-mode,
html.dark-mode,
html[data-theme="dark"],
body[data-theme="dark"] {
    --card-bg: #1e293b;
    --card-border: #334155;
    --body-bg: #0f172a;
    --text-primary: #f1f5f9;
    --text-secondary: #cbd5e1;
    --text-muted: #94a3b8;
    --shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.3);
    --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.4);
}

@media (prefers-color-scheme: dark) {
    :root:not([data-theme="light"]) {
        --card-bg: #1e293b;
        --card-border: #334155;
        --body-bg: #0f172a;
        --text-primary: #f1f5f9;
        --text-secondary: #cbd5e1;
        --text-muted: #94a3b8;
        --shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.3);
        --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.4);
    }
}

/* ========== SUPPORT TICKETS INDEX - ECHOPX COMPLIANT ========== */

.tickets-page-wrapper {
    padding: var(--space-xl);
    min-height: calc(100vh - 60px);
    animation: fadeInUp 500ms cubic-bezier(0.4, 0, 0.2, 1);
    background: var(--body-bg);
    transition: background 300ms ease;
}

@keyframes fadeInUp {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

@keyframes slideDown {
    from { opacity: 0; transform: translateY(-20px); }
    to { opacity: 1; transform: translateY(0); }
}

@keyframes iconPulse {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.05); }
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.5; }
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* ========== PAGE HEADER - GRADIENT ========== */
.tickets-page-header {
    background: linear-gradient(135deg, #6366f1, #4f46e5);
    padding: 28px 32px;
    border-radius: 16px;
    margin-bottom: var(--space-xl);
    box-shadow: 0 8px 24px rgba(99, 102, 241, 0.25);
    animation: slideDown 600ms cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
}

.tickets-page-header::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -10%;
    width: 350px;
    height: 350px;
    background: radial-gradient(circle, rgba(255,255,255,0.12) 0%, transparent 70%);
    border-radius: 50%;
    pointer-events: none;
}

.header-main-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 24px;
    position: relative;
    z-index: 1;
    flex-wrap: wrap;
}

.header-left-section {
    flex: 1;
    min-width: 0;
}

.page-title-main {
    color: #ffffff;
    font-size: 26px;
    font-weight: 700;
    margin-bottom: 8px;
    display: flex;
    align-items: center;
    gap: 12px;
    line-height: 1.2;
}

.page-title-main svg {
    width: 28px;
    height: 28px;
    flex-shrink: 0;
    animation: iconPulse 3s ease-in-out infinite;
}

.page-subtitle-main {
    color: rgba(255, 255, 255, 0.95);
    font-size: 14px;
    font-weight: 500;
    line-height: 1.4;
}

/* Header Search */
.header-search-section {
    flex: 1;
    max-width: 400px;
    min-width: 280px;
}

.search-wrapper-main {
    position: relative;
}

.search-input-main {
    width: 100%;
    padding: 12px 16px 12px 46px;
    border-radius: 12px;
    border: 2px solid rgba(255, 255, 255, 0.25);
    font-size: 14px;
    background: rgba(255, 255, 255, 0.15);
    color: #ffffff;
    font-weight: 500;
    transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
    backdrop-filter: blur(10px);
}

.search-input-main::placeholder {
    color: rgba(255, 255, 255, 0.75);
}

.search-input-main:focus {
    outline: none;
    background: rgba(255, 255, 255, 0.25);
    border-color: #ffffff;
    box-shadow: 0 4px 16px rgba(255, 255, 255, 0.2);
}

.search-icon-main {
    position: absolute;
    left: 16px;
    top: 50%;
    transform: translateY(-50%);
    width: 18px;
    height: 18px;
    color: rgba(255, 255, 255, 0.85);
    pointer-events: none;
}

.header-action-buttons {
    display: flex;
    gap: 12px;
}

.btn-main {
    padding: 12px 24px;
    border-radius: 12px;
    font-weight: 600;
    font-size: 14px;
    border: none;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 8px;
    text-decoration: none;
    transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
}

.btn-main svg {
    width: 16px;
    height: 16px;
    flex-shrink: 0;
}

.btn-primary-main {
    background: #ffffff;
    color: #4f46e5;
    box-shadow: 0 4px 12px rgba(255, 255, 255, 0.3);
}

.btn-primary-main:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(255, 255, 255, 0.4);
    background: #f8fafc;
}

/* ========== STATUS TABS ========== */
.status-tabs-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(160px, 1fr));
    gap: 16px;
    margin-bottom: var(--space-xl);
    animation: fadeInUp 500ms cubic-bezier(0.4, 0, 0.2, 1) 100ms backwards;
}

.status-tab-item {
    background: var(--card-bg);
    padding: 20px 24px;
    border-radius: 14px;
    border: 2px solid var(--card-border);
    cursor: pointer;
    transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
    display: flex;
    flex-direction: column;
    gap: 8px;
    position: relative;
    overflow: hidden;
}

.status-tab-item::before {
    content: '';
    position: absolute;
    top: 0;
    right: 0;
    width: 90px;
    height: 90px;
    background: linear-gradient(135deg, transparent 50%, rgba(99, 102, 241, 0.04) 50%);
    pointer-events: none;
}

.status-tab-item:hover {
    border-color: var(--primary);
    transform: translateY(-3px);
    box-shadow: var(--shadow-md);
}

.status-tab-item.active {
    background: linear-gradient(135deg, #6366f1, #4f46e5);
    color: #ffffff;
    border-color: #4f46e5;
    box-shadow: 0 6px 16px rgba(99, 102, 241, 0.35);
}

.status-tab-item.active::before {
    background: linear-gradient(135deg, transparent 50%, rgba(255, 255, 255, 0.1) 50%);
}

.status-count-display {
    font-size: 30px;
    font-weight: 800;
    line-height: 1;
    position: relative;
    z-index: 1;
}

.status-label-display {
    font-size: 12px;
    opacity: 0.92;
    line-height: 1;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.06em;
    position: relative;
    z-index: 1;
}

/* ========== TABLE CARD ========== */
.table-main-card {
    background: var(--card-bg);
    padding: 32px;
    border-radius: 16px;
    box-shadow: var(--shadow-sm);
    border: 1px solid var(--card-border);
    animation: fadeInUp 500ms cubic-bezier(0.4, 0, 0.2, 1) 200ms backwards;
    transition: all 300ms ease;
}

.table-header-section {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 24px;
    flex-wrap: wrap;
    gap: 16px;
}

.table-title-main {
    font-size: 18px;
    font-weight: 700;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: 10px;
}

.table-title-main svg {
    width: 20px;
    height: 20px;
    color: var(--primary);
}

.table-title-count {
    background: var(--body-bg);
    padding: 4px 12px;
    border-radius: 8px;
    font-size: 14px;
    color: var(--text-secondary);
    font-weight: 600;
    border: 1px solid var(--card-border);
}

/* Filters Section */
.filters-section {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
}

.filter-select-main {
    padding: 10px 16px;
    border-radius: 10px;
    border: 2px solid var(--card-border);
    background: var(--body-bg);
    color: var(--text-primary);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 300ms ease;
    min-width: 150px;
}

.filter-select-main:hover {
    border-color: var(--primary);
}

.filter-select-main:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.checkbox-filter-wrapper {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    background: var(--body-bg);
    border-radius: 10px;
    border: 2px solid var(--card-border);
    cursor: pointer;
    transition: all 300ms ease;
}

.checkbox-filter-wrapper:hover {
    border-color: var(--primary);
    background: var(--card-bg);
}

.checkbox-filter-wrapper input[type="checkbox"] {
    width: 18px;
    height: 18px;
    cursor: pointer;
    accent-color: var(--primary);
}

.checkbox-filter-label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    cursor: pointer;
    user-select: none;
}

/* Bulk Actions */
.bulk-actions-section {
    display: flex;
    gap: 12px;
    margin-bottom: 20px;
}

.btn-bulk-delete {
    padding: 10px 20px;
    background: #ef4444;
    color: #ffffff;
    border: none;
    border-radius: 10px;
    font-weight: 600;
    font-size: 14px;
    cursor: pointer;
    display: none;
    align-items: center;
    gap: 8px;
    transition: all 300ms ease;
}

.btn-bulk-delete:hover {
    background: #dc2626;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
}

.btn-bulk-delete svg {
    width: 16px;
    height: 16px;
}

/* Table Styles */
.table-wrapper-main {
    overflow-x: auto;
    border-radius: 12px;
    border: 1px solid var(--card-border);
    background: var(--card-bg);
}

.table-main {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.table-main thead {
    background: var(--body-bg);
    border-bottom: 2px solid var(--card-border);
}

.table-main th {
    padding: 14px 16px;
    text-align: left;
    font-weight: 700;
    color: var(--text-primary);
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    white-space: nowrap;
}

.table-main tbody tr {
    border-bottom: 1px solid var(--card-border);
    transition: all 200ms ease;
}

.table-main tbody tr:hover {
    background: var(--body-bg);
}

.table-main tbody tr:last-child {
    border-bottom: none;
}

.table-main td {
    padding: 16px;
    color: var(--text-secondary);
    vertical-align: middle;
}

/* Ticket Number Link */
.ticket-number-link {
    color: var(--primary);
    font-weight: 700;
    text-decoration: none;
    font-family: 'Courier New', monospace;
    transition: all 200ms ease;
}

.ticket-number-link:hover {
    color: #4f46e5;
    text-decoration: underline;
}

/* Ticket Subject Cell */
.ticket-subject-cell {
    display: flex;
    align-items: center;
    gap: 8px;
    color: var(--text-primary);
    font-weight: 600;
}

.unread-indicator {
    width: 8px;
    height: 8px;
    background: #3b82f6;
    border-radius: 50%;
    flex-shrink: 0;
    animation: pulse 2s ease-in-out infinite;
}

.merged-ticket-badge {
    display: inline-flex;
    align-items: center;
    gap: 4px;
    padding: 4px 8px;
    background: #f59e0b;
    color: #ffffff;
    border-radius: 6px;
    font-size: 11px;
    font-weight: 600;
}

.merged-ticket-badge svg {
    width: 12px;
    height: 12px;
}

/* Action Buttons */
.action-buttons-group {
    display: flex;
    gap: 8px;
}

.btn-icon-main {
    width: 36px;
    height: 36px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 200ms ease;
}

.btn-icon-main svg {
    width: 18px;
    height: 18px;
}

.btn-view-main {
    background: #dbeafe;
    color: #3b82f6;
}

.btn-view-main:hover {
    background: #3b82f6;
    color: #ffffff;
    transform: scale(1.1);
}

.btn-delete-main {
    background: #fee2e2;
    color: #ef4444;
}

.btn-delete-main:hover {
    background: #ef4444;
    color: #ffffff;
    transform: scale(1.1);
}

/* Empty State */
.empty-state-main {
    text-align: center;
    padding: 80px 24px;
}

.empty-state-main svg {
    width: 80px;
    height: 80px;
    color: var(--text-muted);
    margin-bottom: 20px;
    opacity: 0.5;
}

.empty-state-main h3 {
    font-size: 18px;
    font-weight: 700;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.empty-state-main p {
    color: var(--text-secondary);
    font-size: 14px;
    line-height: 1.6;
}

/* Pagination */
.pagination-section {
    margin-top: 24px;
    padding-top: 24px;
    border-top: 1px solid var(--card-border);
}

.btn-sm-main {
    padding: 8px 16px;
    font-size: 13px;
}

/* Loading Overlay */
.loading-overlay-main {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.7);
    backdrop-filter: blur(4px);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
    opacity: 0;
    visibility: hidden;
    transition: all 300ms ease;
}

.loading-overlay-main.active {
    opacity: 1;
    visibility: visible;
}

.loading-spinner-main {
    width: 60px;
    height: 60px;
    border: 4px solid rgba(255, 255, 255, 0.2);
    border-top-color: #ffffff;
    border-radius: 50%;
    animation: spin 800ms linear infinite;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .status-tabs-grid {
        grid-template-columns: repeat(3, 1fr);
    }
}

@media (max-width: 768px) {
    .tickets-page-wrapper {
        padding: var(--space-lg);
    }

    .tickets-page-header {
        padding: 20px;
    }

    .page-title-main {
        font-size: 22px;
    }

    .header-main-content {
        flex-direction: column;
        align-items: stretch;
    }

    .header-search-section {
        max-width: 100%;
    }

    .status-tabs-grid {
        grid-template-columns: repeat(2, 1fr);
    }

    .table-main-card {
        padding: 20px;
    }

    .table-header-section {
        flex-direction: column;
        align-items: stretch;
    }

    .filters-section {
        width: 100%;
    }

    .filter-select-main {
        width: 100%;
    }
}

@media (max-width: 480px) {
    .status-tabs-grid {
        grid-template-columns: 1fr;
    }

    .action-buttons-group {
        flex-direction: column;
    }
}
</style>

<div class="tickets-page-wrapper">
    <!-- Page Header -->
    <div class="tickets-page-header">
        <div class="header-main-content">
            <div class="header-left-section">
                <h1 class="page-title-main">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M15 5v2m0 4v2m0 4v2M5 5a2 2 0 00-2 2v3a2 2 0 110 4v3a2 2 0 002 2h14a2 2 0 002-2v-3a2 2 0 110-4V7a2 2 0 00-2-2H5z"/>
                    </svg>
                    Support Tickets
                </h1>
                <p class="page-subtitle-main">Manage and track customer support tickets efficiently</p>
            </div>
            
            <div class="header-search-section">
                <div class="search-wrapper-main">
                    <svg class="search-icon-main" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                    </svg>
                    <input type="text" 
                           class="search-input-main" 
                           id="searchInput"
                           placeholder="Search tickets..." 
                           onkeyup="if(event.key === 'Enter') loadTickets()">
                </div>
            </div>

            <div class="header-action-buttons">
                <a href="{{ route('admin.support.tickets.create') }}" class="btn-main btn-primary-main">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M12 4v16m8-8H4"/>
                    </svg>
                    New Ticket
                </a>
            </div>
        </div>
    </div>

    <!-- Status Tabs -->
    <div class="status-tabs-grid">
        @foreach($statuses as $status)
        <div class="status-tab-item" data-status="{{ $status->id }}" onclick="filterByStatus({{ $status->id }})">
            <div class="status-count-display" id="count-{{ $status->id }}">{{ $status->tickets_count }}</div>
            <div class="status-label-display">{{ strtoupper($status->name) }}</div>
        </div>
        @endforeach
    </div>

    <!-- Table Card -->
    <div class="table-main-card">
        <div class="table-header-section">
            <div class="table-title-main">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
                </svg>
                All Tickets
                <span class="table-title-count">( <span id="totalCount">0</span> )</span>
            </div>

            <div class="filters-section">
                <select class="filter-select-main" id="departmentFilter" onchange="loadTickets()">
                    <option value="">All Departments</option>
                    @foreach($departments as $department)
                        <option value="{{ $department->id }}">{{ $department->name }}</option>
                    @endforeach
                </select>

                <select class="filter-select-main" id="priorityFilter" onchange="loadTickets()">
                    <option value="">All Priorities</option>
                    @foreach($priorities as $priority)
                        <option value="{{ $priority->id }}">{{ $priority->name }}</option>
                    @endforeach
                </select>

                <select class="filter-select-main" id="statusFilter" onchange="loadTickets()">
                    <option value="">All Tickets</option>
                    <option value="open">Open</option>
                    <option value="closed">Closed</option>
                </select>

                <label class="checkbox-filter-wrapper">
                    <input type="checkbox" id="unreadOnly" onchange="loadTickets()">
                    <span class="checkbox-filter-label">Unread Only</span>
                </label>
            </div>
        </div>

        <!-- Bulk Actions -->
        <div class="bulk-actions-section">
            <button class="btn-bulk-delete" id="bulkDeleteBtn" onclick="bulkDelete()">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                </svg>
                Delete Selected
            </button>
        </div>

        <!-- Table -->
        <div class="table-wrapper-main">
            <table class="table-main">
                <thead>
                    <tr>
                        <th><input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)"></th>
                        <th>SL NO.</th>
                        <th>TICKET #</th>
                        <th>SUBJECT</th>
                        <th>CONTACT</th>
                        <th>DEPARTMENT</th>
                        <th>PRIORITY</th>
                        <th>STATUS</th>
                        <th>ASSIGNED TO</th>
                        <th>LAST REPLY</th>
                        <th>CREATED</th>
                        <th>ACTIONS</th>
                    </tr>
                </thead>
                <tbody id="ticketsTableBody">
                    <!-- Dynamically loaded -->
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <div class="pagination-section" id="paginationContainer"></div>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay-main" id="loadingOverlay">
    <div class="loading-spinner-main"></div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script>
let currentPage = 1;
let currentStatusFilter = null;
let selectedIds = [];
let isFirstLoad = true;

$(document).ready(function() {
    loadTickets();
});

function loadTickets() {
    const params = {
        page: currentPage,
        search: $('#searchInput').val(),
        department: $('#departmentFilter').val(),
        priority: $('#priorityFilter').val(),
        status: currentStatusFilter || $('#statusFilter').val(),
        unread: $('#unreadOnly').is(':checked') ? 1 : 0
    };

    if (!isFirstLoad) {
        showLoading();
    }

    $.ajax({
        url: '{{ route("admin.support.tickets.data") }}',
        type: 'GET',
        data: params,
        success: function(response) {
            renderTickets(response.data);
            renderPagination(response);
            updateCounts(response);
            
            if (!isFirstLoad) {
                hideLoading();
            }
            
            isFirstLoad = false;
        },
        error: function(xhr) {
            if (!isFirstLoad) {
                hideLoading();
            }
            
            isFirstLoad = false;
            
            showToast('Error loading tickets', 'error');
        }
    });
}

function renderTickets(tickets) {
    const tbody = $('#ticketsTableBody');
    tbody.empty();

    if (!tickets || tickets.length === 0) {
        tbody.append(`
            <tr>
                <td colspan="12">
                    <div class="empty-state-main">
                        <svg fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M15 5v2m0 4v2m0 4v2M5 5a2 2 0 00-2 2v3a2 2 0 110 4v3a2 2 0 002 2h14a2 2 0 002-2v-3a2 2 0 110-4V7a2 2 0 00-2-2H5z"/>
                        </svg>
                        <h3>No Tickets Found</h3>
                        <p>No tickets match your search criteria. Try adjusting your filters or search terms.</p>
                    </div>
                </td>
            </tr>
        `);
        return;
    }

    tickets.forEach(ticket => {
        const row = `
            <tr>
                <td>
                    <input type="checkbox" class="ticket-checkbox" value="${ticket.id}" onchange="toggleSelection(this)">
                </td>
                <td>
                    <strong style="color: var(--text-primary); font-family: 'Courier New', monospace;">${ticket.id}</strong>
                </td>
                <td>
                    <a href="${ticket._show_url}" class="ticket-number-link">${ticket.ticket_number}</a>
                </td>
                <td>
                    <div class="ticket-subject-cell">
                        ${ticket.is_unread ? '<span class="unread-indicator"></span>' : ''}
                        <span>${ticket.subject}</span>
                        ${ticket.merged_badge ? `
                            <span class="merged-ticket-badge">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M8 7h12m0 0l-4-4m4 4l-4 4m0 6H4m0 0l4 4m-4-4l4-4"/>
                                </svg>
                                Merged
                            </span>
                        ` : ''}
                    </div>
                </td>
                <td>${ticket.submitter_name}</td>
                <td>${ticket.department_name}</td>
                <td>${ticket.priority_badge}</td>
                <td>${ticket.status_badge}</td>
                <td>${ticket.assigned_to_name}</td>
                <td>${ticket.last_reply_time || 'No Reply'}</td>
                <td>${ticket.created_at}</td>
                <td>
                    <div class="action-buttons-group">
                        <button class="btn-icon-main btn-view-main" onclick="viewTicket(${ticket.id})" title="View Ticket">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                                <path stroke-linecap="round" stroke-linejoin="round" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                            </svg>
                        </button>
                        <button class="btn-icon-main btn-delete-main" onclick="deleteTicket(${ticket.id})" title="Delete Ticket">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                            </svg>
                        </button>
                    </div>
                </td>
            </tr>
        `;
        tbody.append(row);
    });
}

function renderPagination(response) {
    const container = $('#paginationContainer');
    container.empty();

    if (response.last_page <= 1) return;

    let html = '<div style="display: flex; justify-content: center; gap: 10px; flex-wrap: wrap;">';
    
    html += `<button class="btn-main btn-sm-main" style="background: ${currentPage === 1 ? '#e5e7eb' : 'var(--primary)'}; color: ${currentPage === 1 ? '#6b7280' : 'white'};" 
                     onclick="changePage(${currentPage - 1})" ${currentPage === 1 ? 'disabled' : ''}>Previous</button>`;

    for (let i = 1; i <= response.last_page; i++) {
        if (i === 1 || i === response.last_page || (i >= currentPage - 2 && i <= currentPage + 2)) {
            html += `<button class="btn-main btn-sm-main" style="background: ${i === currentPage ? 'var(--primary)' : '#e5e7eb'}; color: ${i === currentPage ? 'white' : '#6b7280'};" 
                             onclick="changePage(${i})">${i}</button>`;
        } else if (i === currentPage - 3 || i === currentPage + 3) {
            html += `<span style="padding: 0 10px; color: var(--text-secondary); font-weight: 600;">...</span>`;
        }
    }

    html += `<button class="btn-main btn-sm-main" style="background: ${currentPage === response.last_page ? '#e5e7eb' : 'var(--primary)'}; color: ${currentPage === response.last_page ? '#6b7280' : 'white'};" 
                     onclick="changePage(${currentPage + 1})" ${currentPage === response.last_page ? 'disabled' : ''}>Next</button>`;
    html += '</div>';
    
    container.html(html);
}

function updateCounts(response) {
    $('#totalCount').text(response.total);
    
    if (response.status_counts) {
        Object.keys(response.status_counts).forEach(statusId => {
            $(`#count-${statusId}`).text(response.status_counts[statusId]);
        });
    }
}

function filterByStatus(statusId) {
    if (currentStatusFilter === statusId) {
        currentStatusFilter = null;
        $('.status-tab-item').removeClass('active');
    } else {
        currentStatusFilter = statusId;
        $('.status-tab-item').removeClass('active');
        $(`.status-tab-item[data-status="${statusId}"]`).addClass('active');
    }
    
    currentPage = 1;
    loadTickets();
}

function changePage(page) {
    if (page < 1) return;
    currentPage = page;
    loadTickets();
}

function toggleSelection(checkbox) {
    const id = parseInt(checkbox.value);
    if (checkbox.checked) {
        selectedIds.push(id);
    } else {
        selectedIds = selectedIds.filter(item => item !== id);
    }
    $('#bulkDeleteBtn').toggle(selectedIds.length > 0);
}

function toggleSelectAll(checkbox) {
    $('.ticket-checkbox').prop('checked', checkbox.checked);
    selectedIds = checkbox.checked ? 
        $('.ticket-checkbox').map(function() { return parseInt(this.value); }).get() : [];
    $('#bulkDeleteBtn').toggle(selectedIds.length > 0);
}

function viewTicket(id) {
    window.location.href = `{{ route('admin.support.tickets.show', ':id') }}`.replace(':id', id);
}

function deleteTicket(id) {
    if (!confirm('Are you sure you want to delete this ticket?')) return;

    showLoading();
    $.ajax({
        url: `{{ route('admin.support.tickets.destroy', ':id') }}`.replace(':id', id),
        type: 'DELETE',
        data: { _token: '{{ csrf_token() }}' },
        success: function(response) {
            hideLoading();
            showToast(response.message || 'Ticket deleted successfully', 'success');
            loadTickets();
        },
        error: function() {
            hideLoading();
            showToast('Error deleting ticket', 'error');
        }
    });
}

function bulkDelete() {
    if (selectedIds.length === 0) return;
    if (!confirm(`Delete ${selectedIds.length} selected ticket(s)?`)) return;

    showLoading();
    $.ajax({
        url: '{{ route("admin.support.tickets.bulk-delete") }}',
        type: 'POST',
        data: {
            _token: '{{ csrf_token() }}',
            ids: selectedIds
        },
        success: function(response) {
            hideLoading();
            showToast(response.message || 'Tickets deleted successfully', 'success');
            selectedIds = [];
            $('#selectAll').prop('checked', false);
            $('#bulkDeleteBtn').hide();
            loadTickets();
        },
        error: function() {
            hideLoading();
            showToast('Error deleting tickets', 'error');
        }
    });
}

function showLoading() {
    $('#loadingOverlay').addClass('active');
}

function hideLoading() {
    $('#loadingOverlay').removeClass('active');
}

function showToast(message, type) {
    const bgColor = type === 'success' ? '#22c55e' : 
                    type === 'error' ? '#ef4444' : 
                    type === 'warning' ? '#f59e0b' : '#3b82f6';
    
    const toast = $('<div>')
        .css({
            position: 'fixed',
            top: '24px',
            right: '24px',
            background: bgColor,
            color: '#ffffff',
            padding: '14px 24px',
            borderRadius: '12px',
            boxShadow: '0 8px 24px rgba(0, 0, 0, 0.25)',
            zIndex: '10000',
            fontSize: '14px',
            fontWeight: '600',
            opacity: '0',
            transform: 'translateX(400px)',
            transition: 'all 400ms cubic-bezier(0.4, 0, 0.2, 1)'
        })
        .text(message)
        .appendTo('body');
    
    setTimeout(() => toast.css({ opacity: '1', transform: 'translateX(0)' }), 10);
    setTimeout(() => {
        toast.css({ opacity: '0', transform: 'translateX(400px)' });
        setTimeout(() => toast.remove(), 400);
    }, 3000);
}
</script>