<style>
/* ========== NOTES TAB - ECHOPX COMPLIANT ========== */

.notes-wrapper {
    padding: 0;
}

@keyframes fadeInUp {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

@keyframes slideInLeft {
    from { opacity: 0; transform: translateX(-20px); }
    to { opacity: 1; transform: translateX(0); }
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.7; }
}

/* ========== HEADER SECTION ========== */
.notes-header-section {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 24px;
    padding-bottom: 18px;
    border-bottom: 2px solid var(--card-border);
}

.notes-title-main {
    font-size: 18px;
    font-weight: 700;
    color: var(--text-primary);
    display: flex;
    align-items: center;
    gap: 10px;
}

.notes-title-main svg {
    width: 20px;
    height: 20px;
    color: var(--primary);
    flex-shrink: 0;
}

.notes-count-badge {
    font-size: 13px;
    font-weight: 600;
    color: var(--text-secondary);
    background: var(--body-bg);
    padding: 6px 14px;
    border-radius: 20px;
    border: 1px solid var(--card-border);
}

/* ========== ADD NOTE FORM ========== */
.note-form-section {
    margin-bottom: 32px;
    padding-bottom: 28px;
    border-bottom: 2px solid var(--card-border);
}

.note-form-label {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 14px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 12px;
}

.note-form-label svg {
    width: 16px;
    height: 16px;
    color: var(--text-secondary);
}

.note-textarea-styled {
    width: 100%;
    min-height: 130px;
    padding: 14px 16px;
    border: 1.5px solid var(--card-border);
    border-radius: 12px;
    font-size: 14px;
    color: var(--text-primary);
    background: var(--card-bg);
    resize: vertical;
    margin-bottom: 14px;
    font-family: inherit;
    line-height: 1.6;
    transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
}

.note-textarea-styled:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.note-textarea-styled::placeholder {
    color: var(--text-muted);
}

.note-actions-section {
    display: flex;
    justify-content: flex-end;
}

.btn-add-note-styled {
    padding: 11px 24px;
    background: var(--primary);
    color: #ffffff;
    border: none;
    border-radius: 10px;
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 8px;
    transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 2px 8px rgba(99, 102, 241, 0.25);
}

.btn-add-note-styled svg {
    width: 16px;
    height: 16px;
}

.btn-add-note-styled:hover {
    background: #4f46e5;
    transform: translateY(-2px);
    box-shadow: 0 4px 16px rgba(99, 102, 241, 0.35);
}

.btn-add-note-styled:active {
    transform: translateY(0);
}

/* ========== NOTES LIST ========== */
.notes-list-container {
    display: flex;
    flex-direction: column;
    gap: 18px;
}

.note-item-styled {
    background: var(--body-bg);
    padding: 20px;
    border-radius: 12px;
    border-left: 4px solid var(--primary);
    transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
    animation: slideInLeft 400ms cubic-bezier(0.4, 0, 0.2, 1);
}

.note-item-styled:hover {
    transform: translateX(4px);
    box-shadow: var(--shadow-sm);
}

.note-header-section {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 14px;
}

.note-author-section {
    display: flex;
    align-items: center;
    gap: 12px;
}

.note-author-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: linear-gradient(135deg, #6366f1, #4f46e5);
    color: #ffffff;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    font-size: 15px;
    flex-shrink: 0;
    box-shadow: 0 4px 12px rgba(99, 102, 241, 0.25);
}

.note-author-info-section {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.note-author-name-styled {
    font-size: 14px;
    font-weight: 600;
    color: var(--text-primary);
    line-height: 1.2;
}

.note-time-styled {
    font-size: 12px;
    color: var(--text-secondary);
    display: flex;
    align-items: center;
    gap: 6px;
}

.note-time-styled svg {
    width: 12px;
    height: 12px;
    flex-shrink: 0;
}

.note-delete-btn {
    background: rgba(239, 68, 68, 0.12);
    border: none;
    color: var(--danger);
    cursor: pointer;
    padding: 8px;
    border-radius: 8px;
    transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
    flex-shrink: 0;
}

.note-delete-btn svg {
    width: 16px;
    height: 16px;
}

.note-delete-btn:hover {
    background: var(--danger);
    color: #ffffff;
    transform: scale(1.1);
}

.note-content-styled {
    font-size: 14px;
    color: var(--text-primary);
    line-height: 1.7;
    white-space: pre-wrap;
    word-wrap: break-word;
}

/* ========== EMPTY STATE ========== */
.empty-notes-styled {
    text-align: center;
    padding: 60px 24px;
    animation: fadeInUp 500ms cubic-bezier(0.4, 0, 0.2, 1);
}

.empty-notes-icon {
    width: 72px;
    height: 72px;
    color: var(--text-muted);
    margin: 0 auto 16px;
    opacity: 0.6;
}

.empty-notes-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.empty-notes-subtitle {
    font-size: 14px;
    color: var(--text-secondary);
    line-height: 1.6;
    max-width: 400px;
    margin: 0 auto;
}

/* ========== RESPONSIVE DESIGN ========== */
@media (max-width: 768px) {
    .notes-header-section {
        flex-direction: column;
        align-items: flex-start;
        gap: 12px;
    }

    .note-form-section {
        margin-bottom: 24px;
        padding-bottom: 24px;
    }

    .note-textarea-styled {
        min-height: 110px;
    }

    .note-actions-section {
        justify-content: stretch;
    }

    .btn-add-note-styled {
        width: 100%;
        justify-content: center;
    }

    .note-author-section {
        gap: 10px;
    }

    .note-author-avatar {
        width: 36px;
        height: 36px;
        font-size: 14px;
    }
}

@media (max-width: 640px) {
    .notes-title-main {
        font-size: 16px;
    }

    .empty-notes-styled {
        padding: 40px 20px;
    }

    .empty-notes-icon {
        width: 56px;
        height: 56px;
    }

    .note-item-styled {
        padding: 16px;
    }

    .note-header-section {
        flex-direction: column;
        align-items: flex-start;
        gap: 12px;
    }

    .note-delete-btn {
        align-self: flex-start;
    }
}
</style>

<div class="notes-wrapper">
    <!-- Notes Header -->
    <div class="notes-header-section">
        <div class="notes-title-main">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
            </svg>
            Internal Notes
        </div>
        <div class="notes-count-badge" id="notesCount">0 notes</div>
    </div>

    <!-- Add Note Form -->
    <div class="note-form-section">
        <label class="note-form-label">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M12 4v16m8-8H4"/>
            </svg>
            Add New Note
        </label>
        <textarea id="newNoteText" 
                  class="note-textarea-styled" 
                  placeholder="Add an internal note (not visible to customer)..."></textarea>
        <div class="note-actions-section">
            <button type="button" class="btn-add-note-styled" onclick="addNote()">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7"/>
                </svg>
                Add Note
            </button>
        </div>
    </div>

    <!-- Notes List -->
    <div id="notesList" class="notes-list-container">
        <!-- Notes will be loaded here via AJAX -->
    </div>

    <!-- Empty State -->
    <div id="emptyNotes" class="empty-notes-styled" style="display: none;">
        <svg class="empty-notes-icon" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"/>
        </svg>
        <div class="empty-notes-title">No notes yet</div>
        <div class="empty-notes-subtitle">Add the first note to keep track of internal information about this ticket</div>
    </div>
</div>

<script>
$(document).ready(function() {
    loadNotes();
});

function loadNotes() {
    $.ajax({
        url: '{{ route("admin.support.tickets.notes.index", $ticket->id) }}',
        type: 'GET',
        success: function(response) {
            if (response.success && response.notes) {
                renderNotes(response.notes);
            }
        },
        error: function() {
            console.error('Error loading notes');
        }
    });
}

function renderNotes(notes) {
    const container = $('#notesList');
    const emptyState = $('#emptyNotes');
    
    container.empty();
    
    if (notes.length === 0) {
        container.hide();
        emptyState.show();
        $('#notesCount').text('0 notes');
        return;
    }
    
    container.show();
    emptyState.hide();
    $('#notesCount').text(notes.length + ' ' + (notes.length === 1 ? 'note' : 'notes'));
    
    notes.forEach(note => {
        const initials = note.staff_name.split(' ').map(n => n[0]).join('').toUpperCase();
        
        const noteHtml = `
            <div class="note-item-styled" data-note-id="${note.id}">
                <div class="note-header-section">
                    <div class="note-author-section">
                        <div class="note-author-avatar">${initials}</div>
                        <div class="note-author-info-section">
                            <div class="note-author-name-styled">${note.staff_name}</div>
                            <div class="note-time-styled">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                </svg>
                                ${note.created_at}
                            </div>
                        </div>
                    </div>
                    <button class="note-delete-btn" onclick="deleteNote(${note.id})" title="Delete note">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"/>
                        </svg>
                    </button>
                </div>
                <div class="note-content-styled">${escapeHtml(note.note)}</div>
            </div>
        `;
        
        container.append(noteHtml);
    });
}

function addNote() {
    const noteText = $('#newNoteText').val().trim();
    
    if (!noteText) {
        showToast('Please enter a note', 'error');
        return;
    }
    
    showLoading();
    
    $.ajax({
        url: '{{ route("admin.support.tickets.notes.store", $ticket->id) }}',
        type: 'POST',
        data: {
            _token: '{{ csrf_token() }}',
            note: noteText
        },
        success: function(response) {
            hideLoading();
            if (response.success) {
                $('#newNoteText').val('');
                loadNotes();
                showToast('Note added successfully', 'success');
            } else {
                showToast(response.message || 'Error adding note', 'error');
            }
        },
        error: function(xhr) {
            hideLoading();
            showToast('Error adding note: ' + (xhr.responseJSON?.message || 'Unknown error'), 'error');
        }
    });
}

function deleteNote(noteId) {
    if (!confirm('Are you sure you want to delete this note?')) return;
    
    showLoading();
    
    $.ajax({
        url: '{{ route("admin.support.tickets.notes.destroy", [$ticket->id, ":noteId"]) }}'.replace(':noteId', noteId),
        type: 'DELETE',
        data: {
            _token: '{{ csrf_token() }}'
        },
        success: function(response) {
            hideLoading();
            if (response.success) {
                loadNotes();
                showToast('Note deleted successfully', 'success');
            } else {
                showToast(response.message || 'Error deleting note', 'error');
            }
        },
        error: function(xhr) {
            hideLoading();
            showToast('Error deleting note: ' + (xhr.responseJSON?.message || 'Unknown error'), 'error');
        }
    });
}

function escapeHtml(text) {
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}
</script>