<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ticket #{{ $ticket->id }} - {{ $ticket->subject }}</title>
    <style>
        :root {
            --primary: #6366f1;
            --primary-hover: #4f46e5;
            --success: #22c55e;
            --danger: #ef4444;
            --warning: #f59e0b;
            --info: #3b82f6;
            --text-primary: #111827;
            --text-secondary: #6b7280;
            --text-muted: #9ca3af;
            --body-bg: #f9fafb;
            --card-bg: #ffffff;
            --card-border: #e5e7eb;
            --input-border: #d1d5db;
            --shadow-sm: 0 1px 3px rgba(0, 0, 0, 0.1);
            --shadow-md: 0 4px 12px rgba(0, 0, 0, 0.08);
            --shadow-lg: 0 8px 24px rgba(0, 0, 0, 0.12);
            --radius-sm: 8px;
            --radius-md: 12px;
            --radius-lg: 16px;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', sans-serif;
            background: var(--body-bg);
            color: var(--text-primary);
            line-height: 1.6;
        }

        @keyframes slideDown {
            from { opacity: 0; transform: translateY(-20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        @keyframes fadeInUp {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.8; }
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 32px 20px;
        }

        /* ========== STAFF WARNING ========== */
        @if(auth('admin')->check())
        .staff-warning {
            background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
            border-left: 4px solid var(--warning);
            padding: 20px 24px;
            border-radius: var(--radius-md);
            margin-bottom: 24px;
            display: flex;
            align-items: center;
            gap: 16px;
            box-shadow: var(--shadow-sm);
            animation: slideDown 500ms cubic-bezier(0.4, 0, 0.2, 1);
        }

        .staff-warning-icon {
            width: 48px;
            height: 48px;
            background: rgba(245, 158, 11, 0.15);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }

        .staff-warning-icon svg {
            width: 24px;
            height: 24px;
            color: var(--warning);
        }

        .staff-warning-text {
            flex: 1;
        }

        .staff-warning-title {
            font-weight: 700;
            font-size: 16px;
            color: #92400e;
            margin-bottom: 6px;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .staff-warning-desc {
            font-size: 14px;
            color: #92400e;
            line-height: 1.5;
        }

        .staff-warning-link {
            padding: 10px 20px;
            background: var(--warning);
            color: white;
            text-decoration: none;
            border-radius: var(--radius-sm);
            font-weight: 600;
            font-size: 14px;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
            white-space: nowrap;
        }

        .staff-warning-link svg {
            width: 16px;
            height: 16px;
        }

        .staff-warning-link:hover {
            background: #d97706;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(245, 158, 11, 0.3);
        }
        @endif

        /* ========== PUBLIC NOTICE ========== */
        .public-notice {
            background: linear-gradient(135deg, #dbeafe 0%, #bfdbfe 100%);
            border-left: 4px solid var(--info);
            padding: 16px 20px;
            border-radius: var(--radius-md);
            margin-bottom: 24px;
            font-size: 14px;
            color: #1e40af;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: var(--shadow-sm);
            animation: slideDown 500ms cubic-bezier(0.4, 0, 0.2, 1) 100ms backwards;
        }

        .public-notice svg {
            width: 20px;
            height: 20px;
            flex-shrink: 0;
        }

        /* ========== TICKET HEADER ========== */
        .ticket-header {
            background: linear-gradient(135deg, var(--primary) 0%, #4f46e5 100%);
            padding: 32px;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-lg);
            margin-bottom: 28px;
            position: relative;
            overflow: hidden;
            animation: fadeInUp 500ms cubic-bezier(0.4, 0, 0.2, 1) 200ms backwards;
        }

        .ticket-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: radial-gradient(circle at top right, rgba(255, 255, 255, 0.1), transparent 60%);
            pointer-events: none;
        }

        .ticket-title {
            display: flex;
            justify-content: space-between;
            align-items: start;
            flex-wrap: wrap;
            gap: 20px;
            margin-bottom: 20px;
            position: relative;
            z-index: 1;
        }

        .ticket-title-content h1 {
            font-size: 28px;
            font-weight: 700;
            color: #ffffff;
            margin-bottom: 8px;
            line-height: 1.3;
        }

        .ticket-id {
            font-size: 14px;
            color: rgba(255, 255, 255, 0.8);
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            gap: 8px;
            font-weight: 600;
        }

        .ticket-id svg {
            width: 16px;
            height: 16px;
        }

        .status-badge {
            padding: 8px 20px;
            border-radius: 20px;
            font-size: 13px;
            font-weight: 700;
            color: white;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            text-transform: uppercase;
            letter-spacing: 0.05em;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .status-badge svg {
            width: 14px;
            height: 14px;
        }

        .ticket-meta {
            display: flex;
            gap: 28px;
            font-size: 14px;
            color: rgba(255, 255, 255, 0.9);
            flex-wrap: wrap;
            position: relative;
            z-index: 1;
        }

        .ticket-meta-item {
            display: flex;
            align-items: center;
            gap: 8px;
            font-weight: 500;
        }

        .ticket-meta-item svg {
            width: 18px;
            height: 18px;
            opacity: 0.9;
        }

        /* ========== MAIN CONTENT ========== */
        .main-content {
            display: grid;
            grid-template-columns: 1fr 350px;
            gap: 24px;
            animation: fadeInUp 500ms cubic-bezier(0.4, 0, 0.2, 1) 300ms backwards;
        }

        .conversation {
            display: flex;
            flex-direction: column;
            gap: 24px;
        }

        /* ========== MESSAGE CARDS ========== */
        .message-card {
            background: var(--card-bg);
            padding: 28px;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-sm);
            border: 1px solid var(--card-border);
            position: relative;
            transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
        }

        .message-card::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
            border-radius: var(--radius-lg) 0 0 var(--radius-lg);
            background: var(--card-border);
        }

        .message-card.staff::before {
            background: var(--info);
        }

        .message-card.customer::before {
            background: var(--success);
        }

        .message-card:hover {
            box-shadow: var(--shadow-md);
            transform: translateY(-2px);
        }

        .message-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 18px;
            padding-bottom: 14px;
            border-bottom: 2px solid var(--card-border);
        }

        .message-author {
            font-weight: 700;
            font-size: 16px;
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .author-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            background: var(--body-bg);
            border: 2px solid var(--card-border);
        }

        .author-icon svg {
            width: 20px;
            height: 20px;
            color: var(--text-secondary);
        }

        .author-icon.staff-icon {
            background: rgba(59, 130, 246, 0.1);
            border-color: var(--info);
        }

        .author-icon.staff-icon svg {
            color: var(--info);
        }

        .author-badge {
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 700;
            text-transform: uppercase;
            background: var(--info);
            color: white;
            letter-spacing: 0.05em;
        }

        .message-time {
            font-size: 13px;
            color: var(--text-secondary);
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .message-time svg {
            width: 14px;
            height: 14px;
        }

        .message-content {
            color: var(--text-primary);
            line-height: 1.8;
            font-size: 15px;
        }

        /* ========== ATTACHMENTS ========== */
        .message-attachments {
            margin-top: 18px;
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
        }

        .attachment {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 10px 16px;
            background: var(--body-bg);
            border: 1.5px solid var(--card-border);
            border-radius: var(--radius-sm);
            font-size: 13px;
            color: var(--text-primary);
            text-decoration: none;
            transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
            font-weight: 500;
        }

        .attachment svg {
            width: 16px;
            height: 16px;
            color: var(--text-secondary);
        }

        .attachment:hover {
            background: var(--card-bg);
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: var(--shadow-sm);
        }

        .attachment:hover svg {
            color: var(--primary);
        }

        /* ========== REPLY FORM ========== */
        .reply-form-card {
            background: var(--card-bg);
            padding: 32px;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-sm);
            border: 1px solid var(--card-border);
        }

        .reply-form-card h2 {
            font-size: 20px;
            font-weight: 700;
            margin-bottom: 24px;
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .reply-form-card h2 svg {
            width: 24px;
            height: 24px;
            color: var(--primary);
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-label {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 14px;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 10px;
        }

        .form-label svg {
            width: 16px;
            height: 16px;
            color: var(--text-secondary);
        }

        .form-input,
        .form-textarea {
            width: 100%;
            padding: 12px 16px;
            border: 1.5px solid var(--input-border);
            border-radius: var(--radius-md);
            font-size: 14px;
            font-family: inherit;
            color: var(--text-primary);
            background: var(--card-bg);
            transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
        }

        .form-textarea {
            min-height: 140px;
            resize: vertical;
            line-height: 1.6;
        }

        .form-input:focus,
        .form-textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
        }

        /* ========== FILE UPLOAD ========== */
        .file-upload {
            border: 2px dashed var(--input-border);
            border-radius: var(--radius-md);
            padding: 32px 20px;
            text-align: center;
            cursor: pointer;
            transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
            background: var(--body-bg);
        }

        .file-upload:hover {
            border-color: var(--primary);
            background: rgba(99, 102, 241, 0.05);
        }

        .file-upload-icon {
            width: 48px;
            height: 48px;
            margin: 0 auto 12px;
            color: var(--text-secondary);
        }

        .file-upload input {
            display: none;
        }

        .file-upload-text {
            font-size: 14px;
            color: var(--text-primary);
            font-weight: 600;
            margin-bottom: 6px;
        }

        .file-upload-hint {
            font-size: 12px;
            color: var(--text-secondary);
        }

        /* ========== SUBMIT BUTTON ========== */
        .btn-submit {
            width: 100%;
            padding: 14px 28px;
            background: var(--primary);
            color: white;
            border: none;
            border-radius: var(--radius-md);
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            box-shadow: 0 2px 8px rgba(99, 102, 241, 0.25);
        }

        .btn-submit svg {
            width: 18px;
            height: 18px;
        }

        .btn-submit:hover {
            background: var(--primary-hover);
            transform: translateY(-2px);
            box-shadow: 0 4px 16px rgba(99, 102, 241, 0.35);
        }

        .btn-submit:active {
            transform: translateY(0);
        }

        /* ========== SIDEBAR ========== */
        .sidebar {
            display: flex;
            flex-direction: column;
            gap: 20px;
        }

        .info-card {
            background: var(--card-bg);
            padding: 24px;
            border-radius: var(--radius-lg);
            box-shadow: var(--shadow-sm);
            border: 1px solid var(--card-border);
            transition: all 300ms cubic-bezier(0.4, 0, 0.2, 1);
        }

        .info-card:hover {
            box-shadow: var(--shadow-md);
        }

        .info-card h3 {
            font-size: 14px;
            font-weight: 700;
            color: var(--text-secondary);
            text-transform: uppercase;
            margin-bottom: 18px;
            letter-spacing: 0.05em;
            display: flex;
            align-items: center;
            gap: 8px;
            padding-bottom: 12px;
            border-bottom: 2px solid var(--card-border);
        }

        .info-card h3 svg {
            width: 16px;
            height: 16px;
        }

        .info-item {
            margin-bottom: 16px;
        }

        .info-item:last-child {
            margin-bottom: 0;
        }

        .info-label {
            font-size: 12px;
            color: var(--text-secondary);
            margin-bottom: 6px;
            font-weight: 500;
        }

        .info-value {
            font-size: 14px;
            color: var(--text-primary);
            font-weight: 600;
        }

        .help-text {
            font-size: 13px;
            color: var(--text-secondary);
            line-height: 1.6;
        }

        /* ========== RESPONSIVE DESIGN ========== */
        @media (max-width: 1024px) {
            .main-content {
                grid-template-columns: 1fr;
            }

            .sidebar {
                order: -1;
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
                gap: 16px;
            }
        }

        @media (max-width: 768px) {
            .container {
                padding: 20px 16px;
            }

            .ticket-header {
                padding: 24px 20px;
            }

            .ticket-title {
                flex-direction: column;
                gap: 16px;
            }

            .ticket-title-content h1 {
                font-size: 22px;
            }

            .ticket-meta {
                flex-direction: column;
                gap: 12px;
            }

            .message-card {
                padding: 20px;
            }

            .reply-form-card {
                padding: 24px;
            }

            .staff-warning {
                flex-direction: column;
                text-align: center;
            }

            .staff-warning-link {
                width: 100%;
                justify-content: center;
            }

            .sidebar {
                grid-template-columns: 1fr;
            }
        }

        @media (max-width: 640px) {
            .ticket-title-content h1 {
                font-size: 20px;
            }

            .message-attachments {
                flex-direction: column;
            }

            .attachment {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        @if(auth('admin')->check())
        <!-- Staff Warning -->
        <div class="staff-warning">
            <div class="staff-warning-icon">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                </svg>
            </div>
            <div class="staff-warning-text">
                <div class="staff-warning-title">Staff View Warning</div>
                <div class="staff-warning-desc">You're viewing this ticket as staff via the public URL. To access full admin features, use the admin panel.</div>
            </div>
            <a href="{{ route('admin.support.tickets.show', $ticket->id) }}" class="staff-warning-link">
                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" d="M13 7l5 5m0 0l-5 5m5-5H6"/>
                </svg>
                Go to Admin View
            </a>
        </div>
        @endif

        <div class="public-notice">
            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" d="M13.828 10.172a4 4 0 00-5.656 0l-4 4a4 4 0 105.656 5.656l1.102-1.101m-.758-4.899a4 4 0 005.656 0l4-4a4 4 0 00-5.656-5.656l-1.1 1.1"/>
            </svg>
            You're viewing this ticket via a public link. Keep this link private to maintain security.
        </div>

        <!-- Ticket Header -->
        <div class="ticket-header">
            <div class="ticket-title">
                <div class="ticket-title-content">
                    <div class="ticket-id">
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M15 5v2m0 4v2m0 4v2M5 5a2 2 0 00-2 2v3a2 2 0 110 4v3a2 2 0 002 2h14a2 2 0 002-2v-3a2 2 0 110-4V7a2 2 0 00-2-2H5z"/>
                        </svg>
                        Ticket #{{ $ticket->id }}
                    </div>
                    <h1>{{ $ticket->subject }}</h1>
                </div>
                <span class="status-badge" style="background-color: {{ $ticket->status->status_color }};">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                    {{ $ticket->status->name }}
                </span>
            </div>
            <div class="ticket-meta">
                <div class="ticket-meta-item">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                    </svg>
                    Created {{ $ticket->created_at->format('M d, Y g:i A') }}
                </div>
                <div class="ticket-meta-item">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M3 7v10a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-6l-2-2H5a2 2 0 00-2 2z"/>
                    </svg>
                    {{ $ticket->department->name }}
                </div>
                <div class="ticket-meta-item">
                    <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" d="M13 10V3L4 14h7v7l9-11h-7z"/>
                    </svg>
                    {{ $ticket->priority->name }}
                </div>
            </div>
        </div>

        <!-- Main Content -->
        <div class="main-content">
            <!-- Conversation -->
            <div class="conversation">
                <!-- Original Message -->
                <div class="message-card customer">
                    <div class="message-header">
                        <div class="message-author">
                            <div class="author-icon">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                                </svg>
                            </div>
                            <span>{{ $ticket->name }}</span>
                        </div>
                        <div class="message-time">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                            {{ $ticket->created_at->format('M d, Y g:i A') }}
                        </div>
                    </div>
                    <div class="message-content">
                        {!! nl2br(e($ticket->message)) !!}
                    </div>
                    @if($ticket->attachments->count() > 0)
                    <div class="message-attachments">
                        @foreach($ticket->attachments as $attachment)
                        <a href="{{ route('client.support.download-by-key', [$ticket->ticket_key, $attachment->id]) }}" class="attachment">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13"/>
                            </svg>
                            {{ $attachment->file_name }}
                        </a>
                        @endforeach
                    </div>
                    @endif
                </div>

                <!-- Replies -->
                @foreach($ticket->replies as $reply)
                <div class="message-card {{ $reply->staff_id ? 'staff' : 'customer' }}">
                    <div class="message-header">
                        <div class="message-author">
                            <div class="author-icon {{ $reply->staff_id ? 'staff-icon' : '' }}">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    @if($reply->staff_id)
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"/>
                                    @else
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                                    @endif
                                </svg>
                            </div>
                            <span>{{ $reply->staff_id ? $reply->staff->full_name : $reply->customer_name }}</span>
                            @if($reply->staff_id)
                            <span class="author-badge">STAFF</span>
                            @endif
                        </div>
                        <div class="message-time">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                            {{ $reply->replied_at->format('M d, Y g:i A') }}
                        </div>
                    </div>
                    <div class="message-content">
                        {!! $reply->message !!}
                    </div>
                    @if($reply->attachments->count() > 0)
                    <div class="message-attachments">
                        @foreach($reply->attachments as $attachment)
                        <a href="{{ route('client.support.download-by-key', [$ticket->ticket_key, $attachment->id]) }}" class="attachment">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13"/>
                            </svg>
                            {{ $attachment->file_name }}
                        </a>
                        @endforeach
                    </div>
                    @endif
                </div>
                @endforeach

                <!-- Reply Form -->
                <div class="reply-form-card">
                    <h2>
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M3 10h10a8 8 0 018 8v2M3 10l6 6m-6-6l6-6"/>
                        </svg>
                        Add Your Reply
                    </h2>
                    <form action="{{ route('client.support.reply-by-key', $ticket->ticket_key) }}" method="POST" enctype="multipart/form-data">
                        @csrf

                        <div class="form-group">
                            <label class="form-label">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
                                </svg>
                                Your Name
                            </label>
                            <input type="text" name="name" class="form-input" value="{{ old('name') }}" required>
                        </div>

                        <div class="form-group">
                            <label class="form-label">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>
                                </svg>
                                Your Email
                            </label>
                            <input type="email" name="email" class="form-input" value="{{ old('email') }}" required>
                        </div>

                        <div class="form-group">
                            <label class="form-label">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z"/>
                                </svg>
                                Your Message
                            </label>
                            <textarea name="message" class="form-textarea" required placeholder="Type your reply here...">{{ old('message') }}</textarea>
                        </div>

                        <div class="form-group">
                            <label class="form-label">
                                <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13"/>
                                </svg>
                                Attachments (Optional)
                            </label>
                            <div class="file-upload" onclick="document.getElementById('attachments').click()">
                                <svg class="file-upload-icon" fill="none" stroke="currentColor" stroke-width="1.5" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" d="M3 16.5v2.25A2.25 2.25 0 005.25 21h13.5A2.25 2.25 0 0021 18.75V16.5m-13.5-9L12 3m0 0l4.5 4.5M12 3v13.5"/>
                                </svg>
                                <div class="file-upload-text">Click to upload files</div>
                                <div class="file-upload-hint">Max 10MB per file</div>
                                <input type="file" id="attachments" name="attachments[]" multiple>
                            </div>
                        </div>

                        <button type="submit" class="btn-submit">
                            <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"/>
                            </svg>
                            Send Reply
                        </button>
                    </form>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="sidebar">
                <div class="info-card">
                    <h3>
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        Ticket Details
                    </h3>
                    <div class="info-item">
                        <div class="info-label">Status</div>
                        <div class="info-value">{{ $ticket->status->name }}</div>
                    </div>
                    <div class="info-item">
                        <div class="info-label">Priority</div>
                        <div class="info-value">{{ $ticket->priority->name }}</div>
                    </div>
                    <div class="info-item">
                        <div class="info-label">Department</div>
                        <div class="info-value">{{ $ticket->department->name }}</div>
                    </div>
                </div>

                <div class="info-card">
                    <h3>
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"/>
                        </svg>
                        Timeline
                    </h3>
                    <div class="info-item">
                        <div class="info-label">Created</div>
                        <div class="info-value">{{ $ticket->created_at->format('M d, Y g:i A') }}</div>
                    </div>
                    <div class="info-item">
                        <div class="info-label">Last Updated</div>
                        <div class="info-value">{{ $ticket->updated_at->diffForHumans() }}</div>
                    </div>
                    @if($ticket->last_reply)
                    <div class="info-item">
                        <div class="info-label">Last Reply</div>
                        <div class="info-value">{{ $ticket->last_reply->diffForHumans() }}</div>
                    </div>
                    @endif
                </div>

                <div class="info-card">
                    <h3>
                        <svg fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" d="M18.364 5.636l-3.536 3.536m0 5.656l3.536 3.536M9.172 9.172L5.636 5.636m3.536 9.192l-3.536 3.536M21 12a9 9 0 11-18 0 9 9 0 0118 0zm-5 0a4 4 0 11-8 0 4 4 0 018 0z"/>
                        </svg>
                        Need Help?
                    </h3>
                    <p class="help-text">
                        Keep this link bookmarked to track your ticket progress. You'll receive email updates when we reply.
                    </p>
                </div>
            </div>
        </div>
    </div>
</body>
</html>